/*	ZT1 'tape' routines using a disk file and "unix I/O" ...
	w.j.m. feb 1994, after ZT1T (ZT1 assumed to pass filename to tape_init)
*/

#ifndef TRACE
#define TRACE 1
#endif

#include ssdef

#include stdio
#include stdlib
#include string
#include errno

#include unixio
#include file

#include descrip
typedef struct dsc$descriptor DESCR;

extern void LIB$STOP(),Fehler();
#define CHECK(x) do {unsigned s=x; if(!(s&1)) LIB$STOP(s);} while(0)
#define FEHLER(m) do {$DESCRIPTOR(d,m); Fehler(&d);} while(0)

/* data & entries used by "main" program */

extern unsigned char *bufp;
extern unsigned int *bufbctp;

extern int /*logical*/ hw_online,hw_medonline,hw_bot,hw_eof,hw_eot,hw_hwl;

/* forward */
void tape_init(char * /*disk file name*/);
unsigned tape_writemark(void),
	tape_nop(void),
	tape_rewind(int/*logical*/ /*unload*/),
	tape_skiprec(int /*+-#blocks*/,int * /*#blocks skipped*/),
	tape_compare(int/*logical*/ /*reverse*/),
	tape_read(int/*logical*/ /*reverse*/,int/*logical*/ /*check*/),
	tape_write(int/*logical*/ /*check*/);

#if TRACE
static void trace_tape(char *,int,int,int);	/* forward */
static int tpos;	/* true position (in blocks) */
#endif

/* 'disk tape' structure (STREAM-LF file assumed):
 *	int l0; char [l0]; int l0; int l1; char [l1]; int l1; ...
 *		l=0 stands for end of written area,
 *		l=1 stands for tape mark
 * "int" is to mean INTEGER*4
 *
 * This is compatible with a unix [little-endian] FORTRAN BINARY file
 * (checked for ULTRIX & DEC OSF/1).
 *
 * the very first "record" always contains the fixed string "ZT1 "
 *
 * NOTE: I/O errors considered fatal.
 * NOTE: no EOT handling (yet)!
 */

#define T_MAXBLOCK 0xFFFF

	/* room for tape block & 2 length fields */
static unsigned char tbuf[4 + 4 + T_MAXBLOCK];
	/* current byte offset in file (point between 'prev' & 'next' record) */
static int tp;
	/* file descriptor */
static int tfd;
	/* read-only flag */
static int/*logical*/ t_ro;

#define T_EOM 0
#define T_MARK 1

static const struct T_magic {
	int l0_1;
	char magic[4];
	int l0_2;
} t_magic = { 4, {'Z', 'T', '1', ' '}, 4};

static int t_nextlen,t_prevlen;

#define T_nextlen t_nextlen
#define T_prevlen t_prevlen
#define T_data tbuf			/* tape block from TOP_read() */

#define T_prevpos (tp - (T_prevlen + 8))
#define T_nextpos (tp + (T_nextlen + 8))

#define T_bot (tp == sizeof(t_magic))
#define T_eom (T_nextlen == T_EOM)
#define T_eot 0				/* always false for now */

#define I4(bp) (*((int *)(bp)))

#define T_abort() do {\
	fprintf(stderr,"Aborting ...\n");\
	abort();\
} while(0)

#define T_checklen(i) do {\
	if(i < 0 || i > T_MAXBLOCK) {\
		fprintf(stderr,"Bad block length: %d\n",i);\
		T_abort();\
	}\
} while(0)

/*****/

static long diskpos;		/* current file position (byte offset) */
static int diskeof = -1;	/* new EOF position to be set, or -1 */


static void disk_seek(int p)
{
#if 1
	{
		static void disk_write();	/*forward*/
		static int teom = T_EOM;

		if(diskeof > 0 && p < diskeof) {
			disk_write(diskeof,4,(unsigned char *)&teom);
			diskeof = -1;
		}
	}
#endif
	diskpos = p;
	if(lseek(tfd,diskpos,SEEK_SET) < 0) {
		perror("lseek");
		T_abort();
	}
}

static void disk_truncate(int p)
{
#if 0
	if(p != diskpos) disk_seek(p);
#endif
	diskeof = p;
}

static void disk_write(int p, int l, unsigned char *b)
{
	int k;
	int ll = l;
	unsigned char *bb = b;


	if(p != diskpos) disk_seek(p);

	while(ll > 0) {
		if((k = write(tfd,bb,ll)) < 0) {
			perror("write");
			T_abort();
		}
		ll -= k;
		bb += k;
	}
	diskpos = diskeof = p + l;
}

static void disk_read(int p, int l, unsigned char *b)	/* ggf. fill in T_EOM */
{
	int k;
	int ll = l;
	unsigned char *bb = b;


	if(p != diskpos) disk_seek(p);

	while(ll > 0) {
		k = read(tfd,bb,ll);
		if(k < 0) {
			perror("read");
			T_abort();
		}
		if(k == 0) {			/* EOF */
			if(ll == 4) {		/* 4 bytes remain => ok */
				*((int *)bb) = T_EOM;	/* provide sentinel */
				diskpos = p + (l - 4);
				return;			/* success */
			} else {
				fprintf(stderr,"read() EOF at %d\n",p + l - ll);
				T_abort();
			}
		} else {
			ll -= k;
			bb += k;
		}
	}

	diskpos = p + l;
}
	
/*****/

static void TOP_rewind()	/* permitted to occur at BOT, verify magic */
{
	if(T_bot) return;

	tp = sizeof(t_magic);
	disk_read(0,sizeof(t_magic) + 4,tbuf);
	T_nextlen = I4(tbuf + sizeof(t_magic));
	T_checklen(T_nextlen);
	if(memcmp(&t_magic,tbuf,sizeof(t_magic))) T_abort();

	T_prevlen = T_EOM;
}

static void TOP_bsr()		/* must not occur at BOT */
{
	tp = T_prevpos;
	if(T_bot) {
		int oldprevlen = T_prevlen;

		TOP_rewind();
		if(T_nextlen != oldprevlen) T_abort();
	} else {
		disk_read(tp - 4,2*4,tbuf);
		if(I4(tbuf + 4) != T_prevlen) T_abort();
		T_nextlen = T_prevlen;
		T_prevlen = I4(tbuf);
		T_checklen(T_prevlen);
	}
}

static void TOP_fsr()		/* must not occur at EOM */
{
	tp = T_nextpos;
	disk_read(tp - 4,2*4,tbuf);
	if(I4(tbuf) != T_nextlen) T_abort();
	T_prevlen = T_nextlen;
	T_nextlen = I4(tbuf + 4);
	T_checklen(T_nextlen);
}

static void TOP_read()		/* must not occur at EOM */
{
	disk_read(tp + 4,T_nextlen + 2*4,tbuf);
 	tp = T_nextpos;
	if(I4(tbuf + T_nextlen) != T_nextlen) T_abort();
	T_prevlen = T_nextlen;
	T_nextlen = I4(tbuf + T_nextlen + 4);
	T_checklen(T_nextlen);
}

static void TOP_write(int l, unsigned char *b)	/* also used for writemark */
{
	T_checklen(l);

	if(!T_eom) disk_truncate(tp);

	T_nextlen = l;
	disk_write(tp,4,(unsigned char *)&T_nextlen);
	disk_write(tp + 4,T_nextlen,b);
	disk_write(tp + 4 + T_nextlen,4,(unsigned char *)&T_nextlen);
	tp = T_nextpos;

	T_prevlen = l;
	T_nextlen = T_EOM;
}
		
/*****/

void tape_init(char *fn)
{
	char *mp;
	char fnbuf[256];


	diskpos = 0;

	if(!fn) fn = "ZT_DISK_TAPE";	/* could be a logical name */

	fprintf(stderr,"Going to open %s ...\n",fn);
#if 0
	if((tfd = open(fn,O_RDWR,0,"ctx=stm","mbc=16")) >= 0) {
#else
	if((tfd = open(fn,O_RDWR,0)) >= 0) {
#endif
		mp = "opened r/w";
		t_ro = 0;
#if 0
	} else if((tfd = open(fn,O_RDONLY,0,"ctx=stm","mbc=16")) >= 0) {
#else
	} else if((tfd = open(fn,O_RDONLY,0)) >= 0) {
#endif
		mp = "opened r/o";
		t_ro = 1;
#if 0
	} else if((tfd = creat(fn,0,"ctx=stm","mbc=16")) >= 0) {
#else
	} else if((tfd = creat(fn,0)) >= 0) {
#endif
		mp = "created";
		t_ro = 0;
		disk_write(0,sizeof(t_magic),(unsigned char *)&t_magic);
	} else {
		perror("open/creat");
		T_abort();
	}
	fprintf(stderr,"File %s %s\n",getname(tfd,fnbuf),mp);

	TOP_rewind();

#if TRACE
	tpos = 0;
#endif

	hw_online = 1;
	hw_medonline = 1;
	hw_hwl = t_ro;

	hw_eof = 0;
	hw_bot = T_bot;
	hw_eot = T_eot;
}

/*****/

unsigned tape_writemark(void)
{
#if TRACE
	trace_tape("WRITEMARK",0,0,0);
#endif
	if(!hw_online) return SS$_DEVOFFLINE;
	if(!hw_medonline) return SS$_MEDOFL;
	if(hw_hwl) return SS$_WRITLCK;

	TOP_write(T_MARK,(unsigned char *)"\026");
#if TRACE
	tpos ++;
#endif

	hw_eof = 1; 		/* ??? */
	hw_bot = T_bot;
	hw_eot = T_eot;

	return SS$_NORMAL;
}

unsigned tape_nop(void)
{
#if TRACE
	trace_tape("NOP",0,0,0);
#endif
	/* tape status does not change ... */
	if(!hw_online) return SS$_DEVOFFLINE;
	if(!hw_medonline) return SS$_MEDOFL;
	return SS$_NORMAL;
}

unsigned tape_rewind(int/*logical*/ unload)
{
#if TRACE
	trace_tape("REWIND",1,unload,0);
#endif
	if(!hw_online) return SS$_DEVOFFLINE;
	if(!hw_medonline) return SS$_MEDOFL;

	/* leave (mysteriously) medium online in spite of unload */
	/* i.e. 'unload' is ignored(!) */

	TOP_rewind();
#if TRACE
	tpos = 0;
#endif

	hw_eof = 0;
	hw_bot = T_bot;
	hw_eot = T_eot;

	return SS$_NORMAL;
}

unsigned tape_skiprec(int sb,int *sap)
/* sb = (signed) # blocks to be skipped
   sap = pointer to return value: absolute # blocks skipped */
/* terminate after EOF mark, or at BOT */
{
	unsigned xstat;


#if TRACE
	trace_tape("SKIPREC",1,sb,0);
#endif
	if(!hw_online) return SS$_DEVOFFLINE;
	if(!hw_medonline) return SS$_MEDOFL;

	*sap = 0;

	xstat = SS$_NORMAL;	/* assume success, 
				this includes EOF & BOT termination */
	if(sb > 0) {
		do {
			if(T_eom) {
				xstat = SS$_TAPEPOSLOST;
				break;
			}
			hw_eof = (T_nextlen == T_MARK);
			TOP_fsr();
#if TRACE
			tpos ++;
#endif
			(*sap) ++;
		} while(!hw_eof && (*sap < sb));
	} else if(sb < 0) {
		do {
			if(T_bot) break;
			hw_eof = (T_prevlen == T_MARK);
			TOP_bsr();
#if TRACE
			tpos --;
#endif
			(*sap) ++;
		} while(!hw_eof && ((*sap + sb) < 0));
	}

	hw_bot = T_bot;
	hw_eot = T_eot;

	return xstat;
}

unsigned tape_compare(int/*logical*/ reverse)
/* (buffer,bufbct) has memory data.
	compare to tape block,
	return length of tape block in bufbct */
{
	unsigned xstat;


#if TRACE
	trace_tape("COMPARE",1,reverse,0);
#endif
	if(!hw_online) return SS$_DEVOFFLINE;
	if(!hw_medonline) return SS$_MEDOFL;

	if(reverse) return SS$_BADPARAM;	/* reverse not supported */


	xstat = SS$_NORMAL;	/* assume success, includes EOF read */

	if(T_eom) {
		xstat = SS$_TAPEPOSLOST;
	} else {
		TOP_read();
		hw_eof = (T_prevlen == T_MARK);
		if(hw_eof) {
			*bufbctp = 0;
		} else {
			if(*bufbctp < T_prevlen ||
			   memcmp(T_data,bufp,T_prevlen)) {
				xstat = SS$_DATACHECK;
			}
			*bufbctp = T_prevlen;
		}
#if TRACE
		tpos ++;
#endif
	}

	hw_bot = T_bot;
	hw_eot = T_eot;

	return xstat;
}


unsigned tape_read(int/*logical*/ reverse, int/*logical*/ check)
/* put data into (buffer,bufbct) */
/* 'check' (ignored) requests data checking */
{
#if TRACE
	trace_tape("READ",2,reverse,check);
#endif
	if(!hw_online) return SS$_DEVOFFLINE;
	if(!hw_medonline) return SS$_MEDOFL;

	if(reverse) return SS$_BADPARAM;	/* reverse not supported */

	if(T_eom) return SS$_TAPEPOSLOST;

	TOP_read();
	hw_eof = (T_prevlen == T_MARK);
	if(hw_eof) {
		*bufbctp = 0;
	} else {
		*bufbctp = T_prevlen;
		memcpy(bufp,T_data,T_prevlen);
	}
#if TRACE
	tpos ++;
#endif

	hw_bot = T_bot;
	hw_eot = T_eot;

	return SS$_NORMAL;
}

unsigned tape_write(int/*logical*/ check)
/* process data from (buffer,bufbct) */
/* 'check' (ignored) requests data checking */
{
#if TRACE
	trace_tape("WRITE",1,check,0);
#endif
	if(!hw_online) return SS$_DEVOFFLINE;
	if(!hw_medonline) return SS$_MEDOFL;
	if(hw_hwl) return SS$_WRITLCK;

	TOP_write(*bufbctp,bufp);
#if TRACE
	tpos ++;
#endif

	hw_eof = 0;
	hw_bot = T_bot;
	hw_eot = T_eot;

	return SS$_NORMAL;
}


/*****/

#if TRACE

static void trace_tape(char *fname,int ac,int a1,int a2)
{
	fprintf(stdout,"\tTAPE_%s(",fname);
	if(ac > 0) {
		fprintf(stdout,"%d",a1);
		if(ac > 1) {
			fprintf(stdout,",%d",a2);
		}
	}
	fprintf(stdout,") tpos=%d\n",tpos);
}

#endif
