/****************************************************************************/
/*                                                                          */
/*                    ****  COPYRIGHT NOTICE ****                           */
/*                                                                          */
/*                                                                          */
/*  Copyright (c) Digital Equipment Corporation, 1992                       */
/*                                                                          */
/*  All Rights Reserved.  Unpublished rights reserved under the             */
/*  copyright laws of the United States.                                    */
/*                                                                          */
/*  The software contained on this media is proprietary to and              */
/*  embodies the confidential technology of  Digital Equipment              */
/*  Corporation.  Possession, use, duplication or dissemination             */
/*  of the software and media is authorized only pursuant to a              */
/*  valid written license from Digital Equipment Corporation.               */
/*                                                                          */
/*                                                                          */
/*  The information in this software is subject to change  without  notice  */
/*  and  should  not  be  construed  as  a commitment by DIGITAL EQUIPMENT  */
/*  CORPORATION.                                                            */
/*                                                                          */
/*  DIGITAL assumes no responsibility for the use or  reliability  of  its  */
/*  software on equipment which is not supplied by DIGITAL.                 */
/*                                                                          */
/*                                                                          */
/****************************************************************************/

/************************************************************************
 **
 **
 **   FACILITY:          DECss7 IVP (Installation Verification Procedure)
 **
 **   ENVIRONMENT:       OSF/1 V3.0
 **
 **   MODULE NAME:       ss7_hlr.c
 **
 **   DESCRIPTION:       HLR part of the DECss7 IVP.
 **                      Mono-thread version.
 **         
 **   AUTHORS:          Marc Beatini
 **
 **   CREATION DATE:    17 February 1995
 **
 **   MODIFICATION HISTORY:
 **       Name       (UserID)        Date        Comments
 **
 **       Pierre Garnero             07-May-1996 Port on VMS
 **
 **       Yves Schneider             11-Sep-1997 Add TCAP on TCPIP
 **
 **       Yves Schneider   30-Dec-1997   ADD SIGNALS HANDLER.
 **
 **       Yves Schneider             03-Jun-1998 Use the TCAP 96 API.
 **
 ************************************************************************/

/*
**
**  INCLUDE FILES
**
*/
#include <stdio.h>
#ifndef SS7_TCAP_96_API
#define SS7_TCAP_96_API
#endif
#ifdef WIN32
#include "ss7api.h"
#include "ss7_cond_codes.h"
#include "ss7_tcap_k_common.h"
#else
#include <ss7api.h>
#include <ss7_cond_codes.h>
#include <ss7_tcap_k_common.h>
#endif

#ifdef __unix__
#include <errno.h>
#include <unistd.h>
#include <signal.h>
#elif WIN32
#include <windows.h>
#endif

/***************/
/* Local files */
/***************/
#include "ss7_ivp_defines.h"
#include "ss7_ivp_typedefs.h"

/*
**
** Table of contents
**
*/

void hlr_invoke_indication_ass(to_process_t *);
void hlr_write_cell_id_message (int );
static int examine_request();

/*
**
** External References
**
*/
#include "ss7_ivp_tools.h"


/*
**
** Global Definitions
**
*/
char exit_program = SS7_K_FALSE;

#ifdef WIN32
#ifdef CCITT
extern void init_tcapport_ccitt();
#endif
#ifdef ANSI
extern void init_tcapport_ansi();
#endif
#endif

/*
**++
**  FUNCTIONAL DESCRIPTION:
**
**      Main line of HLR.
**
**  FORMAL PARAMETERS:
**
**      None.
**
**  RETURN VALUE:
**
**      None
**
**  SIDE EFFECTS:
**
**      None
**--
*/
void 
main ()
{
   unsigned int status;
   ss7_tcap_vector_t indications = {
	    begin_ind_mess,
 	    continue_ind_mess,
	    end_ind_mess,
	    abort_ind_mess,
	    invoke_ind_mess,
	    result_ind_mess,
	    error_ind_mess,
	    reject_ind_mess,
	    cancel_ind_mess,
	    0,                               /* unidirectional */
	    0,                               /* notice */
	    0                                /* user_distrib */
   };
   void *evp;                                /* ss7 event context block pointer */

#ifdef WIN32
   int delay = 10000;                           /* Before "real" start */
   int delay_exit = 2000;                       /* Before exiting */

#ifdef CCITT
   init_tcapport_ccitt();
#endif
#ifdef ANSI
   init_tcapport_ansi();
#endif

#else
#ifdef VMS
   float delay = 10.0;                       /* Before "real" start */
   float delay_exit = 2.0;                   /* Before exiting */
#else
   int delay = 10;                           /* Before "real" start */
   int delay_exit = 2;                       /* Before exiting */
   /* ADD SIGNALS HANDLER. */
   sigset_t sig_set;

   sigemptyset(&sig_set);
   sigaddset(&sig_set, SIGXCPU);
   sigaddset(&sig_set, SIGABRT);
   sigaddset(&sig_set, SIGTERM);
   sigaddset(&sig_set, SIGURG);
#ifdef V3.2
   sigprocmask( SIG_BLOCK, &sig_set, 0 );
#else
   pthread_sigmask( SIG_BLOCK, &sig_set, 0 );
#endif
#endif /* VMS */
#endif /* WIN32 */

   /* Init queues */
    init_queue_free();
    init_queue_busy();

    /* First register to DECss7 */					
#ifdef CCITT
    status = ss7_tcap_enable_indic_ccitt( 
					 &indications, 
					 SS7_K_TRUE, 
					 SS7_K_TCAP_WHITE_BOOK 
					 ); 
#elif ANSI
    status = ss7_tcap_enable_indic_ansi(
					 &indications, 
					 SS7_K_TRUE, 
					 SS7_K_TCAP_ANSI_96 
					 ); 
#elif TTC
    status = ss7_tcap_enable_indic_ccitt(
					 &indications, 
					 SS7_K_TRUE, 
					 SS7_K_TCAP_WHITE_BOOK 
					 ); 
#elif CCI24
    status = ss7_tcap_enable_indic_ccitt(
					 &indications, 
					 SS7_K_TRUE, 
					 SS7_K_TCAP_WHITE_BOOK 
					 ); 
#else
    PROTOCOL NOT SUPPORTED
#endif
    $check(status);

    /* Wait a little bit ... */
#ifdef VMS
    lib$wait(&delay);
#elif WIN32
    Sleep(delay);
#else
    sleep(delay);
#endif

    /* ... then start */
    printf("\nHLR program started\nWaiting for Message from VLR...\n\n");

    /* Wait VLR messages and process them forever */
    for (;;) {

       /* Wait SS7 indication */
#ifdef CCITT
       status = ss7_tcap_get_event_ccitt( &evp );
#elif ANSI
       status = ss7_tcap_get_event_ansi( &evp );
#elif TTC
       status = ss7_tcap_get_event_ccitt( &evp );
#elif CCI24
       status = ss7_tcap_get_event_ccitt( &evp );
#else
    PROTOCOL NOT SUPPORTED
#endif

       do {

#ifdef CCITT
	  status = ss7_tcap_deliver_indic_ccitt( evp );
#elif ANSI
	  status = ss7_tcap_deliver_indic_ansi( evp );
#elif TTC
	  status = ss7_tcap_deliver_indic_ccitt( evp );
#elif CCI24
	  status = ss7_tcap_deliver_indic_ccitt( evp );
#else
    PROTOCOL NOT SUPPORTED
#endif

       } while (status != SS7_NOMOREINDIC);

       /* We get at least one, so process it */
       do {
	  status = examine_request();
       } while (status != Q_WAS_EMPTY);
       
       if (exit_program) break;

    }      /* for   */

    /* Now we can exit... after some delay... */
#ifdef VMS
    lib$wait(&delay_exit);
#elif WIN32
    Sleep(delay_exit);
#else
    sleep(delay_exit);
#endif

    printf("\nHLR program terminated\n");
    exit(0);
}


/*
**++
**  FUNCTIONAL DESCRIPTION:
**
**      Process an invoke component received from the VLR.
**	Send back a result-component and a end-dialog.
**
**  FORMAL PARAMETERS:
**
**      to_process_t *
**
**  RETURN VALUE:
**
**      None.
**
**  SIDE EFFECTS:
**
**      None.
**--
**/
void hlr_invoke_indication_ass(to_process_t *to_process)
{
    int cell_id;
    unsigned int status;
    char null_param[7]="nothing";  /* Nothing to answer to "MSC" */
    char null_operation[1]="";     /* Nothing to answer to "MSC" */

    cell_id = (int )((*to_process).val.indic.parm_data[0] );         /* Write the cell ID */
    hlr_write_cell_id_message (cell_id);
    
                                                  /* Be correct and return a result component */
#ifdef CCITT
      status = ss7_tcap_result_component_ccitt ((*to_process).val.indic.dialogue_id,
 					        (*to_process).val.indic.component_id,
					        (*to_process).val.indic.ope_data,          /* operation */
						(*to_process).val.indic.ope_len,           /* operation length */
					        SS7_K_TCAP_SET,
					        null_param,
						sizeof(null_param),
					        SS7_K_TRUE );
#elif ANSI
      status = ss7_tcap_result_component_ansi ( (*to_process).val.indic.dialogue_id,
					        (*to_process).val.indic.component_id,
					        null_operation,
					        sizeof(null_operation),
					        SS7_K_TCAP_SET,
                                                null_param,
						sizeof(null_param),
					        SS7_K_TRUE );
#elif TTC
      status = ss7_tcap_result_component_ccitt ((*to_process).val.indic.dialogue_id,
 					        (*to_process).val.indic.component_id,
					        (*to_process).val.indic.ope_data,          /* operation */
						(*to_process).val.indic.ope_len,           /* operation length */
					        SS7_K_TCAP_SET,
					        null_param,
						sizeof(null_param),
					        SS7_K_TRUE );
#elif CCI24
      status = ss7_tcap_result_component_ccitt ((*to_process).val.indic.dialogue_id,
 					        (*to_process).val.indic.component_id,
					        (*to_process).val.indic.ope_data,          /* operation */
						(*to_process).val.indic.ope_len,           /* operation length */
					        SS7_K_TCAP_SET,
					        null_param,
						sizeof(null_param),
					        SS7_K_TRUE );
#else
   PROTOCOL NOT SUPPORTED
#endif
      $check(status);

      status = write_result_component_mess(
					   (*to_process).val.indic.dialogue_id,
					   (*to_process).val.indic.component_id,
					   (*to_process).val.indic.ope_data,          /* operation */
					   (*to_process).val.indic.ope_len,           /* operation length */
					   null_param,
					   sizeof(null_param),
					   SS7_K_TRUE
					   );
      $check(status);

                                                        /* End the dialogue (sends the message) */
#ifdef CCITT
      status = ss7_tcap_end_dialog_ccitt (
#elif ANSI
      status = ss7_tcap_end_dialog_ansi (
#elif TTC
      status = ss7_tcap_end_dialog_ccitt (
#elif CCI24
      status = ss7_tcap_end_dialog_ccitt (
#else
      PROTOCOL NOT SUPPORTED
#endif
					  (*to_process).val.indic.dialogue_id,
					  SS7_K_TCAP_BASIC_END,   /* termination */
#ifdef TCAPIP
					  32,                     /* quality_of_service */
#else
					  0,                      /* quality_of_service */
#endif
                                          0,                      /* ACN obj or int */
					  0,                      /* application_context_name */
					  0,                      /* application_context_name_length */
					  0,                      /* application_context_info */
					  0,                      /* application_context_info_length */
                                          0,                      /* secu obj or int */
                                          0,                      /* security_context */
                                          0,                      /* security_context_length */
                                          0,                      /* confidentiality */
                                          0                       /* confidentiality_length */
					 );
      $check(status);

      status = write_end_dialogue_mess (
					(*to_process).val.indic.dialogue_id,
					SS7_K_TCAP_BASIC_END,     /* termination */
#ifdef TCAPIP
                                        32                        /* quality_of_service */
#else
                                        0                         /* quality_of_service */
#endif
				       );
      $check(status);

      if (cell_id == 0) exit_program = SS7_K_TRUE;
}


/*
**++
**  FUNCTIONAL DESCRIPTION:
**
**      Display the Cell_id coordinates on screen.
**
**  FORMAL PARAMETERS:
**
**	cell_id : Cell_id co-ordinates.
**
**  RETURN VALUE:
**
**      None.
**
**  SIDE EFFECTS:
**
**      None
**--
*/
void hlr_write_cell_id_message (int cell_id )
{
    printf("Cell coordinates: %d\n", cell_id);
}



/*
**++
**  FUNCTIONAL DESCRIPTION:
**
**      Examine and process a request received from the VLR.
**
**  FORMAL PARAMETERS:
**
**	cell_id : Cell_id co-ordinates.
**
**  RETURN VALUE:
**
**      Q_SUCCESS
**      Q_WAS_EMPTY
**      Q_FAIL
**
**  SIDE EFFECTS:
**
**      None
**--
*/
int examine_request()
{
   int status;
   struct queue *queue_elem;
   to_process_t *to_process;

   status = rem_from_busy( &queue_elem, &to_process);     /* get a buffer from busy queue */      

   if (status != Q_SUCCESS ) {
      return (Q_WAS_EMPTY);
   }

   if (to_process != NULL) {
      switch ((*to_process).ind_type)
      {
      case WRITE_INVOKE_C:
	 write_invoke_ass_mess (to_process);
	 hlr_invoke_indication_ass(to_process);
	 break;
	 
      case WRITE_BEGIN_C:
	 write_begin_ass_mess(to_process);
	 break;
	 
      case WRITE_CONTINUE_C:
	 write_continue_ass_mess(to_process);
	 break;
	 
      case WRITE_ABORT_C:
	 write_abort_ass_mess(to_process);
	 break;
	 
      case WRITE_ERROR_C:
	 write_error_ass_mess(to_process);
	 break;
	 
      case WRITE_REJECT_C:
	 write_reject_ass_mess(to_process);
	 break;
	 
      case WRITE_RESULT_C:
	 write_result_ass_mess(to_process);
	 break;
	 
      case WRITE_CANCEL_C:
	 write_cancel_ass_mess(to_process);
	 break;
	 
      case WRITE_END_C:
	 write_end_ass_mess(to_process);
	 break;
	 
      } /* switch */

      (*to_process).ind_type = 0;                                   /* record free              */   

      status = put_at_free( queue_elem);                 /* reput elem in free queue */

      if (status != Q_SUCCESS) {
	 fprintf(stderr,"error reputing a buffer in free queue\n");
      }
   } else {
      fprintf(stderr,"busy queue object pointer NULL, abnormal problem");
      return (Q_FAIL);
   }

   return(status);   
}








