/****************************************************************************
/*
/* Copyright Digital Equipment Corporation 1985,1995.  All rights reserved.
/* 
/* Restricted Rights: Use, duplication, or disclosure by the U.S. Government 
/* is subject to restrictions as set forth in subparagraph (c) (1) (ii) of
/* DFARS 252.227-7013, or in FAR 52.227-19, or in FAR 52.227-14 Alt. III, as 
/* applicable.
/* 
/* This software is proprietary to and embodies the confidential technology 
/* of Digital Equipment Corporation.  Possession, use, or copying of this 
/* software and media is authorized only pursuant to a valid written license 
/* from Digital or an authorised sublicensor.
/*
/*****************************************************************************/


#include <stdio.h>
#include <stddef.h>
#include <stdlib.h>
#include <string.h>

#include <osak_api.h>
#include <osak_api_codes.h>
#include <osak_api_message.h>

#define TRUE 1
#define FALSE 0
#define OSAK_EXAMPLE_WS_SIZE 1024
#define OSAK_EXAMPLE_BUFFER_SIZE 2048
#define OSAK_EXAMPLE_TIMEOUT 60

void open_initiator (osak_port *port, struct osak_parameter_block *pb) ;
unsigned long int assoc_req (osak_port port, struct osak_parameter_block *pb) ;
unsigned long int release_req (osak_port port,
			       struct osak_parameter_block *pb) ;
void give_buffer (osak_port port) ;
void reuse_buffers (struct osak_buffer **ptr) ;
void wait_for_event (osak_port port, struct osak_parameter_block *pb,
		     int queued) ;
void *alloc_memory (unsigned long int size, unsigned long int param) ;
unsigned long int free_memory (unsigned long int size,
		void *ptr, unsigned long int param) ;

extern unsigned long int osak_open_initiator () ;
extern unsigned long int osak_associate_req () ;
extern unsigned long int osak_release_req () ;
extern unsigned long int osak_close_port () ;
extern unsigned long int osak_get_event () ;
extern unsigned long int osak_collect_pb () ;
extern unsigned long int osak_give_buffers () ;
extern unsigned long int osak_select () ;

/*--------------------------------------------------------------------------*/
/* Constants and pre-encoded ASN.1 values				    */
/*--------------------------------------------------------------------------*/

    /* N O T E : Change this to be the NSAP for your machine */
    /* _______						     */

unsigned char remote_nsap[] =
    { 0x49, 0x00, 0x2A, 0xAA, 0x00, 0x04, 0x00, 0x99, 0xA9, 0x21 } ;

    /* Application Context */
unsigned char a_context_buf[] =			/* {1 0 8571 1 1} */
	{0x06, 0x05, 0x28, 0xC2, 0x7B, 0x01, 0x01} ;

    /* DCS proposal list */
unsigned char abstract_1[] =			/* {2 2 1 0 1} (ACSE) */
	{0x06, 0x04, 0x52, 0x01, 0x00, 0x01};
unsigned char abstract_2[] =			/* {2 1 9999 9999 2}   */
	{0x06, 0x06, 0x51, 0xCE, 0x0F, 0xCE, 0x0F, 0x02};

    /* Object Identifier for Basic Encoding Rules */
unsigned char ber[] = {0x06, 0x02, 0x51, 0x01};	/* {2 1 1} */

    /* Presentation context identifier 1 (BER encoding for INTEGER) */
unsigned char pcid_1[] = {0x02, 0x01, 0x01} ;

    /* Presentation context identifier 3 (BER encoding for INTEGER) */
unsigned char pcid_3[] = {0x02, 0x01, 0x03} ;

    /* ASN.1 encoding of user-information to be sent on osak_associate_req  */
    /* call.								    */

unsigned char user_information[] =
    {	0xBE, 0x23, 0x28, 0x21,	    /* [30] IMPLICIT SEQUENCE OF EXTERNAL   */
	0x06, 0x02, 0x51, 0x01,	    /* direct-reference = {2 1 1}	    */
	0x02, 0x01, 0x03,	    /* indirect-reference = 3		    */
	0xA0, 0x18,		    /* single-ASN.1-type 		    */
	0x04, 0x16,		    /* OCTET STRING			    */
	'I', 'n', 'i', 't', 'i', 'a', 't', 'o', 'r', ' ', 't', 'o', ' ',
	'R', 'e', 's', 'p', 'o', 'n', 'd', 'e', 'r'
    } ;

/*--------------------------------------------------------------------------*/
/* Global variables							    */
/*--------------------------------------------------------------------------*/
    /* Application context name */
osak_mem_descriptor application_context ;

    /* Transfer syntax list. Used in the presentation context proposal list */
struct osak_ts_list transfer1 ;

    /* The two presentation contexts in the presentation context proposal list */
struct osak_pcontext_proposal context2 ;
struct osak_pcontext_proposal context1 ;

    /* Local AEI */
struct osak_aei local_address ;

    /* Remote AEI */
struct osak_aei remote_address ;

    /* Transport template */
struct osak_transport_templates transport_template ;

    /* Protocol versions */
struct osak_protocol_versions protocol_versions ;

    /* Functional units */
struct osak_fus fus ;

    /* Data to send on the association req */
struct osak_buffer send_buffer ;

    /* Queue of buffer for osak_give_buffers. These are used in the	    */
    /* routines give_buffer and reuse_buffers.				    */
struct osak_buffer *free_buffers = NULL ;
struct osak_buffer *free_buffers_end = NULL ;


/****************************************************************************/
/* FUNCTION: main							    */
/****************************************************************************/
main (int argc, char *argv[])
{
    unsigned long int status ;		    /* To hold return values from   */
					    /* routines                     */
    struct osak_parameter_block *in_pb ;    /* Parameter block to use in    */
					    /* calls to osak_get_event      */
    struct osak_parameter_block *out_pb ;   /* Parameter block to use in    */
					    /* outbound calls               */
    osak_port port ;			    /* To hold the port handle      */
					    /* returned from                */
					    /* osak_open_initiator          */
    struct osak_buffer *buffer_ptr ;	    /* To hold list of buffers      */
					    /* returned from                */
					    /* osak_close_port              */
    struct osak_parameter_block *pb_ptr ;   /* To hold list of parameter    */
					    /* blocks returned from         */
					    /* osak_close_port              */
    int need_to_collect_pb ;		    /* Flag for call to		    */
					    /* wait_for_event.  TRUE if the */
					    /* call to osak_select should   */
					    /* wait for OSAK_C_WRITEEVENT,  */
					    /* to guarantee the completion  */
					    /* of an outbound call.	    */

    /*----------------------------------------------------------------------*/
    /* Allocate a parameter block.  The same parameter block is used on all */
    /* outbound calls to OSAK.						    */
    /*----------------------------------------------------------------------*/
    out_pb = (struct osak_parameter_block *)
	malloc (sizeof(struct osak_parameter_block) + OSAK_EXAMPLE_WS_SIZE) ;
    if (out_pb == NULL)
    {
	printf ("Failed to allocate outbound parameter block\n") ;
	exit (0) ;
    }

    /*----------------------------------------------------------------------*/
    /* Allocate a parameter block.  The same parameter block is used on all */
    /* inbound calls to OSAK.						    */
    /*----------------------------------------------------------------------*/
    in_pb = (struct osak_parameter_block *)
	malloc (sizeof(struct osak_parameter_block) + OSAK_EXAMPLE_WS_SIZE) ;
    if (in_pb == NULL)
    {
	printf ("Failed to allocate inbound parameter block\n") ;
	exit (0) ;
    }

    /*----------------------------------------------------------------------*/
    /* Open Initiator							    */
    /*----------------------------------------------------------------------*/
    printf ("Opening an initiator port\n") ;
    open_initiator (&port, in_pb) ;

    /*----------------------------------------------------------------------*/
    /* Associate Request						    */
    /*----------------------------------------------------------------------*/
    printf ("Sending A-ASSOCIATE-request.\n") ;
    status = assoc_req (port, out_pb) ;
    if ((status != OSAK_S_NORMAL) && (status != OSAK_S_QUEUED))
    {
	printf ("Call to osak_associate_req failed\n") ;
	exit (0) ;
    }
    if (status == OSAK_S_QUEUED)
	need_to_collect_pb = TRUE ;
    else
	need_to_collect_pb = FALSE ;

    /*----------------------------------------------------------------------*/
    /* Wait for accept cnf and collect assoc_req pb			    */
    /*----------------------------------------------------------------------*/
    printf ("Waiting for A-ACCEPT-confirm\n") ;
    if (need_to_collect_pb)
	printf ("Also waiting for completion of A-ASSOCIATE-request.\n") ;

    wait_for_event (port, in_pb, need_to_collect_pb) ;

    /* A real application would need to handle the receipt of		    */
    /* A-REJECT-confirm and A-ABORT-ind as well				    */

    if (in_pb->event_type != OSAK_C_ACCEPT_CNF)
    {
	printf ("Expected A-ACCEPT-confirm. Received some other event\n") ;
	exit (0) ;
    }
    printf ("Received A-ACCEPT-confirm.\n") ;
    reuse_buffers (&in_pb -> tsdu_ptr) ;

    /*----------------------------------------------------------------------*/
    /* Release the association						    */
    /*----------------------------------------------------------------------*/
    status = release_req (port, out_pb) ;
    if ((status != OSAK_S_NORMAL) && (status != OSAK_S_QUEUED) &&
	(status != OSAK_S_FREE))
    {
	printf ("Failed in call to osak_release_req\n") ;
	exit (0) ;
    }
    if (status == OSAK_S_QUEUED || status == OSAK_S_FREE)
	need_to_collect_pb = TRUE ;
    else
	need_to_collect_pb = FALSE ;

    /*----------------------------------------------------------------------*/
    /* Wait for release-cnf and collect release_req pb			    */
    /*----------------------------------------------------------------------*/
    printf ("Waiting for A-RELEASE-confirm\n") ;
    if (need_to_collect_pb)
	printf ("Also waiting for completion of A-RELEASE-request\n") ;

    wait_for_event (port, in_pb, need_to_collect_pb) ;

    if (in_pb->event_type != OSAK_C_RELEASE_CNF)
    {
	printf ("Expected A-RELEASE-confirm. Received some other event\n") ;
	exit (0) ;
    }
    printf ("Received A-RELEASE-confirm\n") ;
    reuse_buffers (&in_pb -> tsdu_ptr) ;

    /*----------------------------------------------------------------------*/
    /* Close the port							    */
    /*----------------------------------------------------------------------*/
    printf ("Closing initiator port\n");
    status = osak_close_port (port, &buffer_ptr, &pb_ptr,
			      OSAK_C_NON_DESTRUCTIVE) ;
    if (status != OSAK_S_NORMAL)
    {
	printf ("Failed in call to osak_close_port\n") ;
	exit (0) ;
    }
    reuse_buffers (&buffer_ptr) ;
}

/****************************************************************************/
/* FUNCTION: open_initiator						    */
/*									    */
/* This routine sets up the parameter for a call to osak_open_initiator and */
/* makes the call.							    */
/*									    */
/****************************************************************************/

void open_initiator (osak_port *port, struct osak_parameter_block *pb)
{
    unsigned long int status ;

    /* initialize parameter block */
    memset ((void *)pb, '\0',
	sizeof(struct osak_parameter_block) + OSAK_EXAMPLE_WS_SIZE ) ;
    pb->pb_length = sizeof (struct osak_parameter_block) ;
    pb->ws_length = OSAK_EXAMPLE_WS_SIZE ;
    pb->api_version = OSAK_C_API_VERSION_3 ;
    pb->alloc_rtn = (osak_rtn) alloc_memory ;
    pb->dealloc_rtn = (osak_rtn) free_memory ;
    pb->alloc_param = 0 ;
    pb->completion_rtn = NULL ;
    pb->completion_param = 0 ;

    status = osak_open_initiator (port, pb) ;
    if (status != OSAK_S_NORMAL)
    {
	printf ("Call to osak_open_initiator failed\n") ;
	exit (0) ;
    }
}

/****************************************************************************/
/* FUNCTION: assoc_req							    */
/*									    */
/* This routine sets up the parameters for a call to osak_associate_req and */
/* makes the call.							    */
/*									    */
/****************************************************************************/
unsigned long int
assoc_req (osak_port port, struct osak_parameter_block *pb)
{
    unsigned long int status ;

    /* Set up application context name */
    application_context.size = 7 ;
    application_context.pointer = a_context_buf ;

    /* Set up presentation context proposal list */
    
    /* Transfer syntax list */
    transfer1.next = NULL ;
    transfer1.ts_name.size = 4 ;
    transfer1.ts_name.pointer = ber ;

    context1.pcontext_id.size = 3 ;
    context1.pcontext_id.pointer = pcid_1 ;
    context1.ts_list = &transfer1 ;
    context1.as_name.size = 6 ;
    context1.as_name.pointer = abstract_1 ;
    context1.next = &context2 ;

    context2.pcontext_id.size = 3 ;
    context2.pcontext_id.pointer = pcid_3 ;
    context2.ts_list = &transfer1 ;
    context2.as_name.size = 8 ;
    context2.as_name.pointer = abstract_2 ;
    context2.next = NULL ;

    /* Set up local address */
    local_address.aetitle.aptitle.size = 0 ;
    local_address.aetitle.aptitle.pointer = NULL ;
    local_address.aetitle.ae_qualifier.size = 0 ;
    local_address.aetitle.ae_qualifier.pointer = NULL ;
    local_address.aeiid.apiid.size = 0 ;
    local_address.aeiid.apiid.pointer = NULL ;
    local_address.aeiid.aeiid.size = 0 ;
    local_address.aeiid.aeiid.pointer = NULL ;
    local_address.paddress.psel.size = 9 ;
    local_address.paddress.psel.pointer = (unsigned char *)"INIT-PSEL" ;
    local_address.paddress.ssel.size = 9 ;
    local_address.paddress.ssel.pointer = (unsigned char *)"INIT-SSEL" ;
    local_address.paddress.tsel.size = 9 ;
    local_address.paddress.tsel.pointer = (unsigned char *)"INIT-TSEL" ;
    local_address.paddress.nsap.next = NULL ;
    local_address.paddress.nsap.id.size = 0 ;
    local_address.paddress.nsap.id.pointer = 0 ;
    local_address.paddress.nsap.type = OSAK_C_CLNS ;

    /* Set up peer address (the responder's address) */
    remote_address.aetitle.aptitle.size = 0 ;
    remote_address.aetitle.aptitle.pointer = NULL ;
    remote_address.aetitle.ae_qualifier.size = 0 ;
    remote_address.aetitle.ae_qualifier.pointer = NULL ;
    remote_address.aeiid.apiid.size = 0 ;
    remote_address.aeiid.apiid.pointer = NULL ;
    remote_address.aeiid.aeiid.size = 0 ;
    remote_address.aeiid.aeiid.pointer = NULL ;
    remote_address.paddress.psel.size = 9 ;
    remote_address.paddress.psel.pointer = (unsigned char *)"RESP-PSEL" ;
    remote_address.paddress.ssel.size = 9 ;
    remote_address.paddress.ssel.pointer = (unsigned char *)"RESP-SSEL" ;
    remote_address.paddress.tsel.size = 9 ;
    remote_address.paddress.tsel.pointer = (unsigned char *)"RESP-TSEL" ;
    remote_address.paddress.nsap.next = NULL ;
    remote_address.paddress.nsap.id.size = sizeof(remote_nsap) ;
    remote_address.paddress.nsap.id.pointer = remote_nsap ;
    remote_address.paddress.nsap.type = OSAK_C_CLNS ;

    /* Set up transport template */
    transport_template.next = NULL ;
    transport_template.name.size = 7 ;
    transport_template.name.pointer = (unsigned char *)"Default" ;

    /* Set up protocol versions */
    /* Select session version 2 */
    protocol_versions.acse_version.version1 = 1 ;
    protocol_versions.pversion.version1 = 1 ;
    protocol_versions.sversion.version1 = 0 ;
    protocol_versions.sversion.version2 = 1 ;

    /* Set up functional units */
    /* Zero out all functional units before setting those required */
    memset ((void *)&fus, '\0', sizeof(struct osak_fus)) ;

    /* Request either duplex or half duplex. In this example we are	    */
    /* actually expecting the responder to accept with duplex.		    */
    fus.duplex = 1 ;
    fus.half_duplex = 1 ;

    /* Set up the buffer containing the data to send */
    send_buffer.next = NULL ;
    send_buffer.buffer_ptr	= &user_information[0] ;
    send_buffer.buffer_length	= sizeof(user_information) ;
    send_buffer.data_ptr	= &user_information[0] ;
    send_buffer.data_length	= sizeof(user_information) ;

    /* initialize parameter block */
    memset ((void *)pb, '\0',
	sizeof(struct osak_parameter_block) + OSAK_EXAMPLE_WS_SIZE ) ;    
    pb->pb_length = sizeof (struct osak_parameter_block) ;
    pb->ws_length = OSAK_EXAMPLE_WS_SIZE ;

    pb->acontext = &application_context ;
    pb->pcontext_list = &context1 ;
    pb->calling_aei = &local_address ;
    pb->called_aei = &remote_address ;
    pb->transport_template = &transport_template ;
    pb->functional_units = &fus ;
    pb->protocol_versions = &protocol_versions ;
    pb->user_data = &send_buffer ;

    /* All other fields using default values */

    status = osak_associate_req (port, pb) ;
    return status ;
}

/****************************************************************************/
/* FUNCTION: release_req						    */
/*									    */
/* This routine sets up the parameters for a call to osak_release_req and   */
/* makes the call.							    */
/*									    */
/****************************************************************************/
unsigned long int
release_req (osak_port port, struct osak_parameter_block *pb)
{
    unsigned long int status ;

    /* initialize parameter block */
    memset ((void *)pb, '\0',
	sizeof(struct osak_parameter_block) + OSAK_EXAMPLE_WS_SIZE ) ;
    pb->pb_length = sizeof (struct osak_parameter_block) ;
    pb->ws_length = OSAK_EXAMPLE_WS_SIZE ;
    pb->release_reason = OSAK_C_RLRQ_NORMAL ;

    status = osak_release_req (port, pb) ;
    return status ;
}

/****************************************************************************/
/* FUNCTION: wait_for_event						    */
/*									    */
/* This routine waits for an inbound event to occur.  If there is also a    */
/* queued parameter block from a previous call to OSAK then it also waits   */
/* for that parameter block to be returned by OSAK. Osak_select is used to  */
/* wait for the inbound event and outbound completion.  Osak_get_event is   */
/* used to receive the inbound event and osak_collect_pb is used to get the */
/* parameter block returned by OSAK when the outbound event has completed.  */
/*									    */
/* The example osak_example_resp.c does this differently.  It has two	    */
/* routines to do the same job as this one routine:			    */
/* wait_for_outbound_completion and wait_for_inbound.  This routine shows   */
/* how osak_select can be used to combine those two routines.		    */
/****************************************************************************/
void wait_for_event (osak_port port, struct osak_parameter_block *pb,
		    int queued)
{
    struct osak_parameter_block *ret_pb ;
    osak_handle_count handlecount ;
    osak_handle handle ;
    unsigned long int status ;
    int readevent = TRUE ;
    int writeevent = queued ;
    osak_time select_time = OSAK_EXAMPLE_TIMEOUT ;

    /* Give a buffer to OSAK to get inbound event */
    give_buffer (port) ;

    /* Loop until not waiting for any more events */
    do
    {
	/* Set up parameter to call osak_select() */
	handlecount = 1 ;
	handle.id = (unsigned long int) port ;
	handle.request_mask = 0;
	if (readevent)
	    handle.request_mask |= OSAK_C_READEVENT ;
	if (writeevent)
	    handle.request_mask |= OSAK_C_WRITEEVENT ;
	handle.returned_mask = 0 ;

	status = osak_select (handlecount, &handle, &select_time) ;
	if (status != OSAK_S_NORMAL)
	{
	    printf ("Call to osak_select failed\n") ;
	    exit (0) ;
	}

	/* See if the queued parameter block has been returned */
	if (writeevent && (handle.returned_mask & OSAK_C_WRITEEVENT))
	{
	    ret_pb = NULL ;
	    status = osak_collect_pb (port, &ret_pb) ;
	    if ((status != OSAK_S_NORMAL) && (status != OSAK_S_NOEVENT))
	    {
		printf ("Call to osak_collect_pb failed\n") ;
		exit (0) ;
	    }

	    if (status == OSAK_S_NORMAL && ret_pb != NULL)
	    {
		writeevent = FALSE ;

		/* Look at the status block in the PB returned to see if an */
		/* error occurred					    */
		if (ret_pb->status_block.osak_status_1 != OSAK_S_NORMAL)
		{
		    printf ("error is status block of PB returned from collect pb\n");
		    exit (0) ;
		}
	    }
	}

	/* See if there is an inbound event. If so call osak_get_event() */
	if (readevent && (handle.returned_mask & OSAK_C_READEVENT))
	{
	    do
	    {
		/* Initialize parameter block */
		memset ((void *)pb, '\0',
		  sizeof(struct osak_parameter_block) + OSAK_EXAMPLE_WS_SIZE ) ;
		pb->pb_length = sizeof (struct osak_parameter_block) ;
		pb->ws_length = OSAK_EXAMPLE_WS_SIZE ;
		pb->completion_rtn = NULL ;
		pb->completion_param = 0 ;

		status = osak_get_event (port, pb) ;

		/* If OSAK needs more buffer to decode the event then give  */
		/* more buffers.                                            */
		if (status == OSAK_S_NOBUFFERS)
		{
		    give_buffer (port) ;
		}
	    } while (status == OSAK_S_NOBUFFERS) ;

	    if ((status != OSAK_S_NORMAL) && (status != OSAK_S_NOEVENT))
	    {
		printf ("osak_get_event failed\n") ;
		exit (0) ;
	    }

	    if (status == OSAK_S_NORMAL)
	    {
		readevent = FALSE ;
	    }
	}
    } while (readevent || writeevent) ;
}

/****************************************************************************/
/* FUNCTION: give_buffer						    */
/*									    */
/* This routine is called to pass a buffer to OSAK for OSAK to use to	    */
/* receive inbound events.						    */
/*									    */
/* A list of unused buffers is maintained.  One buffer from this list is    */
/* passed to OSAK using osak_give_buffers.  If the list is empty a new	    */
/* buffer is allocated.							    */
/****************************************************************************/
void give_buffer (osak_port port)
{
    unsigned long int status ;
    struct osak_buffer *give_buf ;

    /* Give a buffer to OSAK */
    if (free_buffers == NULL)
    {
	give_buf = (struct osak_buffer *)malloc (sizeof(struct osak_buffer)) ;
	if (give_buf == NULL)
	{
	    printf ("Failed to allocate an osak_buffer.\n");
	    exit (0) ;
	}
	give_buf -> next = NULL ;
	give_buf -> buffer_length = OSAK_EXAMPLE_BUFFER_SIZE ;
	give_buf -> buffer_ptr =
		    (unsigned char *) malloc (OSAK_EXAMPLE_BUFFER_SIZE) ;
	if (give_buf -> buffer_ptr == NULL)
	{
	    printf ("Failed to allocate buffer.\n") ;
	    exit (0) ;
	}
    }
    else
    {
	give_buf = free_buffers ;
	free_buffers = free_buffers -> next ;
	give_buf -> next = NULL ;
    }

    status = osak_give_buffers (port, give_buf) ;
    if (status != OSAK_S_NORMAL)
    {
	printf ("osak_give_buffers failed\n");
	exit (0) ;
    }
}

/****************************************************************************/
/* FUNCTION: reuse_buffers						    */
/*									    */
/* This routine is called to place buffers returned by OSAK onto the list   */
/* of unused buffers.							    */
/****************************************************************************/
void reuse_buffers (struct osak_buffer **buf_ptr)
{
    struct osak_buffer *buf, *last_buf ;

    buf = *buf_ptr ;
    if (buf == NULL)
	return ;

    last_buf = buf ;
    while (last_buf->next != NULL)
	last_buf = last_buf -> next ;

    if (free_buffers == NULL)
    {
	free_buffers = buf ;
    }
    else
    {
	free_buffers_end->next = buf ;
    }
    free_buffers_end = last_buf ;
    *buf_ptr = NULL ;
}


/****************************************************************************/
/* FUNCTION: alloc_memory						    */
/*									    */
/* Memory allocation routine expected by OSAK				    */
/****************************************************************************/
void * alloc_memory (unsigned long int size, unsigned long int param)
{
    return (void *) malloc (size) ;
}

/****************************************************************************/
/* FUNCTION: free_memory						    */
/*									    */
/* Memory deallocation routine expected by OSAK				    */
/****************************************************************************/
unsigned long int free_memory (unsigned long int size, void *ptr,
			       unsigned long int param)
{
    free (ptr) ;
    return 0;
}
