/****************************************************************************/
/*                                                                          */
/*                    ****  COPYRIGHT NOTICE ****                           */
/*                                                                          */
/*                                                                          */
/*  Copyright (c) Digital Equipment Corporation, 1992                       */
/*                                                                          */
/*  All Rights Reserved.  Unpublished rights reserved under the             */
/*  copyright laws of the United States.                                    */
/*                                                                          */
/*  The software contained on this media is proprietary to and              */
/*  embodies the confidential technology of  Digital Equipment              */
/*  Corporation.  Possession, use, duplication or dissemination             */
/*  of the software and media is authorized only pursuant to a              */
/*  valid written license from Digital Equipment Corporation.               */
/*                                                                          */
/*                                                                          */
/*  The information in this software is subject to change  without  notice  */
/*  and  should  not  be  construed  as  a commitment by DIGITAL EQUIPMENT  */
/*  CORPORATION.                                                            */
/*                                                                          */
/*  DIGITAL assumes no responsibility for the use or  reliability  of  its  */
/*  software on equipment which is not supplied by DIGITAL.                 */
/*                                                                          */
/*                                                                          */
/****************************************************************************/

/************************************************************************
 **
 **
 **   FACILITY:          DECss7 IVP (Installation Verification Procedure)
 **
 **   ENVIRONMENT:       OSF/1 V3.0
 **
 **   MODULE NAME:       ss7_ivp_tools.c
 **
 **   DESCRIPTION:       Procedures/functions to HLR and VLR code.
 **         
 **   AUTHORS:          Marc Beatini
 **
 **   CREATION DATE:    17 February 1995
 **
 **   MODIFICATION HISTORY:
 **       Name       (UserID)        Date        Comments
 **       
 **       Pierre Garnero             09-May-1996 Port on VMS.
 **
 **       Yves Schneider             03-Jun-1998 Use the TCAP 96 API.
 **
 ************************************************************************/

#define _PTHREAD_USE_D4
 

/*
**
**  INCLUDE FILES
**
*/

/*************/
/* Libraries */
/*************/
#include <stdio.h>
#ifndef SS7_TCAP_96_API
#define SS7_TCAP_96_API
#endif
#ifdef VMS
#include <unixio>
#include <errno>
#include <descrip>
#include <ssdef>
#elif WIN32
#include "ss7api.h"
#include <windows.h>
#else
#include <errno.h>
#include <pthread.h>
#endif

#ifdef WIN32
#include "ss7_cond_codes.h"
#else
#include <ss7_cond_codes.h>
#endif

/***************/
/* Local files */
/***************/
#include "ss7_ivp_defines.h"
#include "ss7_ivp_typedefs.h"

/*
**
**  Data structure definitions
**
*/

/***********************/
/* Local variables     */
/***********************/

struct queue queue_free = {&queue_free, &queue_free};    /* free buffer queue */
#ifdef __unix__
static pthread_mutex_t	queue_free_mutex;                /* Mutex to synchronize free-queue access */
#elif WIN32
HANDLE queue_free_mutex;
#endif /* __unix__ */

struct queue queue_busy = {&queue_busy, &queue_busy};    /* to be processed buffer queue */ 
#ifdef __unix__
static pthread_mutex_t	queue_busy_mutex;                /* Mutex to synchronize busy-queue access */
#elif WIN32
HANDLE queue_busy_mutex;
#endif /* __unix__ */

#ifdef QUEUE_BUSY_WAIT
#ifdef __unix__
static pthread_cond_t	queue_busy_cond;                 /* Condition variable to wait on busy-queue if empty */
#elif WIN32
HANDLE queue_busy_cond;
#endif /* __unix__ */
#endif /* QUEUE_BUSY_WAIT */

to_process_t to_process_array[MAX_MSG];                  /* Control Blocks buffer pool */


/*************************************************************************
 ** Table of Contents
 *************************************************************************/

char *get_message (unsigned int);                    /* Return an error message from CAT file */

/* Write on stdout received indication from indication queue */
int write_begin_ass_mess(to_process_t *);
int write_invoke_ass_mess(to_process_t *);
int write_continue_ass_mess(to_process_t *);
int write_abort_ass_mess(to_process_t *);
int write_error_ass_mess(to_process_t *);
int write_reject_ass_mess(to_process_t *);
int write_result_ass_mess(to_process_t *);
int write_cancel_ass_mess(to_process_t *);
int write_end_ass_mess(to_process_t *);

/* SS7 indication function. Indications are placed in the busy queue */
int begin_ind_mess(                          /* Begin Dialog indication */
		   unsigned long,
		   char *,
		   unsigned long,
		   char *,
		   unsigned long,
		   unsigned long,
		   unsigned char,
		   unsigned long *,
                   unsigned long,
		   char *,
		   unsigned long,
		   char *,
		   unsigned long,
                   unsigned long,
                   char *,
                   unsigned long,
                   char *,
                   unsigned long
		   );
int invoke_ind_mess(                          /* Invoke Dialog indication */
		    unsigned long,
		    long,
		    long,
		    char *,
                    unsigned long,
		    unsigned long,
		    char *,
		    unsigned long,
		    unsigned char,
		    unsigned long,
		    unsigned char, 
		    unsigned long
		    );
int continue_ind_mess (                       /* Continue Dialog indication */
		       unsigned long,
		       unsigned long,
		       unsigned char,
                       unsigned long,
		       unsigned long,
		       char *,
		       unsigned long,
		       char *,
		       unsigned long,
                       unsigned long,
                       char *,
                       unsigned long,
                       char *,
                       unsigned long
		       );
int abort_ind_mess (                          /* Abort Dialog indication */
		    unsigned long,
		    unsigned long,
		    char *,
		    unsigned long,
		    unsigned long,
                    unsigned long,
		    unsigned long,
		    char *,
		    unsigned long,
		    char *,
		    unsigned long,
                    unsigned long,
                    char *,
                    unsigned long,
                    char *,
                    unsigned long
		    );
int error_ind_mess (                          /* Error Component indication */
		    unsigned long,
		    long,
		    char *,
		    unsigned long,
                    unsigned long,
		    char *,
		    unsigned long,
		    unsigned long,
		    unsigned char,
		    unsigned long
		    );
int reject_ind_mess (                         /* Reject Component indication */
		     unsigned long,
		     long,
		     unsigned long,
		     unsigned long,
		     unsigned long,
                     unsigned long,
		     char *,
		     unsigned long,
		     unsigned char,
		     unsigned long
		     );
int result_ind_mess(                          /* Result Component indication */  
		    unsigned long,
		    long,
		    char *,
		    unsigned long,
                    unsigned long,
		    char *,
		    unsigned long,
		    unsigned char,
		    unsigned char,
		    unsigned long
		    );
int cancel_ind_mess (                        /* Cancel Component indication */
		     unsigned long,
		     long,
		     unsigned long
		     );
int end_ind_mess (                           /* End Dialog indication */
		  unsigned long,
		  unsigned long,
		  unsigned char,
		  unsigned long,
                  unsigned long,
		  char *,
		  unsigned long,
		  char *,
		  unsigned long,
                  unsigned long,
                  char *,
                  unsigned long,
                  char *,
                  unsigned long
		  );


/* Write on stdout parameters of DECss7 requests */
int write_allocate_dialogue_mess( unsigned long, unsigned long);
int write_invoke_component_mess (
				 unsigned long,
				 unsigned long,
				 long,
				 long,
				 char *,
				 unsigned long,
				 char *,
				 unsigned long,
				 unsigned long,
				 unsigned char,
				 unsigned long
				 );
int write_begin_dialogue_mess(                
			      unsigned long,
			      char *,
			      unsigned long,
			      char *,
			      unsigned long,
			      unsigned long
			      );
int write_end_dialogue_mess ( unsigned long,
			      unsigned long,
			      unsigned long
			     );
int write_result_component_mess(
				unsigned long,
				long,
				char *,
				unsigned long,
				char *,
				unsigned long,
				unsigned char
				);

/* Queue management functions */
int init_queue_free();
int rem_from_free( struct queue **, to_process_t **);
int put_at_free( struct queue *);
int init_queue_busy();
int rem_from_busy( struct queue **, to_process_t **);
int put_at_busy( struct queue *);
    

/*   
**++
** "get_message"
**
**  FUNCTIONAL DESCRIPTION:
**
**      Gets a message text from the system message file.
**
**  FORMAL PARAMETERS:
**
**	unsigned int	message_code
**
**  RETURN VALUE:
**
**      char *          message text
**
**  SIDE EFFECTS:
**
**      None
**--
*/

char *get_message (unsigned int message_code)
{
#ifdef VMS
int     ss_status;
short   message_length;
char    message_text [133];
$DESCRIPTOR (message_text_desc, message_text);


    ss_status = sys$getmsg (message_code, &message_length, &message_text_desc,
                            15,
                            0);

    if ((ss_status & 1) == 1)
        message_text [message_length] = '\0';
    else
        {
        sprintf (message_text, "%u", message_code);
        printf (message_text, "%u", message_code);
        }

    sprintf (message_text,"%u", message_code);
    return message_text;
#else
  static unsigned int error_len = 0 ;
  static char error[128] ;

  ss7_get_message( message_code, error, sizeof(error), &error_len);

  error[error_len] = '\0' ;

  return error ;
#endif /* VMS */
}



/*
**++
**  FUNCTIONAL DESCRIPTION:
**
**      The following routines are called to write
**	a monitoring message on stdout.
**	Each of them is associated to a queue element which hold 
**      the DECss7 indication parameters.
**
**  FORMAL PARAMETERS:
**
**      to_process_t *
**
**  RETURN VALUE:
**
**      SS7_NORMAL
**
**  SIDE EFFECTS:
**
**
**--
*/
int write_begin_ass_mess(to_process_t *to_process)
{
   printf ("\nReceived : Begin_indication, dialog_id =%d\n",(*to_process).val.begin.dialogue_id);
   return SS7_NORMAL;
}

int write_invoke_ass_mess(to_process_t *to_process)
{
    printf ("Received : Invoke_Indication, dialog_id = %d\n",(*to_process).val.indic.dialogue_id);
   return SS7_NORMAL;
}

int write_continue_ass_mess(to_process_t *to_process)
{
    printf ("Received : continue_indication, dialog_id = %d\n",(*to_process).val.cont.dialogue_id);
   return SS7_NORMAL;
}

int write_abort_ass_mess(to_process_t *to_process)
{
   printf ("Received : abort_indication, dialog_id = %d\n",(*to_process).val.abrt.dialogue_id);
   return SS7_NORMAL;
}

int write_error_ass_mess(to_process_t *to_process)
{
   printf ("Received : error_indication, dialog_id = %d\n",(*to_process).val.error.dialogue_id); 
   return SS7_NORMAL;
}

int write_reject_ass_mess(to_process_t *to_process)
{
   printf ("Received : reject_indication, dialog_id = %d\n",(*to_process).val.reject.dialogue_id); 
   return SS7_NORMAL;
}

int write_result_ass_mess(to_process_t *to_process)
{
   printf ("Received : result_indication, dialog_id = %d\n",(*to_process).val.result.dialogue_id);
   return SS7_NORMAL;
}

int write_cancel_ass_mess(to_process_t *to_process)
{
   printf ("Received : cancel_indication, dialog_id = %d\n",(*to_process).val.cancel.dialogue_id);
   return SS7_NORMAL;
}
    
int write_end_ass_mess(to_process_t *to_process)
{
   printf ("Received : end_indication, dialog_id = %d\n",(*to_process).val.end.dialogue_id);
   return SS7_NORMAL;
}


/*
**++
**  FUNCTIONAL DESCRIPTION:
**
**      The following routines are called by the DECss7 API when an
**      indication has to be delivered to the application.
**	They store all indication parameters in a control block, and
**      enqueue this Control Block in the "to be processed" queue.
**
**  FORMAL PARAMETERS:
**
**      The parameters exactly match those required for ss7 indication
**	routines. Most of the routines are declared to ss7 as indication
**	routines.
**
**  RETURN VALUE:
**
**      SS7_NORMAL
**      Q_WAS_EMPTY         "to be processed" queue is empty.
**
**  SIDE EFFECTS:
**
**
**--
*/
int begin_ind_mess(   
		   unsigned long dialogue_id,
		   char *dest_addr,
		   unsigned long dest_addr_len,
		   char *orig_addr,
		   unsigned long orig_addr_len,
		   unsigned long quality_of_service,
		   unsigned char components_present,
		   unsigned long *user_info,
                   unsigned long object_or_integer_app_con_name,
		   char *application_context_name,
		   unsigned long application_context_name_length,
		   char *application_context_info,
		   unsigned long application_context_info_length,
                   unsigned long object_or_integer_secu_con,
                   char *security_context,
                   unsigned long security_context_length,
                   char *confidentiality,
                   unsigned long confidentiality_length
		   )
{
  to_process_t *to_process = NULL;
  struct queue  *queue_elem;
  int status;

  status = rem_from_free( &queue_elem, &to_process);        /* get a free buffer */ 
  if (status != Q_SUCCESS) {
    fprintf(stderr,"free queue empty\n");
    return (Q_WAS_EMPTY);
  }

 (*to_process).ind_type = WRITE_BEGIN_C;

 (*to_process).val.begin.dialogue_id = dialogue_id; 
 (*to_process).val.begin.dest_len = dest_addr_len;
 memcpy((*to_process).val.begin.dest_data, dest_addr, dest_addr_len);
 (*to_process).val.begin.orig_len = orig_addr_len;
 memcpy((*to_process).val.begin.orig_data, orig_addr, orig_addr_len);
 (*to_process).val.begin.quality_of_service = quality_of_service;
 (*to_process).val.begin.components_present = components_present;
  *user_info = dialogue_id;
  (*to_process).val.begin.user_info = *user_info;
                                 /* Now chain buffer  in busy queue */
  put_at_busy( queue_elem);                           
  return SS7_NORMAL;
}

int invoke_ind_mess(  
		    unsigned long dialogue_id,
		    long component_id,
		    long linked_id,
		    char *operation,
		    unsigned long operation_len,
                    unsigned long set_or_sequence,
		    char *param,
		    unsigned long param_len,
		    unsigned char last,
		    unsigned long identifier,
		    unsigned char last_component, 
		    unsigned long user_info)
{
  to_process_t *to_process = NULL;
  struct queue  *queue_elem;
  int status;

  status = rem_from_free( &queue_elem, &to_process);        /* get a free buffer */ 
  if (status != Q_SUCCESS) {
    fprintf(stderr,"free queue empty\n");
    return (Q_WAS_EMPTY);
  }

 (*to_process).ind_type = WRITE_INVOKE_C;

 (*to_process).val.indic.dialogue_id = dialogue_id;
 (*to_process).val.indic.component_id = component_id;
 (*to_process).val.indic.linked_id = linked_id;
 (*to_process).val.indic.ope_len = operation_len;
 memcpy((*to_process).val.indic.ope_data, operation, operation_len);
 (*to_process).val.indic.parm_len = param_len;
 memcpy((*to_process).val.indic.parm_data, param, param_len);
 (*to_process).val.indic.last = last;
 (*to_process).val.indic.identifier = identifier;
 (*to_process).val.indic.last_component = last_component;
 (*to_process).val.indic.user_info = user_info;
                                 /* Now chain buffer  in busy queue */
  put_at_busy( queue_elem);                           

  return SS7_NORMAL;
}

int continue_ind_mess (
		       unsigned long dialogue_id,
		       unsigned long quality_of_service,
		       unsigned char component_present,
		       unsigned long user_info,
                       unsigned long object_or_integer_app_con_name,
		       char *application_context_name,
		       unsigned long application_context_name_length,
		       char *application_context_info,
		       unsigned long application_context_info_length,
                       unsigned long object_or_integer_secu_con,
                       char *security_context,
                       unsigned long security_context_length,
                       char *confidentiality,
                       unsigned long confidentiality_length
		       )
{
  to_process_t *to_process = NULL;
  struct queue  *queue_elem;
  int status;

  status = rem_from_free( &queue_elem, &to_process);        /* get a free buffer */ 
  if (status != Q_SUCCESS) {
    fprintf(stderr,"free queue empty\n");
    return (Q_WAS_EMPTY);
  }

 (*to_process).ind_type = WRITE_CONTINUE_C;

 (*to_process).val.cont.dialogue_id = dialogue_id; 
 (*to_process).val.cont.quality_of_service = quality_of_service; 
 (*to_process).val.cont.components_present = component_present;
 (*to_process).val.cont.user_info = user_info;
                                 /* Now chain buffer  in busy queue */
  put_at_busy( queue_elem);                           
  return SS7_NORMAL;
}

int abort_ind_mess (
		    unsigned long dialogue_id,
		    unsigned long source,
		    char *reason,
		    unsigned long reason_len,
		    unsigned long quality_of_service,
		    unsigned long user_info,
                    unsigned long object_or_integer_app_con_name,
		    char *application_context_name,
		    unsigned long application_context_name_length,
		    char *application_context_info,
		    unsigned long application_context_info_length,
                    unsigned long object_or_integer_secu_con,
                    char *security_context,
                    unsigned long security_context_length,
                    char *confidentiality,
                    unsigned long confidentiality_length
		    )
{
  to_process_t *to_process = NULL;
  struct queue  *queue_elem;
  int status;

  status = rem_from_free( &queue_elem, &to_process);        /* get a free buffer */ 
  if (status != Q_SUCCESS) {
    fprintf(stderr,"free queue empty\n");
    return (Q_WAS_EMPTY);
  }

 (*to_process).ind_type = WRITE_ABORT_C;

 (*to_process).val.abrt.dialogue_id = dialogue_id;
 (*to_process).val.abrt.source = source;
 (*to_process).val.abrt.reason_len = reason_len;
 memcpy((*to_process).val.abrt.reason_data, reason, reason_len);
 (*to_process).val.abrt.quality_of_service = quality_of_service;
 (*to_process).val.abrt.user_info = user_info;
                                 /* Now chain buffer  in busy queue */
  put_at_busy( queue_elem);                           
  return SS7_NORMAL;
}

int error_ind_mess (  
		    unsigned long dialogue_id,
		    long component_id,
		    char *error,
		    unsigned long error_len,
                    unsigned long set_or_sequence,
		    char *param,
		    unsigned long param_len,
		    unsigned long identifier,
		    unsigned char last_component,
		    unsigned long user_info
		    )
{
  to_process_t *to_process = NULL;
  struct queue  *queue_elem;
  int status;

  status = rem_from_free( &queue_elem, &to_process);        /* get a free buffer */ 
  if (status != Q_SUCCESS) {
    fprintf(stderr,"free queue empty\n");
    return (Q_WAS_EMPTY);
  }

 (*to_process).ind_type = WRITE_ERROR_C;

 (*to_process).val.error.dialogue_id = dialogue_id;
 (*to_process).val.error.component_id = component_id;
 (*to_process).val.error.error_len = error_len;
 memcpy((*to_process).val.error.error_data, error, error_len);
 (*to_process).val.error.parm_len = param_len;
 memcpy((*to_process).val.error.parm_data, param, param_len);
 (*to_process).val.error.identifier = identifier;
 (*to_process).val.error.last_component = last_component;
 (*to_process).val.error.user_info = user_info;
                                 /* Now chain buffer  in busy queue */
  put_at_busy( queue_elem);                           
  return SS7_NORMAL;
}

int reject_ind_mess ( 
		     unsigned long dialogue_id,
		     long component_id,
		     unsigned long source,
		     unsigned long type,
		     unsigned long problem,
                     unsigned long set_or_sequence,
		     char *param,
		     unsigned long param_len,
		     unsigned char last_component,
		     unsigned long user_info
		     )
{
  to_process_t *to_process = NULL;
  struct queue  *queue_elem;
  int status;

  status = rem_from_free( &queue_elem, &to_process);        /* get a free buffer */ 
  if (status != Q_SUCCESS) {
    fprintf(stderr,"free queue empty\n");
    return (Q_WAS_EMPTY);
  }

 (*to_process).ind_type = WRITE_REJECT_C;

 (*to_process).val.reject.dialogue_id = dialogue_id;
 (*to_process).val.reject.component_id = component_id;
 (*to_process).val.reject.source = source;
 (*to_process).val.reject.type = type;
 (*to_process).val.reject.problem = problem;
 (*to_process).val.reject.parm_len = param_len;
 memcpy((*to_process).val.reject.parm_data, param, param_len);
 (*to_process).val.reject.last_component = last_component;
 (*to_process).val.reject.user_info = user_info;
                                 /* Now chain buffer  in busy queue */
  put_at_busy( queue_elem);                           
  return SS7_NORMAL;
}

int result_ind_mess(  
		    unsigned long dialogue_id,
		    long component_id,
		    char *operation,
		    unsigned long operation_len,
                    unsigned long set_or_sequence,
		    char *param,
		    unsigned long param_len,
		    unsigned char last,
		    unsigned char last_component,
		    unsigned long user_info
		    )
{
  to_process_t *to_process = NULL;
  struct queue  *queue_elem;
  int status;

  status = rem_from_free( &queue_elem, &to_process);        /* get a free buffer */ 
  if (status != Q_SUCCESS) {
    fprintf(stderr,"free queue empty\n");
    return (Q_WAS_EMPTY);
  }

 (*to_process).ind_type = WRITE_RESULT_C;

 (*to_process).val.result.dialogue_id = dialogue_id; 
 (*to_process).val.result.component_id = component_id;
 (*to_process).val.result.parm_len = param_len;
 memcpy((*to_process).val.result.parm_data, param, param_len);
 (*to_process).val.result.ope_len = operation_len;
 memcpy((*to_process).val.result.ope_data, operation, operation_len);
 (*to_process).val.result.last = last; 
 (*to_process).val.result.last_component = last_component; 
 (*to_process).val.result.user_info = user_info; 
                                 /* Now chain buffer  in busy queue */
  put_at_busy( queue_elem);                           
  return SS7_NORMAL;
}

int cancel_ind_mess ( 
		     unsigned long dialogue_id,
		     long component_id,
		     unsigned long user_info
		     )
{
  to_process_t *to_process = NULL;
  struct queue  *queue_elem;
  int status;

  status = rem_from_free( &queue_elem, &to_process);        /* get a free buffer */ 
  if (status != Q_SUCCESS) {
    fprintf(stderr,"free queue empty\n");
    return (Q_WAS_EMPTY);
  }

 (*to_process).ind_type = WRITE_CANCEL_C;

 (*to_process).val.cancel.dialogue_id = dialogue_id;
 (*to_process).val.cancel.component_id = component_id;;
 (*to_process).val.cancel.user_info = user_info;
                                 /* Now chain buffer  in busy queue */
  put_at_busy( queue_elem);                           
  return SS7_NORMAL;
}
   
int end_ind_mess (
		  unsigned long dialogue_id,
		  unsigned long quality_of_service,
		  unsigned char components_present,
		  unsigned long user_info,
                  unsigned long object_or_integer_app_con_name,
		  char *application_context_name,
		  unsigned long application_context_name_length,
		  char *application_context_info,
		  unsigned long application_context_info_length,
                  unsigned long object_or_integer_secu_con,
                  char *security_context,
                  unsigned long security_context_length,
                  char *confidentiality,
                  unsigned long confidentiality_length
		  )
{
  to_process_t *to_process = NULL;
  struct queue  *queue_elem;
  int status;

  status = rem_from_free( &queue_elem, &to_process);        /* get a free buffer */ 
  if (status != Q_SUCCESS) {
    fprintf(stderr,"free queue empty\n");
    return (Q_WAS_EMPTY);
  }

 (*to_process).ind_type = WRITE_END_C;

 (*to_process).val.end.dialogue_id = dialogue_id;
 (*to_process).val.end.quality_of_service = quality_of_service;
 (*to_process).val.end.components_present = components_present;
 (*to_process).val.end.user_info = user_info;
                                  /* Now chain buffer  in busy queue */
  put_at_busy( queue_elem);                           
  return SS7_NORMAL;
}


/*
**++
**  FUNCTIONAL DESCRIPTION:
**
**      These routines write a message on sdtout to
**	monitor calls to ss7.
**
**  FORMAL PARAMETERS:
**
**      These exactly match the corresponding ss7 call.
**
**  RETURN VALUE:
**
**      SS7_NORMAL.
**
**  SIDE EFFECTS:
**
**      The message is formatted in message_buffer.
**
**  DESIGN:
**
**      None
**--
*/ 
int write_allocate_dialogue_mess ( 
				  unsigned long dialogue_id,
				  unsigned long user_info
				  )
{
    printf("Sent : Allocate_Dialoge_Id, dialog_id= %d\n", dialogue_id );
    return SS7_NORMAL;
}

int write_invoke_component_mess (
				 unsigned long dialogue_id,
				 unsigned long component_class,
				 long component_id,
				 long linked_id,
				 char *operation,
				 unsigned long operation_len,
				 char *param,
				 unsigned long param_len,
				 unsigned long timeout,
				 unsigned char last,
				 unsigned long identifier
				)
{
   printf("Sent : Invoke_Component, dialog_id = %d\n", dialogue_id);
   return SS7_NORMAL;
}

int write_begin_dialogue_mess(
			      unsigned long dialogue_id,
			      char *dest_addr,
			      unsigned long dest_addr_len,
			      char *orig_addr,
			      unsigned long orig_addr_len,
			      unsigned long quality_of_service
			      )
{
   printf("Sent : Begin_Dialogue, dialog_id = %d\n", dialogue_id);
   return SS7_NORMAL;
}

int write_end_dialogue_mess(
			    unsigned long dialogue_id,
			    unsigned long termination,
			    unsigned long quality_of_service
			    )
{
   printf("Sent : End_Dialogue, dialog_id = %d\n", dialogue_id);
   return SS7_NORMAL;
}

int write_result_component_mess(
				unsigned long dialogue_id,
				long component_id,
				char *operation,
				unsigned long operation_len,
				char *param,
				unsigned long param_len,
				unsigned char last
				)
{
   printf("Sent : Result_Component, dialog_id = %d\n", dialogue_id);
   return SS7_NORMAL;
}


/*
**++
**  FUNCTIONAL DESCRIPTION:
**
**      Basic routines of queues (free and busy) management (DIGITAL UNIX only).
**
**--
*/
#ifdef __unix__
#define LOCK_FREE \
    if (pthread_mutex_lock( &queue_free_mutex) < 0) { \
      fprintf(stderr,"Cannot lock queue_free mutex. Exiting.\n"); \
      exit(-1); \
   }

#define UNLOCK_FREE \
    if (pthread_mutex_unlock( &queue_free_mutex) < 0) { \
      fprintf(stderr,"Cannot unlock queue_free mutex. Exiting.\n"); \
      exit(-1); \
   }


#define LOCK_BUSY \
    if (pthread_mutex_lock( &queue_busy_mutex) < 0) { \
      fprintf(stderr,"Cannot lock queue_busy mutex. Exiting.\n"); \
      exit(-1); \
   }

#define UNLOCK_BUSY \
    if (pthread_mutex_unlock( &queue_busy_mutex) < 0) { \
      fprintf(stderr,"Cannot unlock queue_busy mutex. Exiting.\n"); \
      exit(-1); \
   }
#elif WIN32 
#define LOCK_FREE \
    if (WaitForSingleObject(queue_free_mutex, INFINITE) == WAIT_FAILED) { \
      fprintf(stderr,"Cannot lock queue_free mutex. Exiting.\n"); \
      exit(-1); \
   }

#define UNLOCK_FREE \
    if (ReleaseMutex( queue_free_mutex) == FALSE) { \
      fprintf(stderr,"Cannot unlock queue_free mutex. Exiting.\n"); \
      exit(-1); \
   }
#define LOCK_BUSY \
    if (WaitForSingleObject(queue_busy_mutex, INFINITE) == WAIT_FAILED) { \
      fprintf(stderr,"Cannot lock queue_busy mutex. Exiting.\n"); \
      exit(-1); \
   }

#define UNLOCK_BUSY \
    if (ReleaseMutex( queue_busy_mutex) == FALSE) { \
      fprintf(stderr,"Cannot unlock queue_busy mutex. Exiting.\n"); \
      exit(-1); \
   }
#endif

int init_queue_free()
{
   struct queue *tmp;
   int i;
   
   for (i=0; i<MAX_MSG; i++) {
      tmp = (struct queue *)malloc(sizeof(struct queue));
      tmp->flink = 0;
      tmp->blink = 0;
      tmp->object = &to_process_array[i];
       _INSQUE( tmp,  queue_free.blink);
   }
#ifdef __unix__
   if (pthread_mutex_init( &queue_free_mutex, pthread_mutexattr_default) < 0) {
      fprintf(stderr,"Cannot create queue_free mutex. Exiting.\n"); 
      exit(-1); 
   }
#elif WIN32
   if ( (queue_free_mutex = CreateMutex( NULL, FALSE, NULL ) ) == NULL ) {
      fprintf(stderr,"Cannot create queue_free mutex. Exiting.\n"); 
      exit(-1); 
   }
#endif 
   return Q_SUCCESS;
}

int rem_from_free(struct queue **queue_elem,
                  to_process_t **to_process)
{
  int status;
    struct linkage {
    struct linkage *fl;
    struct linkage *bl;
    } *xxtmp;

#ifdef VMS
  xxtmp = (struct linkage *)(queue_free.flink);
  (xxtmp->bl)->fl = xxtmp->fl;
  (xxtmp->fl)->bl = xxtmp->bl;
  *(queue_elem) = (void *) xxtmp;
  if ((*queue_elem)->object == NULL) status = Q_WAS_EMPTY; else  status = Q_SUCCESS;

  switch(status)
   {
    case Q_WAS_EMPTY:
    break;

    case Q_SUCCESS :
    case Q_EMPTY:
    (*to_process) = (to_process_t *)(*queue_elem)->object;
    status = Q_SUCCESS;
  }
#else
  LOCK_FREE ;
  
  xxtmp = (struct linkage *)(queue_free.flink);
  (xxtmp->bl)->fl = xxtmp->fl;
  (xxtmp->fl)->bl = xxtmp->bl;
  *(queue_elem) = (void *) xxtmp;
  if ((*queue_elem)->object == NULL) status = Q_WAS_EMPTY; else  status = Q_SUCCESS;

  switch(status) 
   {
    case Q_WAS_EMPTY:
    break;

    case Q_SUCCESS :
    case Q_EMPTY:
    (*to_process) = (to_process_t *)(*queue_elem)->object;
    status = Q_SUCCESS;
  }

  UNLOCK_FREE ;
#endif /* VMS */
 
 return(status);
}

int put_at_free( struct queue *queue_elem)
{
#ifndef VMS
  LOCK_FREE ;
#endif /* VMS */
  _INSQUE( queue_elem, queue_free.blink);
#ifndef VMS
  UNLOCK_FREE ;
#endif /* VMS */
   return Q_SUCCESS;
}

int init_queue_busy()
{
#ifdef __unix__
   if (pthread_mutex_init( &queue_busy_mutex, pthread_mutexattr_default) < 0) {
      fprintf(stderr,"Cannot create queue_busy mutex. Exiting.\n"); 
      exit(-1); 
   }
#ifdef QUEUE_BUSY_WAIT
   if (pthread_cond_init( &queue_busy_cond, pthread_condattr_default) < 0) {
      fprintf(stderr,"Cannot create queue_busy cond var. Exiting.\n"); 
      exit(-1); 
   }
#endif /* QUEUE_BUSY_WAIT */
#elif WIN32
  if ( (queue_busy_mutex = CreateMutex( NULL, FALSE, NULL ) ) == NULL ) {
      fprintf(stderr,"Cannot create queue_busy mutex. Exiting.\n"); 
      exit(-1); 
   }
#ifdef QUEUE_BUSY_WAIT
   if ( (queue_busy_cond = CreateEvent( NULL, FALSE, FALSE, NULL ) ) == NULL ) {
      fprintf(stderr,"Cannot create queue_busy event. Exiting.\n"); 
      exit(-1); 
   }
#endif /* QUEUE_BUSY_WAIT */
#endif /* __unix__ */

   return Q_SUCCESS;
}

int rem_from_busy(struct queue **queue_elem,
                  to_process_t **to_process)
{
  int status;
    struct linkage {
    struct linkage *fl;
    struct linkage *bl;
    } *xxtmp;

#ifndef VMS 
  LOCK_BUSY ;

#ifdef QUEUE_BUSY_WAIT
  while ( queue_busy.flink->object == NULL ) {
#ifdef __unix__
     if ( pthread_cond_wait( &queue_busy_cond, &queue_busy_mutex) < 0) {
	fprintf(stderr,"Cannot wait on queue_busy cond var. Exiting.\n"); 
	exit(-1); 
     }
#elif WIN32
// Potential bug here ! Atomicity is not guaranteed
// pthread_cond_wait semantic: (from Unix man page)
// "This routine atomically releases the mutex and causes the calling thread to
// wait on the condition.  The atomicity is important, because it means the
// thread cannot miss a wakeup while the mutex is unlocked.  When the wait is
// satisfied as a result of some thread calling pthread_cond_signal or
// pthread_cond_broadcast, the mutex is reacquired before returning to the
// caller."
	UNLOCK_BUSY;
    if ( WaitForSingleObject( queue_busy_cond, INFINITE ) == WAIT_FAILED ) {
	fprintf(stderr,"Cannot wait on queue_busy event. Exiting.\n"); 
	exit(-1); 
     }
	LOCK_BUSY;
#endif

  }
#endif /* QUEUE_BUSY_WAIT */
#endif /* VMS */

  xxtmp = (struct linkage *)(queue_busy.flink);
  (xxtmp->bl)->fl = xxtmp->fl;
  (xxtmp->fl)->bl = xxtmp->bl;
  *(queue_elem) = (void *) xxtmp;
  if ((*queue_elem)->object == NULL) status = Q_WAS_EMPTY; else  status = Q_SUCCESS;

  switch(status) 
   {
    case Q_WAS_EMPTY:
    break;

    case Q_SUCCESS:
    case Q_EMPTY:
    (*to_process) = (to_process_t *)(*queue_elem)->object;
    status = Q_SUCCESS;
  }
#ifndef VMS
  UNLOCK_BUSY ;
#endif

 return(status);
}

int put_at_busy( struct queue *queue_elem)
{
#ifdef VMS
   _INSQUE( queue_elem , queue_busy.blink );
#else
#ifdef QUEUE_BUSY_WAIT
   char queue_was_empty = Q_SUCCESS;
#endif /* QUEUE_BUSY_WAIT */

   LOCK_BUSY ;

#ifdef QUEUE_BUSY_WAIT
   if (queue_busy.blink->object == NULL) queue_was_empty = Q_WAS_EMPTY;
#endif /* QUEUE_BUSY_WAIT */
     
  _INSQUE( queue_elem, queue_busy.blink);
  
#ifdef QUEUE_BUSY_WAIT
#ifdef WIN32
  if (1) {
#else
  if (queue_was_empty == Q_WAS_EMPTY) {
#endif     
#ifdef WIN32
     if (PulseEvent( queue_busy_cond ) == FALSE) { 
	fprintf(stderr,"Cannot signal queue_busy event. Exiting.\n"); 
	exit(-1); 
     }
#else
  if (pthread_cond_broadcast( &queue_busy_cond) < 0) { 
	fprintf(stderr,"Cannot broadcast queue_busy cond var. Exiting.\n"); 
	exit(-1); 
     }
#endif /* WIN32 */
  }
#endif /* QUEUE_BUSY_WAIT */

  UNLOCK_BUSY ;
#endif /* VMS */
  return Q_SUCCESS; 
}
