#ifndef lint
    static char *sccsid = "@(#)osif_api_exam.c	V3.2+  26-SEP-1995";
#endif

/*
**
**   FTAM API Example code, for initiator side.
**   
**   To use this code on your system, you may have to set the local and remote
**   addresses correctly (see the macro definitions below).
**
*****
**   
**   Copyright (c) Digital Equipment Corporation 1995. All rights reserved.
**
**   Restricted Rights: Use, duplication, or disclosure by the U.S.
**   Government is subject to restrictions as set forth in subparagraph
**   (c) (1) (ii) of DFARS 252.227-7013, or in FAR 52.227-19, or in FAR
**   52.227-14 Alt. III, as applicable.
**
**   This software is proprietary to and embodies the confidential
**   technology of Digital Equipment Corporation. Possession, use, or
**   copying of this software and media is authorized only pursuant to a
**   valid written license from Digital or an authorized sublicensor.
**
*/


/*
**
**  INCLUDE FILES
**
*/
#include <stdio.h>              /* Needed for EOF constant */
#include <stdlib.h>		/* Needed for EOF constant */
#include <string.h>
#include <osif.h>		/* FTAM API */

/*
**
** TABLE OF CONTENTS
**
*/

extern int diagnostic_checking ();
extern int print_error_message ();
extern int print_pbname ();
extern int send_change_attributes ();
extern int send_close ();
extern int send_create ();
extern int send_data ();
extern int send_data_end ();
extern int send_delete ();
extern int send_initialize ();
extern int send_read_attributes();
extern int send_transfer_end ();
extern int send_write ();

/*
**
**  MACRO DEFINITIONS
**
*/

/*
** ***** The following are system specific and may need to be changed *****
**
**  The following macros are for the local and remote presentation addresses.
**
**	LOCAL_P_ADDRESS	    presentation selector, session selector, and
**			    transport selector (psel.ssel.tsel.)
**	LOCAL_NSAP	    local NSAP
**	LOCAL_TEMPLATE	    local transport template
**	LOCAL_PROVIDER	    local transport provider
**
**	REMOTE_P_ADDRESS    presentation selector, session selector, and
**			    transport selector (psel.ssel.tsel.)
**	REMOTE_NSAP	    remote NSAP
**	REMOTE_TEMPLATE	    remote transport template
**	REMOTE_PROVIDER	    remote transport provider
**
**  For OpenVMS systems, if you are running the FTAM API example initiator and
**  responder on the same system, no modifications are required to the
**  definitions.  If you wish to run the initiator and responder on different
**  systems, you must set the REMOTE_NSAP macro to be the NSAP of the system
**  running the responder, and set REMOTE_TEMPLATE to a non-loop template
**  (such as "default").
**
**  For Digital UNIX systems, regardless of whether the initiator and responder
**  are running on the same system, you must set the REMOTE_NSAP macro to be
**  the NSAP of the system running the responder.
**
**  If you wish, you may change any of the address information for the local or
**  remote systems (e.g. to use RFC1006 instead of CLNS, to use a different
**  template, or to use a different psel.ssel.tsel.).
*/

#if defined (VMS)
#define LOCAL_P_ADDRESS  "RMS.FTAM.OSIF."
#define LOCAL_NSAP       ""
#define LOCAL_TEMPLATE   "osit$loop_clns"
#define LOCAL_PROVIDER   "osi"

#define REMOTE_P_ADDRESS  "APIPPM.APISPM.APITPM."
#define REMOTE_NSAP       "%x21"
#define REMOTE_TEMPLATE   "osit$loop_clns"
#define REMOTE_PROVIDER   "osi"

#define FTAMOIDS	  "sys$library:osif$oids.txt"

#else
#define LOCAL_P_ADDRESS  "%x0001.%x0001.%x0001."
#define LOCAL_NSAP       ""
#define LOCAL_TEMPLATE   "default"
#define LOCAL_PROVIDER   "osi"

#define REMOTE_P_ADDRESS  "apippm.apispm.apitpm."
#define REMOTE_NSAP       "%x414541871500410800ffffffffff."
#define REMOTE_TEMPLATE   "default"
#define REMOTE_PROVIDER   "osi"
#endif

/*
** These are constants to describe the size and number
** of buffers passed down to the interface
*/
#define  BUFFER_SIZE    (8*1024)
#define  BUFFER_COUNT   1
#define  STR_LENGTH     100

#define ZERO(Thing) {memset((void *)&Thing,0x00,sizeof(Thing));}

#define FTAM_MESSAGES_CAT   1
#define FTAM_ASN1_CAT	    2

#define CHECK_STATUS {			\
            if (status)			\
                {                       \
                msg_print_stack();      \
                exit (1);               \
                };}


/*
  This example program will act as an FTAM Initiator.
  It will perform the following actions:

  1. Establish an FTAM Association with a remote FTAM responder

  2. Create a file on the remote filestore

  3. Write file data to the remote file

  4. Close the remote file

  5. Read and Display attributes of the newly created file

  6. Rename the file that was just created.

  7. Delete the file that was just renamed

  8. End the FTAM Association


  The following schema describe the FTAM primitives that
  will be processed by this example.

  
  Establish an association with a FTAM responder
  
  Send:    
  f-initialize-request
  Receive: 
  f-initialize-response
  
  Open and create a file on the remote system
  
  Send:    
  f-begin-group-request
  f-create-request
  f-open-request
  f-end-group-request
  
  Receive: 
  f-begin-group-response
  f-create-response
  f-open-response
  f-end-group-response
  
  
  Write file data to the remote file
  
  Send:    	
  f-write-request
  Send:    	
  f-data-request, f-data-request . . . f-data-request
  
  Tell the FTAM responder we are through sending data
  
  Send:    
  f-data-end-request
  Send:    
  f-transfer-end-request

  Receive: 
  f-transfer-end-response
  
  
  Close the remote file

  Send:    
  f-begin-group-request
  f-close-request
  f-deselect-request
  f-end-group-request
  
  Receive:
  f-begin-group-response
  f-close-response
  f-deselect-response
  f-end-group-response
  
  Read and display the attributes of the file we just
  created.
  
  Send:
  f-begin-group-request
  f-select-request
  f-read-attributes-request
  f-deselect-request
  f-end-group-request

  Receive:    
  f-begin-group-response
  f-select-response
  f-read-attributes-response
  f-deselect-request
  f-end-group-response
  
  Rename the file we just created.

  Send:    
  f-begin-group-request
  f-select-request
  f-change-attributes-request
  f-deselect-request
  f-end-group-request

  Receive:    
  f-begin-group-response
  f-select-response
  f-change-attributes-response
  f-deselect-response
  f-end-group-response
  
  Delete the file we just renamed
  
  Send: 
  f-begin-group-request
  f-select-request
  f-delete-request
  f-end-group-request

  Receive: 
  f-begin-group-response
  f-select-response
  f-delete-response
  f-end-group-response
  
  End the FTAM Association

  Send:    
  f-terminate-request

  Receive: 
  f-terminate-response
  
*/


main(argc, argv)
int     argc;
char    **argv;

{  
    extern int state_logging;
    unsigned  status;          /* Checks for the status of the function */
    unsigned  error_status;    /* Contains the error if the function failed */
    unsigned  port_id;			    /* FTAM association identifier */
    struct osif_buffer_list *buffer_list;   /* Variable for buffer allocation */
    struct osif_buffer_list *buf_entry;     /* Variable for buffer allocation */
    struct osif_buffer_list *tmp_buf_entry;
    int    i;                               /* Variable for loop control */
    int	   c;

    char *init_id   = "username";          /* initiator id  */
    char *fs_passwd = "password";          /* filestore password */

    printf("FTAM API example initiator starting\n");
    
    /*
     * Call the FTAM API to get an identifier to be used for
     * subsequent FTAM operations. This call does not cause
     * any FTAM protocol actions to occur.
     */
    status = osif_assign_port( &port_id,
			       (struct osifpb *)0, 
			       OSIF_ASSIGN_INITIATOR, 
			       &error_status );
    CHECK_STATUS;

    /*
     * Dynamically allocate buffers and buffer descriptors for received
     * data and protocol information.  Pass the buffers down to the interface.
     * These buffers will be used to receive transport events. The
     * osif_get_event call may return a pointer to one of these buffers
     * as the result of processing an FTAM event.
     */
    for ( i=0; i<BUFFER_COUNT; i++ )
    {
	buffer_list = 
	(struct osif_buffer_list *) malloc(sizeof(struct osif_buffer_list ));
	
	if (!buffer_list)
	    {
	    printf("ran out of memory; i = %d\n", i);      
	    exit(0);
	    }
	
	buffer_list->next = 0;
	buffer_list->buffer_length = BUFFER_SIZE;
	buffer_list->bufferptr = (char *) malloc( BUFFER_SIZE );
	
	status = osif_give_buffer( port_id, buffer_list, &error_status );
	if (status)
	    print_error_message("osif_give_buffer: ", error_status);
	    
    }

    /*
     * Send the f-initialize request to the peer, and wait for the
     * f-initialize-response.  This exchange establishs the 'connection'
     * for subsequent use, and describes (in general terms) the type
     * of operations to be performed over the life of the association.
     */
    
    send_initialize( port_id, init_id, fs_passwd );
    
    /*
     * Now create and open a remote file for writing.  This will
     * require sending a grouped sequence of PDUs.   First a grouped
     * sequence of requests is sent, and then a grouped sequence
     * of responses is received.
     */
    send_create( port_id );
    
    /*
     * We now have a remote file open.  Send a f-write-request
     * to the peer to enter the data-transfer regime.  We then
     * send the file data to fill the file.  A f-data-end request
     * sent to  the peer signals that we have completed sending data.
     * These requests are unconfirmed, so we do not expect responses.
     */
    send_write( port_id );
    
    send_data( port_id );
    
    send_data_end( port_id );
    
    
    
    
    /*
     * We now send a f-transfer-end-request to exit the data transfer
     * regime.  This is a confirmed service, so we will wait for
     * the response. 
     */
    send_transfer_end( port_id );
    
    /*
     * We now send a f-begin-group, f-close, f-deselect, and f-end-group.
     * These are all confirmed services, so we wait for the responses.
     */
    
    send_close (port_id );
    
    /*
     * Now We select a file for reading.  This requires sending
     * a grouped sequence of PDU's.  First a grouped sequence of
     * f-begin-group request, f-select request, f-read-attribute request, 
     * f-end-group-request is sent; then a grouped sequence of 
     * f-begin-group response, f-select response, f-read-attribute 
     * response, and f-end-group-response is received.
     */
    
    send_read_attributes( port_id );
    
    /*
     * Now we select the file created and change its name.
     * This requires sending a grouped sequence of pdus.  First
     * a sequence of f-begin-group request, f-select request,
     * f-change-attributes, f-deselect-request and a f-end-group-
     * request is sent; then a sequence of f-begin-group response,
     * f-select-response, f-change-attributes-response, f-deselect-
     * response, and f-end-group-response is received.
     */
    send_change_attributes (port_id );
    
    /*
     * We are now ready to delete the file created.  We send a
     * grouped sequence of f-begin-group-request, f-select-request,
     * f-delete-request, f-end-group-request; then a grouped
     * sequence of f-begin-group-response, f-select-response, f-
     * delete-response, and f-end-group response is received.
     */
    send_delete( port_id );
    
    
    /*
     * Send the f-terminate-request to the peer to 'tear-down' the
     * association.  This is a confirmed service, so we wait for a
     * response.
     */
    send_terminate( port_id );
    
    /*
     * Deassign the port.  The deassign call can return a list of buffers
     * previously passed down by the interface.  If this list is non-null,
     * we deallocate both the buffers and the buffer descriptors on the
     * list.
     */
    status = osif_deassign_port( port_id, &buffer_list, 
				 OSIF_ASSIGN_INITIATOR, &error_status );
    CHECK_STATUS;

    /*
     * Deallocate any buffers returned, as well as the associated
     * buffer descriptors.
     */
    if ( buffer_list )
    {
	for ( buf_entry = buffer_list; buf_entry; )
	{
	    free( buf_entry->bufferptr );    /* Free the buffer */
	    tmp_buf_entry = buf_entry;
	    buf_entry = buf_entry->next;     /* Step to next list entry */
	    free( tmp_buf_entry );           /* Free buffer descriptor */
	}
    }

    printf("FTAM API example initiator completed successfully\n");

}

send_initialize( port_id , init_id, fs_passwd)
/*
**++
**  FUNCTIONAL DESCRIPTION:
**
**      This routine sends a request to the FTAM API to construct
**      and send an FTAM F-INITIALIZE request PDU.  This routine
**      also waits for API to send the F-INITIALIZE response.  
**
**
**  FORMAL PARAMETERS:
**
**      port_id
**      init_id        initiator id
**      fs-passwd      file-store password
**
**  IMPLICIT INPUTS:
**
**      None
**
**  IMPLICIT OUTPUTS:
**
**      None
**
**  COMPLETION CODES:
**
**
**
**  SIDE EFFECTS:
**
**      Contents type list elements are allocated and stuck on the fileptr
**      structure
**
**--
*/
unsigned       port_id;
char *init_id;
char *fs_passwd;
{
    unsigned  status;                      /* Call completion status        */
    unsigned  error_status;                /* Additional status info value  */
    struct osifpb f_initialize_request;    /* Request parameter block       */
    struct osifpb f_initialize_response;   /* Parameter block for responses */
    struct osif_buffer_list *buffer_list; /* Variable for buffer allocation */
    struct osif_ctl  ctlblk_1;             /* Temp structures for building  */
    struct osif_ctl  ctlblk_2;             /* a contents type list          */
    struct osif_ctl  ctlblk_3;  
    int i;
    
    printf("initializing connection...\n");
    
    /*
     * Zero fill the parameter block
     */
    ZERO (f_initialize_request );
    ZERO (f_initialize_response);
    
    /*
     * Zero fill the contents type list blocks
     */
    ZERO (ctlblk_1);
    ZERO (ctlblk_2);
    ZERO (ctlblk_3);

    /*
     * Set the f-initialize function code in the parameter block
     */
    f_initialize_request.osif_block_type = OSIF_PBDEF_INIT_REQ;
    f_initialize_request.osif_block_size = sizeof(f_initialize_request);
    
    /*
     * Store the our systems presentation address in the parameter block
     */
    f_initialize_request.osif_local_p_addrs.p_address.address =
        (unsigned char *)LOCAL_P_ADDRESS;
    f_initialize_request.osif_local_p_addrs.p_address.length  =
        strlen (LOCAL_P_ADDRESS);
    f_initialize_request.osif_local_p_addrs.nsap_queue[0].nsap.address =
       (unsigned char *)LOCAL_NSAP;
    f_initialize_request.osif_local_p_addrs.nsap_queue[0].nsap.length  =
       strlen (LOCAL_NSAP);
    f_initialize_request.osif_local_p_addrs.nsap_queue[0].template.address =
       (unsigned char *)LOCAL_TEMPLATE;
    f_initialize_request.osif_local_p_addrs.nsap_queue[0].template.length  =
       strlen (LOCAL_TEMPLATE);
    f_initialize_request.osif_local_p_addrs.nsap_queue[0].provider.address =
       (unsigned char *)LOCAL_PROVIDER;
    f_initialize_request.osif_local_p_addrs.nsap_queue[0].provider.length  =
       strlen (LOCAL_PROVIDER);
    
    /*
     * Store the remote systems presentation address in the parameter block
     */
    f_initialize_request.osif_peer_p_addrs.p_address.address =
        (unsigned char *)REMOTE_P_ADDRESS;
    f_initialize_request.osif_peer_p_addrs.p_address.length  =
        strlen (REMOTE_P_ADDRESS);
    f_initialize_request.osif_peer_p_addrs.nsap_queue[0].nsap.address =
       (unsigned char *)REMOTE_NSAP;
    f_initialize_request.osif_peer_p_addrs.nsap_queue[0].nsap.length  =
       strlen (REMOTE_NSAP);
    f_initialize_request.osif_peer_p_addrs.nsap_queue[0].template.address =
       (unsigned char *)REMOTE_TEMPLATE;
    f_initialize_request.osif_peer_p_addrs.nsap_queue[0].template.length  =
       strlen (REMOTE_TEMPLATE);
    f_initialize_request.osif_peer_p_addrs.nsap_queue[0].provider.address =
       (unsigned char *)REMOTE_PROVIDER;
    f_initialize_request.osif_peer_p_addrs.nsap_queue[0].provider.length  =
       strlen (REMOTE_PROVIDER);
    
    /*
     * Propose the transfer, management, and transfer and management 
     * service class.
     * Note the FTAM responder will choose which one of these service classes
     * it will support for this association.
     */
    f_initialize_request.osif_service_class.length = 4;
    f_initialize_request.osif_service_class.value = (  OSIF_CLASS_XFR_MGMT |
						     OSIF_CLASS_MGMT |
						     OSIF_CLASS_XFR );
    
    /*
     * Propose the read, write, limited file management, enhanced 
     * file management and grouping functional units.
     */
    f_initialize_request.osif_functional_units.length = 4;
    f_initialize_request.osif_functional_units.value = 
			( OSIF_FU_READ |
			 OSIF_FU_WRITE |
               OSIF_FU_LIMIT_FILE_MGMT | 
		 OSIF_FU_ENH_FILE_MGMT |
		      OSIF_FU_GROUPING );
    
    /*
     *   Propose the storage attribute group
     */
    f_initialize_request.osif_attribute_groups.length = 4;
    f_initialize_request.osif_attribute_groups.value = 
		OSIF_ATG_STORAGE | OSIF_ATG_SECURITY;
    
    /*
     * Set the FTAM Quality of service to No Recovery
     */
    f_initialize_request.osif_ftam_qual_service.length = 4;
    f_initialize_request.osif_ftam_qual_service.value = 
	OSIF_FQOS_NO_RECOVERY;
    
    /*
     * Tell the responder which files type we can support
     * add FTAM-1, FTAM-2 and FTAM-3 to the contents_type_list
     */
    ctlblk_1.document_name.address = (unsigned char *)"FTAM-1";
    ctlblk_1.document_name.length = strlen( ctlblk_1.document_name.address );
    ctlblk_1.next = &ctlblk_2;
    
    ctlblk_2.document_name.address = (unsigned char *)"FTAM-2";
    ctlblk_2.document_name.length = strlen( ctlblk_2.document_name.address );
    ctlblk_2.next = &ctlblk_3;
    
    ctlblk_3.document_name.address = (unsigned char *)"FTAM-3";
    ctlblk_3.document_name.length = strlen( ctlblk_3.document_name.address );
    ctlblk_3.next = 0;
    
    f_initialize_request.osif_contents_type_list = &ctlblk_1;
    
    /*
     * Store the filestore password and the initiator id in the
     * parameter block.
     * Note that the type field of the sdesc structure is used for the
     * filestore password. This is because the password could be encoded
     * as either a graphic string or an octet string. The type field tells
     * the asn1 encoder how to encode this parameter.
     */
    f_initialize_request.osif_filestore_password.address = 
	(unsigned char *)fs_passwd;
    f_initialize_request.osif_filestore_password.length = 
	strlen(f_initialize_request.osif_filestore_password.address);
    f_initialize_request.osif_filestore_password.type = OSIF_UC_GRAPHIC;
    
    f_initialize_request.osif_initiator_identity.address = 
	(unsigned char *) init_id;
    f_initialize_request.osif_initiator_identity.length = 
	strlen( f_initialize_request.osif_initiator_identity.address);
    

    
    /*
     * Send the f-initialize request to the remote responder
     */
    status = osif_send( port_id, &f_initialize_request, &error_status );
    if ( OSIF_SUCCESS != status )
    {
	print_error_message("send_initialize:", error_status);
	exit(1);
    }

    /*
     *  Get the Response from the FTAM responder
     */
    
    status = osif_get_event( port_id, 
			    &f_initialize_response, 
			    -1, 
			    &error_status );
    if (status)
    {
	print_error_message("osif_get_event: ", error_status);
	exit(1);
    }
	
    /*
     * check the state and action results
     * Display the state and action results if they are not successful
     * If a diagnostic was sent with this primitive then display the
     * contents of the diagnostic
     */

    diagnostic_checking("send_initialize: ", &f_initialize_response);
    
    /*
     * Exit if we received an abort
     */
    if ( OSIF_PBDEF_P_ABORT == f_initialize_response.osif_block_type )
    {
	printf("EXITING:  Abort received\n");
	exit( 0 );
    }
    
    
    /*
     * A buffer MAY be returned along with the parameter block.
     * If so, pass the buffer back to the interface for re-use.
     */
    if ( f_initialize_response.osif_returned_buffer )
    {
	status = osif_give_buffer(  port_id, 
				  f_initialize_response.osif_returned_buffer,
				  &error_status );
    }
    
}



send_create( port_id )
/*
**++
**  FUNCTIONAL DESCRIPTION:
**
**     This routine builds the following primitives:
**	      	f_begin_group_request
**           	f_create_request
**		f_open_request
**		f_end_group_request
**
**	It receives the following FTAM events:
**	      	f_begin_group_response
**           	f_create_response
**		f_open_response
**		f_end_group_response
**
**  FORMAL PARAMETERS:
**
**      port_id
**
**  IMPLICIT INPUTS:
**
**      None
**
**  IMPLICIT OUTPUTS:
**
**      None
**
**  COMPLETION CODES:
**
**
**  SIDE EFFECTS:
**
**
**--
*/
unsigned port_id;
{
    unsigned                status;
    unsigned                error_status;
    struct osifpb           f_begin_group_request;
    struct osifpb           f_create_request;
    struct osifpb           f_open_request;
    struct osifpb           f_end_group_request;
    struct osifpb           f_response;
    
    struct osif_fn          create_filename;
    struct osif_access_ctl  access_ctl_element;

    printf("creating file...\n");
    
    ZERO( f_begin_group_request );
    ZERO( f_create_request );
    ZERO( f_open_request );
    ZERO( f_end_group_request );
    ZERO( f_response);  
    
    ZERO( create_filename );
    ZERO( access_ctl_element );

    /*
     * BEGIN GROUP REQUEST
     */
    
    /*
     * Set the begin group function code in the parameter block 
     */
    f_begin_group_request.osif_block_type = OSIF_PBDEF_BG_REQ;
    f_begin_group_request.osif_block_size = sizeof( struct osifpb );
    
    /*
     * Set the Threshold parameter
     * The value used reflects the number of
     * FTAM primitives within a grouping sequence.
     * In this case 2, f-create-req + f-open-req
     */
    f_begin_group_request.osif_threshold.value = 2;
    f_begin_group_request.osif_threshold.length = 1;
    
    /*
     * Send the Begin Group Request to the API
     */
    status = osif_send( port_id, &f_begin_group_request, &error_status );
    if ( OSIF_SUCCESS != status )
    {
	print_error_message("send_create:", error_status);
	exit(1);
    }
    
    
    /*
     * CREATE REQUEST 
     */
    
    /*
     * Set the create request function code in the parameter block 
     */
    f_create_request.osif_block_type = OSIF_PBDEF_CRE_REQ;
    f_create_request.osif_block_size = sizeof( struct osifpb );
    
    
    /*
     * Set up the filename structure. This structure is a linked list
     * At the moment the FTAM Profiles (ie NIST, EWOS, FTAM ISP)
     * only allow a single element file name
     */
    create_filename.next = 0;
    create_filename.filename.address = 
		(unsigned char *)"ftam_api_example.test";
    create_filename.filename.length = 
		strlen( create_filename.filename.address );
    
    /*
     * store the address of the filename structure in the parameter block
     */
    f_create_request.osif_filename = &create_filename;
    
    
    /*
     * Set the create override parameter to delete and create with new
     * attributes. This will over write the file if it exists.
     */
    f_create_request.osif_override.value = OSIF_OVR_DEL_CRE_NEW_ATTRIB;
    f_create_request.osif_override.length = 
		sizeof( f_create_request.osif_override.value );
    
    /*
     * Set the permitted actions to read (read, read_attribute)
     * write (insert, replace, extend, change_attribute, delete_file)
     */
    f_create_request.osif_permitted_actions.value = 
					( OSIF_PA_READ |
				          OSIF_PA_INSERT |
					  OSIF_PA_REPLACE |
					  OSIF_PA_EXTEND |
					  OSIF_PA_READ_ATTRIBUTE |
					  OSIF_PA_CHANGE_ATTRIBUTE |
					  OSIF_PA_DELETE_FILE );
    f_create_request.osif_permitted_actions.length=1;
    
    
    
    /*
     * set the file type to be an unstructured text file (FTAM-1)        
     */
    f_create_request.osif_contents_type.document_name.address = 
	(unsigned char *)"FTAM-1";
    f_create_request.osif_contents_type.document_name.length  = 
	strlen( f_create_request.osif_contents_type.document_name.address );
    
    /*
     * Set up the semantics of the file contents:
     * String lengths are unbounded (max_string_length)
     * String Significance is not significant (string_significance)
     * Supported character set is General String (universal_class)
     *
     */

    f_create_request.osif_contents_type.document_param.
	string_significance.value = OSIF_STRSIG_NS;
    f_create_request.osif_contents_type.document_param.
	string_significance.length = 4;
    f_create_request.osif_contents_type.document_param.
	max_string_length.value = OSIF_MSL_UNLIMITED;
    f_create_request.osif_contents_type.document_param.
	max_string_length.length = 4;
    f_create_request.osif_contents_type.document_param.
	universal_class.value = OSIF_UC_GENERAL;
    f_create_request.osif_contents_type.document_param.
	universal_class.length = 4;
    
    /*
     * Request replace, extend, read_attribute access           
     */
    f_create_request.osif_requested_access.value = (  OSIF_AR_REPLACE |
						    OSIF_AR_EXTEND |
						    OSIF_AR_READ_ATTRIBUTE );
    f_create_request.osif_requested_access.length = 1;

    /*
     * Access control - action list.
     */
    f_create_request.osif_access_control.action_list.length =
	sizeof (int);
    f_create_request.osif_access_control.action_list.value = 
	(int) ( OSIF_AR_REPLACE | OSIF_AR_EXTEND | OSIF_AR_READ_ATTRIBUTE );

    /*
     * Access control - concurrency access.
     */
    f_create_request.osif_access_control.concurrency_access.
	replace_cc.value = OSIF_CA_NO_ACCESS;
    f_create_request.osif_access_control.concurrency_access.
	replace_cc.length = sizeof (int);
    f_create_request.osif_access_control.concurrency_access.
	extend_cc.value = OSIF_CA_NO_ACCESS;
    f_create_request.osif_access_control.concurrency_access.
	extend_cc.length = sizeof (int);
    f_create_request.osif_access_control.concurrency_access.
	read_attrib_cc.value = OSIF_CA_SHARED;
    f_create_request.osif_access_control.concurrency_access.
	read_attrib_cc.length = sizeof (int);
    
    /*
     * Access control - identity.
     */
    f_create_request.osif_access_control.identity.address = 
	(unsigned char *) "User-1";
    f_create_request.osif_access_control.identity.length = 
	strlen (f_create_request.osif_access_control.identity.address);

    f_create_request.osif_access_control.next = 
	(struct osif_access_ctl *) &access_ctl_element;

    /*
     * Access control - action list.
     */
    access_ctl_element.action_list.length =
	sizeof (int);
    access_ctl_element.action_list.value = (int) (  OSIF_AR_READ_ATTRIBUTE );

    /*
     * Access control - concurrency access.
     */
    access_ctl_element.concurrency_access.
	read_attrib_cc.value = OSIF_CA_SHARED;
    access_ctl_element.concurrency_access.
	read_attrib_cc.length = sizeof (int);
    
    /*
     * Access control - identity.
     */
    access_ctl_element.identity.address = 
	(unsigned char *) "User-2";
    access_ctl_element.identity.length = 
	strlen (access_ctl_element.identity.address);

    access_ctl_element.next = (struct osif_access_ctl *) 0;

    /*
     * Send the create request to the API
     */
    status = osif_send( port_id, &f_create_request, &error_status );
    if ( OSIF_SUCCESS != status )
    {
	print_error_message("send_create: ", error_status);
	exit(1);
    }
    
    
    /*
     * OPEN REQUEST
     */
    
    /* 
     * Set the open request function code in the parameter block 
     */
    f_open_request.osif_block_type = OSIF_PBDEF_OPEN_REQ;
    f_open_request.osif_block_size = sizeof( struct osifpb );
    
    /*   
     * Set the Processing mode to replace and extend so we can write to file 
     */
    /* oss - change OSIF_PA_* to OSIF_PM_* */
    f_open_request.osif_processing_mode.value = 
	( OSIF_PM_REPLACE | OSIF_PM_EXTEND );
    f_open_request.osif_processing_mode.length = sizeof(int);
    
    /*
     * set the file type to be an unstructured text file (FTAM-1)        
     */
    f_open_request.osif_contents_type.document_name.address = 
	(unsigned char *)"FTAM-1";
    f_open_request.osif_contents_type.document_name.length  = 
	strlen( f_open_request.osif_contents_type.document_name.address );
    
    /*
     * Set up the semantics of the file contents:
     * String lengths are unbounded (max_string_length)
     * String Significance is not significant (string_significance)
     * Supported character set is General String (universal_class)
     *
     */

    f_open_request.osif_contents_type.document_param.
	string_significance.value = OSIF_STRSIG_NS;
    f_open_request.osif_contents_type.document_param.
	string_significance.length = 4;
    f_open_request.osif_contents_type.document_param.
	max_string_length.value = OSIF_MSL_UNLIMITED;
    f_open_request.osif_contents_type.document_param.
	max_string_length.length = 4;
    f_open_request.osif_contents_type.document_param.
	universal_class.value = OSIF_UC_GENERAL;
    f_open_request.osif_contents_type.document_param.
	universal_class.length = 4;
    
    /*
     *  Send the open request to the API
     */
    status = osif_send( port_id, &f_open_request, &error_status );
    if ( OSIF_SUCCESS != status )
    {
	print_error_message("send_create: ", error_status);
	exit(1);
    }
    
    
    /*
     * END GROUP REQUEST
     */
    
    /*
     * Set the end group request function code in the parameter block 
     */
    f_end_group_request.osif_block_type =  OSIF_PBDEF_EG_REQ;
    f_end_group_request.osif_block_size = sizeof( struct osifpb );
    
    /* 
     * Send the end group request to the API.
     * This primitive will result in the entire group being 
     * sent to the FTAM responder.
     */
    status = osif_send( port_id, &f_end_group_request, &error_status );
    if ( OSIF_SUCCESS != status )
    {
	print_error_message("send_create: ", error_status);
	exit(1);
    }
    
    /* 
     * Get the responses from the FTAM responder
     */
    for ( f_response.osif_block_type=0;
	 ( OSIF_PBDEF_EG_RSP != f_response.osif_block_type )
	 ; )
    {
	
	/*
	 * If the block is to be re-used, be sure to initialize it
         * before it is used again
         */
	ZERO( f_response);
	status = osif_get_event( port_id, &f_response, OSIF_WAIT_INFINITE,
				&error_status );
	if (status)
	{
	    print_error_message("osif_get_event: ", error_status);
	    exit(1);
	}
	
	/*
	 * Check the state and action results.
	 * Display the state and action results if they are not 
	 * successful.  If a diagnostic was sent with this 
	 * primitive then display the contents of the diagnostic
	 */
	diagnostic_checking("send_create: ", &f_response);
	
	
	/*
	 * If the event we received was an abort, then
	 * the diagnostic displayed applies to this abort,
	 * exit
	 */
	if ( OSIF_PBDEF_P_ABORT == f_response.osif_block_type )
	{
	    printf("EXITING:  Abort received \n");
	    exit( 0 );
	}
	
	/*
	 * If the buffer suppiled to give buffers is returned to
	 * the user a point to it will be stored in the returned buffer
	 * parameter. Check this parameter if there is a buffer return
	 * it to the API pool through the osif_give_buffer call
	 */
	
	if ( f_response.osif_returned_buffer )
	{
	    status = osif_give_buffer(  port_id, 
				      f_response.osif_returned_buffer,
				      &error_status );
	    if (status)
	    {
		print_error_message("osif_give_buffer: ", error_status);
		exit(1);
	    }
	
		
	}
	
    }
    
    
}



send_read_attributes( port_id)
/*
**++
**  FUNCTIONAL DESCRIPTION:
**
**    This routine builds the following primitives:
** 	      f_begin_group_request
**            f_select_request
**            f_read_attribute_request
**            f_deselect_request
**            f_end_group_request
**    
**    It receives the following FTAM events:
** 	      f_begin_group_response
**            f_select_response
**            f_read_attribute_response
**            f_deselect_response
**            f_end_group_response
**
**  FORMAL PARAMETERS:
**
**      port_id
**
**  IMPLICIT INPUTS:
**
**      None
**
**  IMPLICIT OUTPUTS:
**
**      None
**
**  COMPLETION CODES:
**
**
**  SIDE EFFECTS:
**
**
**--
*/
unsigned port_id;
{
    unsigned                status;
    unsigned                error_status;
    struct osifpb           f_begin_group_request;
    struct osifpb           f_select_request;
    struct osifpb           f_read_attribute_request;
    struct osifpb           f_deselect_request;
    struct osifpb           f_end_group_request;
    struct osifpb           f_response;  
    struct osif_fn          select_filename;
    
    printf("reading file attributes...\n");
    
    ZERO( f_begin_group_request );
    ZERO( f_select_request);
    ZERO( f_read_attribute_request );
    ZERO( f_deselect_request );
    ZERO( f_end_group_request );
    ZERO( f_response );
    
    ZERO( select_filename );
    
    /*
     * BEGIN GROUP REQUEST
     */
    
    /* 
     * Set the begin group function code in the parameter block
     */
    f_begin_group_request.osif_block_type = OSIF_PBDEF_BG_REQ;
    f_begin_group_request.osif_block_size = sizeof( struct osifpb );
    
    /* 
     * Set the Threshold parameter
     * The value of used reflects the number of
     * FTAM primitives within a grouping sequence.
     * In this case 3, f-select-req + f-read-attributes-req + f-deselect-req
     */
    f_begin_group_request.osif_threshold.value = 3;
    f_begin_group_request.osif_threshold.length = 1;
    
    /* 
     * Send the Begin Group Request to the API
     */
    status = osif_send( port_id, &f_begin_group_request, &error_status );
    if ( OSIF_SUCCESS != status )
	print_error_message("send_read_attributes: ", error_status);
    
    /*
     * SELECT REQUEST
     */
    
    /* 
     * Set the select request function code in the parameter block
     */
    f_select_request.osif_block_type = OSIF_PBDEF_SEL_REQ;
    f_select_request.osif_block_size = sizeof( struct osifpb );
    
    
    /* 
     * Set up the filename structure. This structure is a linked list
     * At the moment the FTAM Profiles (ie NIST, EWOS, FTAM ISP)
     * only allow a single element file name
     */
    select_filename.next = 0;
    select_filename.filename.address = 
	(unsigned char *)"ftam_api_example.test";
    select_filename.filename.length = 
	strlen( select_filename.filename.address );

    /*
     * store the address of the filename structure in the parameter block
     */
    f_select_request.osif_filename = &select_filename;
    
    
    /* 
     * Set the requested access calue to read (read_attribute),
     * write (replace, extend)
     */
    f_select_request.osif_requested_access.value = ( OSIF_AR_READ_ATTRIBUTE |
						    OSIF_AR_REPLACE |
						    OSIF_AR_EXTEND );
    
    f_select_request.osif_requested_access.length=sizeof(int);
    
    /* 
     * Send the select request to the API
     */
    status = osif_send( port_id, &f_select_request, &error_status );
    if ( OSIF_SUCCESS != status )
	print_error_message("send_read_attributes: ", error_status);
    
    
    /* 
     * READ ATTRIBUTE REQUEST
     */
    
    /* 
     * Set the read-attribute-request function code in the parameter block
     */
    f_read_attribute_request.osif_block_type = OSIF_PBDEF_RAT_REQ;
    f_read_attribute_request.osif_block_size = 
	sizeof( f_read_attribute_request );
    
    /*   
     * Set the attribute names to the following
     */
    
    f_read_attribute_request.osif_attribute_names.value = 
	( OSIF_ATT_FILENAME
	 | OSIF_ATT_PERMITTED_ACTIONS
	 | OSIF_ATT_CONTENTS_TYPE
	 | OSIF_ATT_MODIFICATION_TIME
	 | OSIF_ATT_CREATOR_ID
	 | OSIF_ATT_FILESIZE
	 | OSIF_ATT_STORAGE_ACCOUNT
	 | OSIF_ATT_CREATION_TIME
	 | OSIF_ATT_READ_TIME
	 | OSIF_ATT_ATTRIBUTE_MODIFICATION_TIME
	 | OSIF_ATT_MODIFIER_ID
	 | OSIF_ATT_READER_ID
	 | OSIF_ATT_ATTRIBUTE_MODIFIER_ID
	 | OSIF_ATT_FILE_AVAILABILITY
	 | OSIF_ATT_FUTURE_FILESIZE 
	 | OSIF_ATT_ACCESS_CONTROL 
	 );
    
    f_read_attribute_request.osif_attribute_names.length = sizeof(int); 
    
    
    /* 
     * Send the read attribute request to the API
     */
    status = osif_send( port_id, &f_read_attribute_request, &error_status );
    if ( OSIF_SUCCESS != status )
	print_error_message("send_read_attributes: ", error_status);
    
    
    
    /*
     * DESELECT REQUEST
     */

    /*
     * Set the deselect request function code in the parameter block
     */
    f_deselect_request.osif_block_type = OSIF_PBDEF_DESELECT_REQ;
    f_deselect_request.osif_block_size = sizeof (struct osifpb);
    
    /* 
     * Send the Deselect request to the API
     */
    status = osif_send( port_id, &f_deselect_request, &error_status );
    if ( OSIF_SUCCESS != status )
	print_error_message("send_read_attributes: ", error_status);
    
    
    /*
     * END GROUP REQUEST
     */
    
    /* 
     * Set the End group request code in the parameter block
     */
    f_end_group_request.osif_block_type =  OSIF_PBDEF_EG_REQ;
    f_end_group_request.osif_block_size = sizeof( struct osifpb );
    
    /* 
     * Send the End group request code in the parameter block
     */
    status = osif_send( port_id, &f_end_group_request, &error_status );
    if ( OSIF_SUCCESS != status )
	print_error_message("send_read_attributes: ", error_status);
    
    /*
     * Get the responses from the responder
     */    
    for ( f_response.osif_block_type=0;
	 ( OSIF_PBDEF_EG_RSP != f_response.osif_block_type )
	 ; )
    {
	ZERO(f_response);
	status = osif_get_event( port_id, &f_response, -1,
				&error_status );
	if (status)
	    print_error_message("osif_get_event: ", error_status);
	
	/*
	 * check the state and action results
         * Display the state and action results if they are not successful
         * If a diagnostic was sent with this primitive then display the
         * contents of the diagnostic
	 */
	diagnostic_checking("send_read_attributes: ", &f_response);
	
	
	/*
	 * If the event we received was an abort, then
         * the diagnostic displayed applies to this abort,
         * exit
	 */
	if ( OSIF_PBDEF_P_ABORT == f_response.osif_block_type )
	{
	    printf("EXITING:  Abort received \n");
	    exit( 0 );
	}
	
	
	/*
	 * If the response we receive is of type ReadAtributes, then
	 * print out the filename
	 */
	if (f_response.osif_block_type == OSIF_PBDEF_RAT_RSP )
	{
	    if (f_response.osif_filename)
	    {
	        printf("file name is: %.*s\n",
			f_response.osif_filename->filename.length,
			f_response.osif_filename->filename.address);	    
	    }
	}
	
	
	/*
         * If the buffer suppiled to give buffers is returned to
         * the user a point to it will be stored in the returned buffer
         * parameter. Check this parameter if there is a buffer return
         * it to the API pool through the osif_give_buffer call
         */
	if ( f_response.osif_returned_buffer )
	{
	    status = osif_give_buffer(  port_id, 
				      f_response.osif_returned_buffer,
				      &error_status );
	    if (status)
	    {
		print_error_message("osif_give_buffer: ", error_status);
		exit(1);
	    }
	
	    
	}
    }
    
}




send_change_attributes ( port_id )
/*
**++
**  FUNCTIONAL DESCRIPTION:
**
**     This routine builds the following primitives:
**             f_begin_group_request
**             f_select_request
**             f_change_request
**             f_deselect_request
**             f_end_group_request
**
**     It receives the following FTAM events:
**             f_begin_group_response
**             f_select_response
**             f_change_response
**             f_deselect_response
**             f_end_group_response
**
**  FORMAL PARAMETERS:
**
**	port_id
**
**  IMPLICIT INPUTS:
**
**	None
**
**  IMPLICIT OUTPUTS:
**
**	None
**
**  COMPLETION CODES:
**
**
**  SIDE EFFECTS:
**
**	The 'osifpb' structure is filled in.
**
**--
*/


unsigned port_id;
{
    unsigned                status;
    unsigned                error_status;
    struct osifpb           f_begin_group_request;
    struct osifpb           f_select_request;
    struct osifpb           f_change_attributes_request;
    struct osifpb           f_deselect_request;
    struct osifpb           f_end_group_request;
    struct osifpb           f_response;
    struct osif_fn          select_filename;
    struct osif_fn          change_attributes_filename;
    
    printf("changing file attributes (name)...\n");

    ZERO( f_begin_group_request );
    ZERO( f_select_request);
    ZERO( f_change_attributes_request );
    ZERO( f_deselect_request );
    ZERO( f_end_group_request );
    ZERO( f_response);

    ZERO( select_filename );
    ZERO( change_attributes_filename );
    
    /*
     * BEGIN GROUP REQUEST
     */
    
    /* 
     * Set the Begin group request function code in the parameter block
     */
    f_begin_group_request.osif_block_type = OSIF_PBDEF_BG_REQ;
    f_begin_group_request.osif_block_size = sizeof( struct osifpb );
    
    /* 
     * Set the Threshold parameter
     * The value of used reflects the number of
     * FTAM primitives within a grouping sequence.
     * In this case, it is three: select + change_attributes + deselect
     */
    f_begin_group_request.osif_threshold.value = 3; 
    f_begin_group_request.osif_threshold.length = 1;
    
    /* 
     * Send the Begin Group Request to the API
     */
    status = osif_send( port_id, &f_begin_group_request, &error_status );
    
    if ( OSIF_SUCCESS != status )
	print_error_message("send_change_attributes: ", error_status);
    
    
    /*
     * SELECT REQUEST
     */
    
    /* 
     * Set the select request function code in the parameter block
     */
    f_select_request.osif_block_type = OSIF_PBDEF_SEL_REQ;
    f_select_request.osif_block_size = sizeof( struct osifpb );
    
    
    /* 
     * Set up the filename structure. This structure is a linked list
     * At the moment the FTAM Profiles (ie NIST, EWOS, FTAM ISP)
     * only allow a single element file name
     */
    select_filename.next = 0;
    select_filename.filename.address = 
			(unsigned char *)"ftam_api_example.test";
    select_filename.filename.length = 
			strlen( select_filename.filename.address );


    /*
     * store the address of the filename structure in the parameter block
     */
    f_select_request.osif_filename = &select_filename;
    
    
    /* 
     * Set Requested Access value to OSIF_PA_CHANGE_ATTRIBUTE
     */
    f_select_request.osif_requested_access.value = OSIF_AR_CHANGE_ATTRIBUTE;
    f_select_request.osif_requested_access.length=sizeof(int);
    
    /* 
     * Send the select request to the Api
     */
    status = osif_send( port_id, &f_select_request, &error_status );
    if ( OSIF_SUCCESS != status )
	print_error_message("send_change_attributes: ", error_status);
    
    /* 
     * CHANGE ATTRIBUTE REQUEST 
     */
    
    /* 
     * Set the change attribute request function code in the 
     * paramter block
     */
    f_change_attributes_request.osif_block_type = OSIF_PBDEF_CHAT_REQ;
    f_change_attributes_request.osif_block_size = 
		sizeof( f_change_attributes_request );
    
    /* 
     * Set up the filename structure.  The selected file will be 
     * changed to the filename specified here.
     */
    change_attributes_filename.next = 0;
    change_attributes_filename.filename.address = 
		(unsigned char *)"ftam_api_example_moved.dat";
    change_attributes_filename.filename.length = 
		strlen( change_attributes_filename.filename.address );

    /*
     * store the address of the filename structure in the parameter block
     */
    f_change_attributes_request.osif_filename 
	= &change_attributes_filename;

    /*
     * Change attributes insert values - action list.
     */
    f_change_attributes_request.osif_insert_values.
	action_list.length = sizeof (int);
    f_change_attributes_request.osif_insert_values.
	action_list.value = (int) ( OSIF_AR_DELETE_OBJECT );

    /*
     * Access control - concurrency access.
     */
    f_change_attributes_request.osif_insert_values.concurrency_access.
	delete_file_cc.value = OSIF_CA_SHARED;
    f_change_attributes_request.osif_insert_values.concurrency_access.
	delete_file_cc.length = sizeof (int);
    
    /*
     * Access control - identity.
     */
    f_change_attributes_request.osif_insert_values.identity.address = 
	(unsigned char *) "User-3";
    f_change_attributes_request.osif_insert_values.identity.length = 
	strlen (f_change_attributes_request.osif_insert_values.identity.address);

    f_change_attributes_request.osif_insert_values.next = 
	(struct osif_access_ctl *) 0;

    /*
     * Change attributes delete values - action list.
     */
    f_change_attributes_request.osif_delete_values.
	action_list.length = sizeof (int);
    f_change_attributes_request.osif_delete_values.
	action_list.value = (int) (  OSIF_AR_READ_ATTRIBUTE );

    /*
     * Access control - concurrency access.
     */
    f_change_attributes_request.osif_delete_values.concurrency_access.
	read_attrib_cc.value = OSIF_CA_SHARED;
    f_change_attributes_request.osif_delete_values.concurrency_access.
	read_attrib_cc.length = sizeof (int);
    
    /*
     * Access control - identity.
     */
    f_change_attributes_request.osif_delete_values.identity.address = 
	(unsigned char *) "User-1";
    f_change_attributes_request.osif_delete_values.identity.length = 
	strlen (f_change_attributes_request.osif_delete_values.identity.address);

    f_change_attributes_request.osif_delete_values.next = 
	(struct osif_access_ctl *) 0;
    
    /*
     * Send the Change attribute request to the API
     */
    status = 
	osif_send( port_id, &f_change_attributes_request, &error_status );
    if ( OSIF_SUCCESS != status )
	print_error_message("send_change_attributes: ", error_status);
    
    
    
    /* 
     * DESELECT REQUEST 
     */
    
    /* 
     * Set the deselect request function code in the parameter block
     */
    f_deselect_request.osif_block_type = OSIF_PBDEF_DESELECT_REQ;
    f_deselect_request.osif_block_size = sizeof( f_deselect_request );
    
    /*
     * Send the deselect request to the API
     */
    status = osif_send( port_id, &f_deselect_request, &error_status );
    if ( OSIF_SUCCESS != status )
    {
	print_error_message("send_change_attributes: ", error_status);
    }
    
    /*
     * END GROUP REQUEST
     */
    
    /* 
     * Set the end group request function code in the parameter block
     */
    f_end_group_request.osif_block_type =  OSIF_PBDEF_EG_REQ;
    f_end_group_request.osif_block_size = sizeof( struct osifpb );
    
    /* 
     * Send the end group request to the API.
     * This primitive will result in the entire group being
     * sent to the FTAM responder.
     */
    status = osif_send( port_id, &f_end_group_request, &error_status );
    if ( OSIF_SUCCESS != status )
	print_error_message("send_change_attributes: ", error_status);
    
    /*
     * Get the responses from the FTAM responder
     */
    for ( f_response.osif_block_type=0;
	 ( OSIF_PBDEF_EG_RSP != f_response.osif_block_type )
	 ; )
    {
	ZERO(f_response);
	status = osif_get_event( port_id, &f_response, -1,
				&error_status );
	
	if (status)
	    print_error_message("osif_get_event: ", error_status);
	
	/*
	 * check the state and action results
         * Display the state and action results if they are not successful
         * If a diagnostic was sent with this primitive then display the
         * contents of the diagnostic
	 */
	diagnostic_checking("send_change_attributes: ", &f_response);
	
	/*
	 * If the event we received was an abort, then
         * the diagnostic displayed applies to this abort,
         * exit
	 */
	if ( OSIF_PBDEF_P_ABORT == f_response.osif_block_type )
	{
	    printf("EXITING:  Abort received \n");
	    exit( 0 );
	}
	
	/*
	 *  When the block type in the response parameter block is
         *  CHangeATtibutes, print out the changed filename
	 */
	
	if ( OSIF_PBDEF_CHAT_RSP == f_response.osif_block_type )
	{
	    if (f_response.osif_filename)
	    {
		printf("new file name is: %.*s\n",
			f_response.osif_filename->filename.length,
			f_response.osif_filename->filename.address);
	    }
	    
	}

        /*
         * If the buffer suppiled to give buffers is returned to
         * the user a point to it will be stored in the returned buffer
         * parameter. Check this parameter if there is a buffer return
         * it to the API pool through the osif_give_buffer call
         */

	if ( f_response.osif_returned_buffer )
	{
	    status = osif_give_buffer(  port_id, 
				      f_response.osif_returned_buffer,
				      &error_status );
	    if (status)
	    {
		print_error_message("osif_give_buffer: ", error_status);
		exit(1);
	    }
	
	}
    }
    
}



send_terminate( port_id )
/*
**++
**  FUNCTIONAL DESCRIPTION:
**
**     This routine builds the following primitives:
**             f_terminate_request
**
**     It receives the following FTAM events:
**             f_terminate_response
**
**  FORMAL PARAMETERS:
**
**	port_id
**
**  IMPLICIT INPUTS:
**
**	None.
**
**  IMPLICIT OUTPUTS:
**
**	None.
**
**  COMPLETION CODES:
**
**
**  SIDE EFFECTS:
**
**
**--
*/
unsigned       port_id;
{
    unsigned status;                    /* Call completion status */
    unsigned error_status;              /* Additional status info value  */
    struct osifpb f_terminate_request;  /* Request parameter block */
    struct osifpb f_terminate_response;
    
    printf("terminating connection...\n");

    ZERO(f_terminate_request );
    ZERO(f_terminate_response);
    
    /*
     * Set the terminate_request function code in the parameter block.
     * This will terminate the FTAM regime closing the association.
     */
    f_terminate_request.osif_block_type = OSIF_PBDEF_TERM_REQ;
    f_terminate_request.osif_block_size = sizeof(f_terminate_request);
    
    /*
     * Send the terminate request to the API.
     */
    status = osif_send( port_id, &f_terminate_request, &error_status );
    if ( OSIF_SUCCESS != status )
	print_error_message("send_terminate: ", error_status);
    
    
    /*
     * Get response from the FTAM responder
     */
    status = osif_get_event( port_id, &f_terminate_response, -1,
			    &error_status );
    if (status)
	print_error_message("osif_get_event: ", error_status);
    
    
    /*
     * check the state and action results
     * Display the state and action results if they are not successful
     * If a diagnostic was sent with this primitive then display the
     * contents of the diagnostic
      */
    diagnostic_checking("send_terminate: ", &f_terminate_response);
	
    
}


send_close ( port_id )
/*
**++
**  FUNCTIONAL DESCRIPTION:
**  
**	This routine builds the following primitives:
**             f_begin_group_request
**             f_close_request
**             f_deselect_request
**             f_end_group_request
**
**     It receives the following FTAM events:
**             f_begin_group_response
**             f_close_response
**             f_deselect_response
**             f_end_group_response
**
**  FORMAL PARAMETERS:
**
**	port_id
**
**  IMPLICIT INPUTS:
**
**	None
**
**  IMPLICIT OUTPUTS:
**
**	None
**
**  FUCTION VALUE:
**  COMPLETION CODES:
**
**	None
**
**  SIDE EFFECTS:
**
**
**--
*/


unsigned port_id;
{ 
    unsigned                status;
    unsigned                error_status;
    struct osifpb           f_begin_group_request;
    struct osifpb           f_close_request;
    struct osifpb           f_deselect_request;
    struct osifpb           f_end_group_request;
    struct osifpb           f_response;
    
    printf("closing file...\n");

    ZERO( f_begin_group_request );
    ZERO( f_close_request );
    ZERO( f_deselect_request );
    ZERO( f_end_group_request );
    ZERO( f_response);
    
    /*
     * BEGIN GROUP REQUEST
     */
    
    /*
     * Set the begin group function code in the parameter block
     */
    f_begin_group_request.osif_block_type = OSIF_PBDEF_BG_REQ;
    f_begin_group_request.osif_block_size = sizeof( struct osifpb );
    
    /* 
     * Set the Threshold parameter
     * The value of used reflects the number of
     * FTAM primitives within a grouping sequence
     * In this case 2, f-close-req + f-deselect-req  
     */
    f_begin_group_request.osif_threshold.value = 2;  
    f_begin_group_request.osif_threshold.length = 1;
    
    /*
     * Send Begin group request to the API
     */
    status = osif_send( port_id, &f_begin_group_request, &error_status );
    
    if ( OSIF_SUCCESS != status )
	print_error_message("send_close: ", error_status);
    
    /* 
     * CLOSE REQUEST 
     */
    
    /*
     * Set the Close request function code in the parameter block
     * This will cause the remote file to be closed.
     */
    f_close_request.osif_block_type = OSIF_PBDEF_CLOSE_REQ;
    f_close_request.osif_block_size = sizeof( f_close_request );
    
    /*
     * Send the Close request to the API.
     */
    status = osif_send( port_id, &f_close_request, &error_status );
    if ( OSIF_SUCCESS != status )
	print_error_message("send_close: ", error_status);
    
    
    /*
     * DESELECT REQUEST 
     */
    
    /* 
     * Set the Deselect request function code in the parameter block
     */
    f_deselect_request.osif_block_type = OSIF_PBDEF_DESELECT_REQ;
    f_deselect_request.osif_block_size = sizeof( f_deselect_request );
    
    /*
     * Send the Deselect request to the API
     */
    status = osif_send( port_id, &f_deselect_request, &error_status );
    if ( OSIF_SUCCESS != status )
	print_error_message("send_close: ", error_status);
    
    
    /*
     * END GROUP REQUEST
     */
    
    /*
     * Set the end group request function code in the parameter block
     */
    f_end_group_request.osif_block_type =  OSIF_PBDEF_EG_REQ;
    f_end_group_request.osif_block_size = sizeof( struct osifpb );
    
    /*
     * Send the End group request function code in the parameter block
     */
    status = osif_send( port_id, &f_end_group_request, &error_status );
    
    if ( OSIF_SUCCESS != status )
	print_error_message("send_close: ", error_status);
    
    
    
    /*
     * Get the responses from the FTAM responder
     */
    for ( f_response.osif_block_type=0;
	 ( OSIF_PBDEF_EG_RSP != f_response.osif_block_type )
	 ; )
    {
	ZERO(f_response);
	status = osif_get_event( port_id, &f_response, -1,
				&error_status );
	
	if (status)
	    print_error_message("osif_get_event: ", error_status);
		
	/*
	 * check the state and action results
         * Display the state and action results if they are not successful
         * If a diagnostic was sent with this primitive then display the
         * contents of the diagnostic
	 */
	diagnostic_checking("send_terminate: ", &f_response);
	
	/*
	 * If the event we received was an abort, then
         * the diagnostic displayed applies to this abort,
         * exit
	 */
	if ( OSIF_PBDEF_P_ABORT == f_response.osif_block_type )
	{
	    printf("EXITING:  Abort received \n");
	    exit( 0 );
	}
	
	
        /*
         * If the buffer suppiled to give buffers is returned to
         * the user a point to it will be stored in the returned buffer
         * parameter. Check this parameter if there is a buffer return
         * it to the API pool through the osif_give_buffer call
         */
	if ( f_response.osif_returned_buffer )
	{
	    status = osif_give_buffer(  port_id, 
				      f_response.osif_returned_buffer,
				      &error_status );
	    if (status)
	    {
		print_error_message("osif_give_buffer: ", error_status);
		exit(1);
	    }
	
	}
    }
    
}



send_delete ( port_id )
/*
**++
**  FUNCTIONAL DESCRIPTION:
**
**     This routine builds the following primitives:
**             f_begin_group_request
**             f_select_request
**             f_delete_request
**             f_end_group_request
**
**     It receives the following FTAM events:
**             f_begin_group_response
**             f_select_response
**             f_delete_response
**             f_end_group_request
**
**  FORMAL PARAMETERS:
**
**	port_id
**
**  IMPLICIT INPUTS:
**
**	None
**
**  IMPLICIT OUTPUTS:
**
**	None
**
**  FUCTION VALUE:
**  COMPLETION CODES:
**
**	None
**
**  SIDE EFFECTS:
**
**	None
**
**--
*/
unsigned port_id;
{
    unsigned                status;
    unsigned                error_status;
    struct osifpb           f_begin_group_request;
    struct osifpb           f_select_request;
    struct osifpb           f_delete_request;
    struct osifpb           f_end_group_request;
    struct osifpb           f_response;
    
    struct osif_fn       select_filename;
    
    printf("deleting file...\n");

    ZERO( f_begin_group_request );
    ZERO( f_select_request );
    ZERO( f_delete_request );
    ZERO( f_end_group_request );
    ZERO( f_response);
    
    ZERO( select_filename );

    /*
     * BEGIN GROUP REQUEST
     */
    
    /*
     * Set the begin group function code in the parameter block
     */
    f_begin_group_request.osif_block_type = OSIF_PBDEF_BG_REQ;
    f_begin_group_request.osif_block_size = sizeof( struct osifpb );
    
    /*
     * Set the Threshold parameter
     * The value of used reflects the number of
     * FTAM primitives within a grouping sequence.
     * In this case 2, f-select + f-delete
     */
    f_begin_group_request.osif_threshold.value = 2; 
    f_begin_group_request.osif_threshold.length = 1;
    
    /*
     * Send the Begin group request to the API
     */
    status = osif_send( port_id, &f_begin_group_request, &error_status );
    if ( OSIF_SUCCESS != status )
	print_error_message("send_delete: ", error_status);
    
        
    /*
     * SELECT REQUEST
     */
    
    /*
     * Set the Select request function code in the parameter block
     */
    f_select_request.osif_block_type = OSIF_PBDEF_SEL_REQ;
    f_select_request.osif_block_size = sizeof( struct osifpb );
    
    
    /*
     * Set up the filename structure.  This contains the name of file
     * to be deleted.
     */
    select_filename.next = 0;
    select_filename.filename.address = 
		(unsigned char *)"ftam_api_example_moved.dat";
    select_filename.filename.length = 
		strlen( select_filename.filename.address );

    /*
     * store the address of the filename structure in the parameter block
     */
    f_select_request.osif_filename = &select_filename;
    
    
    /*
     * Set the Requested Access to OSIF_PA_DELETE_FILE.  We need to
     * set it to this requested access for the file to be deleted.
     */
    f_select_request.osif_requested_access.value = OSIF_AR_DELETE_OBJECT;
    f_select_request.osif_requested_access.length=sizeof(int);
    
    /*
     * Send the Select request to the API
     */
    status = osif_send( port_id, &f_select_request, &error_status );
    
    if ( OSIF_SUCCESS != status )
	print_error_message("send_delete: ", error_status);
    
    
    
    /*
     * DELETE REQUEST
     */

    /*
     * Set the delete request function code in the parameter block.
     * This will cause the remote file to be deleted.
     */
    f_delete_request.osif_block_type = OSIF_PBDEF_DELETE_REQ;
    f_delete_request.osif_block_size = sizeof( f_delete_request );
    
    status = osif_send( port_id, &f_delete_request, &error_status );
    if ( OSIF_SUCCESS != status )
	print_error_message("send_delete: ", error_status);
    
    /*
     * END GROUP REQUEST
     */
    
    /*
     * Send the End group request function code in the parameter block
     */
    f_end_group_request.osif_block_type =  OSIF_PBDEF_EG_REQ;
    f_end_group_request.osif_block_size = sizeof( struct osifpb );
    
    /*
     * Send the End group request to the API
     */
    status = osif_send( port_id, &f_end_group_request, &error_status );
    if ( OSIF_SUCCESS != status )
	print_error_message("send_delete: ", error_status);
    
    
    /*
     * Now wait for the grouped responses from the FTAM responser.  
     * We're expecting begin-group, select, delete and 
     * end-group responses.  Read until we get the end group.
     *
     */
    for ( f_response.osif_block_type=0;
	 ( OSIF_PBDEF_EG_RSP != f_response.osif_block_type )
	 ; )
    {
	ZERO(f_response);
	status = osif_get_event( port_id, &f_response, -1,
				&error_status );
	if (status)
	    print_error_message("osif_get_event: ", error_status);
	
	
	/*
	 * check the state and action results
         * Display the state and action results if they are not successful
         * If a diagnostic was sent with this primitive then display the
         * contents of the diagnostic
	 */
	diagnostic_checking("send_delete: ", &f_response);
	
	/*
	 * If the event we received was an abort, then
         * the diagnostic displayed applies to this abort,
         * exit
	 */
	if ( f_response.osif_returned_buffer )
	{
	    status = osif_give_buffer(  port_id, 
				      f_response.osif_returned_buffer,
				      &error_status );
	    if (status)
	    {
		print_error_message("osif_give_buffer: ", error_status);
		exit(1);
	    }
	
	}
    }
    
}





send_write ( port_id )
/*
**++
**  FUNCTIONAL DESCRIPTION:
**
**     This routine builds the following primitives:
**             f_write_request
**
**     Note that the write request is an unconfirmed service, so
**     we do not wait for a response to be returned.
**
**  FORMAL PARAMETERS:
**
**	port_id
**
**  IMPLICIT INPUTS:
**
**	None
**
**  IMPLICIT OUTPUTS:
**
**	None
**
**  COMPLETION CODES:
**
**	Completion code returned by the API 'osif_send' routine.
**
**  SIDE EFFECTS:
**
**	The 'osifpb' structure is filled in.
**
**--
*/  
unsigned port_id;
{
    int                     status;
    int                     error_status;
    struct osifpb           f_write_request;
    
    printf("writing file...\n");

    ZERO( f_write_request );
    
    /* 
     * Set the write request function code in the parameter block
     */
    f_write_request.osif_block_type = OSIF_PBDEF_WRITE_REQ;
    f_write_request.osif_block_size = sizeof( f_write_request );
    
    /*
     * Set the operation value to extend
     */
    f_write_request.osif_fadu_operation.value = OSIF_FADU_EXTEND;
    f_write_request.osif_fadu_operation.length = 
	sizeof( f_write_request.osif_fadu_operation.value );
    
    f_write_request.osif_fadu.fadu_number.value = 0;
    f_write_request.osif_fadu.fadu_number.length = 0;
    
    f_write_request.osif_fadu.fadu_ref_begin_end.value = 0;
    f_write_request.osif_fadu.fadu_ref_begin_end.length = 0;
    
    f_write_request.osif_fadu.fadu_ref_first_last.value = 
	OSIF_FADU_ID_FIRST;
    f_write_request.osif_fadu.fadu_ref_first_last.length = 
	sizeof( f_write_request.osif_fadu.fadu_ref_first_last.value );
    
    f_write_request.osif_fadu.fadu_ref_relative.value = 0;
    f_write_request.osif_fadu.fadu_ref_relative.length = 0;
    
    f_write_request.osif_fadu.name_list.value = 0;
    f_write_request.osif_fadu.name_list.length = 0;
    
    f_write_request.osif_fadu.single_name.value = 0;
    f_write_request.osif_fadu.single_name.length = 0;
    
    f_write_request.osif_fadu_lock.value = OSIF_FADU_LOCK_OFF;
    f_write_request.osif_fadu_lock.length = 
	sizeof( f_write_request.osif_fadu_lock.value );
    
    /*
     * Send the write request to the API
     */
    status = osif_send ( port_id, &f_write_request, &error_status );   
    if ( OSIF_SUCCESS != status )
	print_error_message("send_write: ", error_status);
    
    return( status );
}


send_data ( port_id )
/*
**++
**  FUNCTIONAL DESCRIPTION:
**
**     This routine builds the following primitives:
**             f_data_request
**
**     This is an unconfirmed service, so we don't expect a
**     response.
**
**  FORMAL PARAMETERS:
**
**	port_id
**
**  IMPLICIT INPUTS:
**
**	None
**
**  IMPLICIT OUTPUTS:
**
**	None
**
**  COMPLETION CODES:
**
**
**  SIDE EFFECTS:
**
**
**--
*/
unsigned                      port_id;
{
    unsigned                      status;
    unsigned                      error_status;
    struct osifpb                 f_data_request;
    
    printf("sending file data...\n");

    ZERO( f_data_request );
    
    /*
     * Set the data request function code in the parameter block
     */
    f_data_request.osif_block_type = OSIF_PBDEF_DATA_REQ;
    f_data_request.osif_block_size = sizeof( f_data_request );
    
    /*
     * data to be sent
     */
    f_data_request.osif_userdata.address = 
	(unsigned char *)"Hello, world\nHello, world\n"; 
    f_data_request.osif_userdata.length = 
	strlen( f_data_request.osif_userdata.address );
    
    /*
     * send the data to the API
     */
    status = osif_send ( port_id, &f_data_request, &error_status );   
    if ( OSIF_SUCCESS != status )
    {
	print_error_message("send_data: ", error_status);
    }
    
    return( status );
}



send_data_end ( port_id )
/*
**++
**  FUNCTIONAL DESCRIPTION:
**
**     This routine builds the following primitives:
**             f_data_end_request
**
**  FORMAL PARAMETERS:
**
**	port_id
**
**  IMPLICIT INPUTS:
**
**	None
**
**  IMPLICIT OUTPUTS:
**
**	None
**
**  COMPLETION CODES:
**
**
**  SIDE EFFECTS:
**
**
**--
*/
unsigned                      port_id;
{
    unsigned                      status;
    unsigned                      error_status;
    struct osifpb                 f_data_end_request;
    
    printf("end of sending file data...\n");
    
    ZERO( f_data_end_request );
    
    /*
     * Set the data end function code in the parameter block
     */
    f_data_end_request.osif_block_type = OSIF_PBDEF_DATA_END_REQ;
    f_data_end_request.osif_block_size = sizeof( f_data_end_request );
    
    /*
     * Set the Action result
     */
    f_data_end_request.osif_action_result.value = 0;
    f_data_end_request.osif_action_result.length = 
	sizeof( f_data_end_request.osif_action_result.value );
    
    /*
     * Send the data end request to the API.  This marks the end of
     * the data.
     */
    status = osif_send ( port_id, &f_data_end_request, &error_status );   
    
    if ( OSIF_SUCCESS != status )
	print_error_message("send data end: ", error_status);
    
}




send_transfer_end ( port_id )
/*
**++
**  FUNCTIONAL DESCRIPTION:
**
**     This routine builds the following primitives:
**             f_transfer_end_request
**
**     It receives the following FTAM events:
**             f_transfer_end_response
**
**  FORMAL PARAMETERS:
**
**      port_id
**
**  IMPLICIT INPUTS:
**
**	None
**
**  IMPLICIT OUTPUTS:
**
**	None
**
**  COMPLETION CODES:
**
**
**  SIDE EFFECTS:
**
**
**--
*/  
unsigned                      port_id;
{
    unsigned                      status;
    unsigned                      error_status;
    struct osifpb                 f_transfer_end_request;
    struct osifpb                 f_transfer_end_response;
    
    printf("end of data transfer...\n");

    ZERO( f_transfer_end_request );
    ZERO( f_transfer_end_response );
    
    /*
     * Set the transfer end function code in the parameter block
     */
    f_transfer_end_request.osif_block_type = OSIF_PBDEF_TRANSFER_END_REQ;
    f_transfer_end_request.osif_block_size = sizeof( f_transfer_end_request );
    
    /*
     * Set the Action result value
     */
    f_transfer_end_request.osif_action_result.value = 0;
    f_transfer_end_request.osif_action_result.length = 
	sizeof( f_transfer_end_request.osif_action_result.value );
    
    /*
     * Send the transfer end request to the API.  This marks the end
     * of the data transfer and ends the bulkd data transfer regime.
     */
    status = osif_send ( port_id, &f_transfer_end_request, &error_status );   
    if ( OSIF_SUCCESS != status )
    {
	print_error_message("send_transfer_end: ", error_status);
    }
    
    /*
     * Get responses from the FTAM responder
     */
    status = osif_get_event( port_id, &f_transfer_end_response, -1,
			    &error_status );
    if (status)
	print_error_message("osif_get_event: ", error_status);
    
    /*
     * check the state and action results
     * Display the state and action results if they are not successful
     * If a diagnostic was sent with this primitive then display the
     * contents of the diagnostic
     */
    diagnostic_checking("send_transfer_end: ", &f_transfer_end_response);
    
    
    if ( f_transfer_end_response.osif_returned_buffer )
    {
	status = osif_give_buffer(port_id, 
			  f_transfer_end_response.osif_returned_buffer,
			  &error_status );
	    if (status)
	    {
		print_error_message("osif_give_buffer: ", error_status);
		exit(1);
	    }
	
    }
}



/*
*  This structure is simply a list of parameter block type
*  values, and their text names.
*/

static struct
{
    int      item_value;
    char     *item_name;
}  pbnames[] =
{
    OSIF_PBDEF_P_ABORT,           "provider abort",
    OSIF_PBDEF_BG_REQ,            "begin group request",
    OSIF_PBDEF_BG_RSP,            "begin group response",
    OSIF_PBDEF_CANCEL_REQ,        "cancel request",
    OSIF_PBDEF_CHAT_REQ,          "change attribute request",
    OSIF_PBDEF_CHAT_RSP,          "change attribute response",
    OSIF_PBDEF_CRE_REQ,           "create request",
    OSIF_PBDEF_CRE_RSP,           "create response",
    OSIF_PBDEF_CLOSE_REQ,         "close request",
    OSIF_PBDEF_CLOSE_RSP,         "close response",
    OSIF_PBDEF_DATA_REQ,          "data request",
    OSIF_PBDEF_DATA_END_REQ,      "data end request",
    OSIF_PBDEF_DELETE_REQ,        "delete request",
    OSIF_PBDEF_DELETE_RSP,        "delete response",
    OSIF_PBDEF_DESELECT_REQ,      "deselect request",
    OSIF_PBDEF_DESELECT_RSP,      "deselect response",
    OSIF_PBDEF_EG_REQ,            "end group request",
    OSIF_PBDEF_EG_RSP,            "end group response",
    OSIF_PBDEF_INIT_REQ,          "initialize request",
    OSIF_PBDEF_INIT_RSP,          "initialize response",
    OSIF_PBDEF_NODE_DE_REQ,       "node descriptor data element",
    OSIF_PBDEF_OPEN_REQ,          "open request",
    OSIF_PBDEF_OPEN_RSP,          "open response",
    OSIF_PBDEF_RAT_REQ,           "read attribute request",
    OSIF_PBDEF_RAT_RSP,           "read attribute response",
    OSIF_PBDEF_READ_REQ,          "read request",
    OSIF_PBDEF_SEL_REQ,           "select request",
    OSIF_PBDEF_SEL_RSP,           "select response",
    OSIF_PBDEF_TERM_REQ,          "terminate request",
    OSIF_PBDEF_TERM_RSP,          "terminate response",
    OSIF_PBDEF_TRANSFER_END_REQ,  "transfer end request",
    OSIF_PBDEF_TRANSFER_END_RSP,  "transfer end response",
    OSIF_PBDEF_WRITE_REQ,         "write request",
    -1,                           "unknown parameter block type"
    };



print_pbname( txt, event )
/*
**++
**  FUNCTIONAL DESCRIPTION:
**
**     This routine simply display a text message and
**     a parameter block name.
**
**
**  FORMAL PARAMETERS:
**
**     text
**     event
**
**  IMPLICIT INPUTS:
**
**	None
**
**  IMPLICIT OUTPUTS:
**
**	None
**
**  COMPLETION CODES:
**
**
**  SIDE EFFECTS:
**
**
**--
*/  
char  *txt;       /* Input text pointer         */
int   event;      /* Input parameter block type */
{
    int   i;      /* Temp for loop control      */
    
    for (i=0; (pbnames[i].item_value >= 0 ) &&
	 (pbnames[i].item_value != event); i++)   ;
    
    printf(  "%-20s: %s (%d)\n",
		   txt,
		   pbnames[i].item_name,
		   event   );
}





/*
  This structure is simply a list of parameter block type
  values, and their text names.
  */

static struct
{
    int      item_value;
    char     *item_name;
}  error_names[] =
{
    0,     "No Reason ",
    1,     "Responder error ",
    2,	   "System shutdown ",
    3,     "FTAM management problem (unspecific) ",
    4,     "FTAM management, bad account ",
    5,     "FTAM management, security not passed ",
    6,     "Delay may be encountered ",
    7,     "Initiator Error, unspecific ",
    8,     "Subsequent Error ",
    9,     "Temporal insufficiency of resources",
    10,    "Access request violates VFS security",
    11,    "Access request violates local security",
    
    /* Protocol and supporting service related diagnostic definitions */
    1000,  "Conflicting parameter values ",
    1001,  "Unsuported parameter values ",
    1002,  "Mandatory parameter not set ",
    1003,  "Unsupported parameter ",
    1004,  "Duplicated parameter ",
    1005,  "Illegal parameter type ",
    1006,  "Unsupported parameter types ",
    1007,  "FTAM protocol error (unspecific) ",
    1008,  "FTAM protocol error, procedure error ",
    1009,  "FTAM protocol error, functional unit error ",
    1010,  "FTAM protocol error, corruption error ",
    1011,  "Lower layer failure ",
    1012,  "Lower layer addressing error ",
    1013,  "Timeout",
    1014,  "System shutdown ",
    1015,  "Illegal grouping sequence ",
    1016,  "Grouping threshold violation ",
    1017,  "Specific PDU request inconsistent with current requested access",
    
    /* Association related diagnostics */
    2000,  "Association with user not allowed ",
    2001,  "(not assigned) ",
    2002,  "Unsupported service class ",
    2003,  "Unsupported functional unit ",
    2004,  "Attribute group error (unspecific) ",
    2005,  "Attribute group not supported ",
    2006,  "Attribute group not allowed ",
    2007,  "Bad account ",
    2008,  "Association management (unspecifc) ",
    2009,  "Association management - bad address ",
    2010,  "Association management - bad acount ",
    2011,  "Checkpoint window error - too large ",
    2012,  "Checkpoint window error - too small ",
    2013,  "Checkpoint window unsupported ",
    2014,  "Communications QoS not supported ",
    2015,  "Initiator Identity unacceptable ",
    2016,  "Context management refused ",
    2017,  "Rollback not available ",
    2018,  "Contents-type list cut by responder ",
    2019,  "Contests-type list cut by presentation ",
    2020,  "Invalid filestore password ",
    2021,  "Incompatible service classes ",
    
    /* Selection related diagnostic definitions */
    3000,  "File not found ",
    3001,  "Selection attributes not matched ",
    3002,  "Initial attributes not possible  ",
    3003,  "Bad atrribute name ",
    3004,  "Nonexistent file ",
    3005,  "File already exists ",
    3006,  "File cannot be created ",
    3007,  "File cannot be deleted ",
    3008,  "Concurrency control not available ",
    3009,  "Concurrency control not supported ",
    3010,  "Concurrency control not possible ",
    3011,  "More restrictive lock	   ",
    3012,  "File busy ",
    3013,  "File not available  ",
    3014,  "Access control not available ",
    3015,  "Access control not supported ",
    3016,  "Access control inconsistent ",
    3017,  "File name truncated ",
    3018,  "Initial attributes altered ",
    3019,  "Bad account ",
    3020,  "Override selected existing file ",
    3021,  "Override recreated file with old attributes ",
    3022,  "Override recreated file with new attributes",
    3023,  "Override not possible	",
    3024,  "Ambiguous file specification ",
    3025,  "Invalid create password ",
    3026,  "Invalid delete password ",
    3027,  "Bad attribute value ",
    3028,  "Requested access violates permitted actions",
    3029,  "Function unit not available for requested access ",
    3030,  "File was created, not selected ",
    
    /* File management realted diagnostic definitions */
    
    4000,  "Attribute non-existent ",
    4001,  "Attribute cannot be read ",
    4002,  "Attribute cannot be changed ",
    4003,  "Attribute not supported ",
    4004,  "Bad attribute name ",
    4005,  "Bad attribute value ",
    4006,  "Attribute partialy supported ",
    4007,  "Additional set attribute value not distinct ",
    
    /* Access related diagnostic definitions */
    
    5000,  "Bad FADU (unspecific) ",
    5001,  "Bad FADU - size error	",
    5002,  "Bad FADU - type error	",
    5003,  "Bad FADU - poorly specified ",
    5004,  "Bad FADU - bad location  ",
    5005,  "FADU does not exist	",
    5006,  "FADU not available ",
    5007,  "FADU not available for reading ",
    5008,  "FADU not available for writing ",
    5009,  "FADU not available for location ",
    5010,  "FADU not available for erasure ",
    5011,  "FADU cannot be inserted   ",
    5012,  "FADU cannot be replaced   ",
    5013,  "FADU cannot be located    ",
    5014,  "Bad data element type	  ",
    5015,  "Operation not available  ",
    5016,  "Operation not supported  ",
    5017,  "Operation inconsistent  ",
    5018,  "Concurrency control not available ",
    5019,  "Concurrency control not supaported ",
    5020,  "Concurrency control inconsistent ",
    5021,  "Processing mode not available ",
    5022,  "Processing mode not supported ",
    5023,  "Processing mode inconsistent ",
    5024,  "Access context not available ",
    5025,  "Access context not supported ",
    5026,  "Bad write (unspecific) ",
    5027,  "Bad read  (unspecific) ",
    5028,  "Local error (unspecific) ",
    5029,  "Local error - filespace exhausted  ",
    5030,  "Local error - data corrupted ",
    5031,  "Local error - device failure ",
    5032,  "Future filesize exceeded ",
    5033,  "                         ",
    5034,  "Future filesize increased  ",
    5035,  "Functional unit invalid in processing mode ",
    5036,  "Contents-type incosistent   ",
    5037,  "Contents-type simplified   ",
    5038,  "Duplicate FADU name  ",
    5039,  "Damage to select/open regime	",
    5040,  "FADU locking not available on file  ",
    5041,  "FADU locked by another user  ",
    
    /* Recovery related diagnostics */
    6000,  "Bad checkpoint (unspecific) ",
    6001,  "Activity not unique	",
    6002,  "Checkpoint outside of window	",
    6003,  "Activity no longer exists ",
    6004,  "Activity not recognized ",
    6005,  "No docket  ",
    6006,  "Corrupt docket ",
    6007,  "File waiting restart ",
    6008,  "Bad recovery point ",
    6009,  "Non-existent recovery point ",
    6010,  "Recovery mode not available ",
    6011,  "Recovery mode inconsistent ",
    6012,  "Recovery mode reduced	",
    6013,  "Access control not available ",
    6014,  "Access control not supported ",
    6015,  "Access control inconsistent ",
    6016,  "Contents type inconsistent ",
    6017,  "Contents type simplified ",
    
    9000,  "OSIF_BAD_ACCCNTX ",
    9001,  "OSIF_BAD_ACCCNTRL ",
    9002,  "OSIF_BAD_ACCPWD ",
    9003,  "OSIF_BAD_ACCOUNT ",
    9004,  "OSIF_BAD_ACTRES ",
    9005,  "OSIF_BAD_ACTID ",
    9006,  "OSIF_BAD_APPCNTX ",
    9007,  "OSIF_BAD_ARCLEN ",
    9008,  "OSIF_BAD_ATTRGRP ",
    9009,  "OSIF_BAD_ATTNAME ",
    9010,  "OSIF_BAD_CHARGE ",
    9011,  "OSIF_BAD_CHKPWIN ",
    9012,  "OSIF_BAD_CCCNTRL ",
    9013,  "OSIF_BAD_CNTTYPE  ",
    9014,  "OSIF_BAD_CNTTYLST ",
    9015,  "OSIF_BAD_CREPWD ",
    9016,  "OSIF_BAD_DTCRE ",          /* Date time creation */
    9017,  "OSIF_BAD_DTLATMD ",	       /* Date time last att mod */
    9018,  "OSIF_BAD_DTLMOD  ",        /* Date time last mod */
    9019,  "OSIF_BAD_DTLSTRD ",
    9020,  "OSIF_BAD_DELPWD  ",
    9021,  "OSIF_BAD_DELVAL  ",
    9022,  "OSIF_BAD_DELCNTX ",
    9023,  "OSIF_BAD_DIAG    ",
    9025,  "OSIF_BAD_FADU  ",
    9026,  "OSIF_BAD_FADULK ",
    9027,  "OSIF_BAD_FADULKG ",
    9028,  "OSIF_BAD_FADUOP ",
    9029,  "OSIF_BAD_FILENM   ",
    9030,  "OSIF_BAD_FILESZ  ",
    9031,  "OSIF_BAD_FILEAV  ",
    9032,  "OSIF_BAD_FSPWD  ",
    9033,  "OSIF_BAD_FQOS   ",
    9034,  "OSIF_BAD_FUNITS   ",	/* Functional Units */
    9035,  "OSIF_BAD_FUTFISZ   ",
    9036,  "OSIF_BAD_INITID   ",	
    9037,  "OSIF_BAD_IDCRE   ",
    9038,  "OSIF_BAD_IDLATMD  ",
    9039,  "OSIF_BAD_IDLMOD   ",
    9040,  "OSIF_BAD_IDLREAD   ",
    9041,  "OSIF_BAD_IMPINFO ",         /* Implementatin Information */
    9042,  "OSIF_BAD_INVAL  ",
    9043,  "OSIF_BAD_LAEQUAL ",
    9044,  "OSIF_BAD_LAPTITLE   ",
    9045,  "OSIF_BAD_LPADDR    ", 	/* Local Presenation Address */
    9046,  "OSIF_BAD_LEQUAL   ",        /* legal qualifications */
    9047,  "OSIF_BAD_OVRRIDE   ",
    9048,  "OSIF_BAD_RAEQUAL   ",
    9049,  "OSIF_BAD_RAPTITLE   ",
    9050,  "OSIF_BAD_RPADDR  ",         /* Remote Presentation Address */
    9051,  "OSIF_BAD_PERACT  ",
    9052,  "OSIF_BAD_PCTXMGT  ",
    9053,  "OSIF_BAD_PROMODE  ",
    9055,  "OSIF_BAD_PROTID   ",
    9056,  "OSIF_BAD_REMCNTX  ",
    9057,  "OSIF_BAD_REQACC  ",
    9058,  "OSIF_BAD_RECMODE ",
    9059,  "OSIF_BAD_SRVCLASS  ",
    9060,  "OSIF_BAD_STRES ", 		/* State Result */
    9061,  "OSIF_BAD_STOACC ",		/* Storage Account */
    9062,  "OSIF_BAD_THRES ",
    9063,  "OSIF_BAD_USRDATA  ",
    9065,  "OSIF_BAD_CHATPWD  ",
    9067,  "OSIF_BAD_ERAPWD   ",
    9068,  "OSIF_BAD_EXTPWD   ",
    9069,  "OSIF_BAD_INSPWD   ",
    9070,  "OSIF_BAD_RDATPWD  ",
    9071,  "OSIF_BAD_RDPWD    ",
    9072,  "OSIF_BAD_RPLPWD   ",
    10000, "OSIF_BADITEMSIZE ",
    10002, "OSIF_PROTOCOL_ERROR",
    10003, "OSIF_INVPORT",
    10004, "OSIF_NOPORT",
    10005, "OSIF_PARAMNOWRT ",
    10006, "OSIF_PARAMNORD ",
    10007, "OSIF_NOBUFFS ",
    10008, "OSIF_NOMEM ",
    10009, "OSIF_XPORTFAILURE",
    10010, "OSIF_NO_EVENT",
    -1,"Error not specified"
    };


print_error_message( txt, event )
/*
**++
**  FUNCTIONAL DESCRIPTION:
**
**     This routine simply display a text message and
**     an error message.
**
**
**  FORMAL PARAMETERS:
**
**     text
**     event
**
**  IMPLICIT INPUTS:
**
**	None
**
**  IMPLICIT OUTPUTS:
**
**	None
**
**  COMPLETION CODES:
**
**
**  SIDE EFFECTS:
**
**
**--
*/  
char  *txt;       /* Input text pointer         */
int   event;      /* Input parameter block type */
{
    int   i;       /* Temp for loop control      */
    
    for (i=0; (error_names[i].item_value >= 0 ) &&
	 (error_names[i].item_value != event); i++)   ;
    
    printf(  "ERROR: %s: %s\n",
		   txt,
		   error_names[i].item_name   );
}






diagnostic_checking( intxt, r_pb )
/*
**++
**  FUNCTIONAL DESCRIPTION:
**
**     If either the action or state results or the diagnostics
**     are not zero, then the parameter block where the error 
**     occured is printed out, and the contents are displayed.
**
**  FORMAL PARAMETERS:
**
**     intxt
**     r_pb
**
**  IMPLICIT INPUTS:
**
**	None
**
**  IMPLICIT OUTPUTS:
**
**	None
**
**  COMPLETION CODES:
**
**
**  SIDE EFFECTS:
**
**
**--
*/  
char                       *intxt;
struct osifpb              *r_pb;
{
    char                       *txt;
    struct osif_diagnostics_pb *te;     /* Place holder for this entry   */
    
    txt = (intxt)?intxt:"";
    
    if (( r_pb->osif_action_result.value != 0 ) || 
		( r_pb->osif_state_result.value != 0) ||
		( r_pb->osif_diagnostic != 0 ) )
	{
	print_pbname( txt, r_pb->osif_block_type  );
	}
    
    if ( r_pb->osif_action_result.value != 0 )
	printf( "  osif_action_result = %d\n", 
		r_pb->osif_action_result.value );

    if ( r_pb->osif_state_result.value != 0)
	printf( "  osif_state_result = %d\n",  
		r_pb->osif_state_result.value );
    
    if ( r_pb->osif_diagnostic != 0 )
    {
	for ( te = r_pb->osif_diagnostic;
	     (0 != te) ;
	     te = te->next )
	{
	    printf( "  osif_diagnostic structure:\n" );
	
	    if ( te->diagnostic_type.length )
            {
		printf(  "   %-20s ", "diagnostic_type" );
		printf( "%d\n", te->diagnostic_type.value );
            }
	    
	    if ( te->error_identifier.length )
            {
		printf(  "   %-20s ", "error_identifier" );
		printf( "%d\n", te->error_identifier.value );
            }
	    if ( te->error_observer.length )
            {
		printf(  "   %-20s ", "error_observer" );
		printf( "%d\n", te->error_observer.value );
            }
	    
	    if ( te->error_source.length )
            {
		printf(  "   %-20s ", "error_source" );
		printf( "%d\n", te->error_source.value );
            }
	    
	    if ( te->suggested_delay.length )
            {
		printf(  "   %-20s ", "suggested_delay" );
		printf( "%d\n", te->suggested_delay.value );
            }
	    
	    if ( te->further_details.length )
            {
		printf(  "   %-20s ", "further_details" );
		printf( "%0.*s\n",     te->further_details.length,
		       te->further_details.address );
		
            }
	}
    }
}
