/*
 *  gen_msghlp.c
 *
 *  Lon S Hilde
 *  Written 18-jan-1994
 *  Ruslan R. Laishev
 *  Some changes at 20-oct-1999
 *
 *  This tool translates a GNM file into an MSGHLP file, allowing you to
 *  have a common source file that generates output suitable for the
 *  HELP/MESSAGE facility.  Used in conjunction with GEN_MSG.
 *
 *  No claims to style or substance.
 *
 */
#include <ctype.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#define MAX_LINE_LENGTH 512
#define GENERAL_LINE_LIMIT 30

#define MSG_NAME_ARRAY_SIZE 20480

#define BLANK_LINE "\n"
#define MSGHLP_MSG_TEXT_1 "1"
#define MSGHLP_MSG_TEXT_3 "3"
#define MSGHLP_MSG_TEXT_4 "4"

typedef struct n_and_p {
    char *name;
    fpos_t fptr;
} name_and_fpos;

int copy_substr_but_single_space (char in_string[], char out_string[]);
int find_next_word_index (char in_string[]);
int copy_str_but_no_indent (char in_string[], char out_string[]);
int names_compare (const void *, const void *);


main (int argc, char *argv[])
{
    FILE *ostream, *istream;
    char *prog = argv[0], *infile = argv[1], *outfile = argv[2], *status;
    char buf1[MAX_LINE_LENGTH], buf2[MAX_LINE_LENGTH], buf3[MAX_LINE_LENGTH];
    char *facility;
    int l, index1, index2, failsafe;
    name_and_fpos *names[MSG_NAME_ARRAY_SIZE];
    int names_index, names_count, name_length, changed;
    fpos_t fptr;

    if ( argv[3] )
	facility = argv[3];
    else
	facility = "Generic";


    /* open the files we need for input and output
     */
    istream = fopen (infile, "r");
    if (istream == NULL)
    {
	fprintf (stderr, "%s error, failed to open %s\n", prog, infile);
	exit (1);
    }

    ostream = fopen (outfile, "w");
    if (ostream == NULL)
    {
	fprintf (stderr, "%s error, failed to open %s\n", prog, outfile);
	exit (2);
    }

    /* I'm going to walk through the input file and build name
     * structures in dymnamic memory.  This will allow me to sort them
     * before outputing the whole msghlp message text to the output file.
     */
    names_index = 0;
    while (fgets (&buf1[0], MAX_LINE_LENGTH, istream) == &buf1[0])
    {
	if ((strncmp (&buf1[0], ".NAME", 5) == 0) ||
	    (strncmp (&buf1[0], ".name", 5) == 0))
	{
	    /* allocate space for another msghlp_message structure
             */
	    names[names_index] = malloc (sizeof (*names[0]));

	   /* find the actual name text and move it into memory (and into
	    * the a name array element).
	    */
	    index1 = find_next_word_index (&buf1[0]);
	    name_length = strlen (&buf1[index1]);
	    names[names_index]->name = malloc (name_length+1);
	    strcpy (names[names_index]->name, &buf1[index1]);

	    /* Also record the file position.  We will use that later when
	     * we go after the message explanation and user_action text.
	     */
	    if (fgetpos (istream, &(names[names_index]->fptr)) != 0)
	    {
		fprintf (stderr, "%s error, unable to get current istream file position\n", prog);
		exit (3);
	    }

	    names_index++;
	}
    }
    names_count = names_index;

    /* Okay, let's sort the names array.
     */
    qsort ((char *)names, names_count, sizeof (names[0]), names_compare);

    /* Okay, let's build the output file.
     *
     * First, put in the header.
     */
     
    /* Then, build one message at a time and line by line.
     */
    for (names_index = 0; names_index < names_count; names_index++)
    {
	/* start with the first standard MSG_TEXT and the NAME
         */
	index2 = copy_substr_but_single_space (MSGHLP_MSG_TEXT_1, &buf2[0]);
	l = copy_substr_but_single_space (names[names_index]->name, &buf2[index2]);
	index2 = index2 + l;
	buf2 [index2++] = ',';

	/* set the file position to the line we left off on when we found the name.
	 * This should be the MESSAGE lines.
	 */
	if (fsetpos (istream, &(names[names_index]->fptr)) != 0)
	{
	    fprintf (stderr, "%s error, unable to set current istream file position\n", prog);
	    exit (5);
	}

	/* tack on the MESSAGE text lines to the NAME text
	 */
	failsafe = 0;
	while (failsafe < GENERAL_LINE_LIMIT)
	{
	    if ((status = fgets (&buf1[0], MAX_LINE_LENGTH, istream)) == NULL)
	    {
		buf2[index2++] = '\0';
		fprintf (stderr, "%s error, message not terminated, %s\n", prog, &buf2[0]);
		exit (6);
	    }
	    if ((strncmp (&buf1[0], ".MESSAGE", 8) == 0) ||
		(strncmp (&buf1[0], ".message", 8) == 0))
		index1 = find_next_word_index (&buf1[0]);
	    else
	    {
		if ((buf1[0] == '.') ||
		    (buf1[0] == '!'))
		    break;
		else
		    index1 = 0;
	    }
	    l = copy_str_but_no_indent (&buf1[index1], &buf2[index2]);
	    index2 = index2 + l;
	    while (isgraph (buf2[--index2]) == 0);
	    buf2[++index2] = ' ';
	    buf2[++index2] = '\n';
	    index2++;
	    failsafe++;
	}
	if (failsafe >= GENERAL_LINE_LIMIT)
	{
	    buf2[index2++] = '\0';
	    fprintf (stderr, "%s error, MESSAGE not terminated, %s\n", prog, &buf2[0]);
	    exit (7);
	}
	while (isgraph (buf2[--index2]) == 0);
	buf2[++index2] = ' ';
	buf2[++index2] = '\n';
	buf2[++index2] = '\0';

	/* and, finally, output it
	 */
	if (fputs (&buf2[0], ostream) == EOF)
	{
	    fprintf (stderr, "%s error, write failure to %s\n", prog, outfile);
	    exit (8);
	}


	if (0 > fprintf(ostream,"2%s\n",facility) )
	{
	    fprintf (stderr, "%s error, write failure to %s\n", prog, outfile);
	    exit (8);
	}


	/* Now start the EXPLANATION lines, if there are any.
	 */
	if ((strncmp (&buf1[0], ".EXPLANATION", 12) == 0) ||
	    (strncmp (&buf1[0], ".explanation", 12) == 0))
	{
	    if (fputs (MSGHLP_MSG_TEXT_3, ostream) == EOF)
	    {
		fprintf (stderr, "%s error, write failure to %s\n", prog, outfile);
		exit (11);
	    }
	    index1 = find_next_word_index (&buf1[0]);
	    index1 = copy_str_but_no_indent (&buf1[index1], &buf2[0]);

	    /* skip blank lines
	     */
	    if (strcmp (BLANK_LINE, &buf2[0]) != 0)
	    {
		if (fputs (&buf2[0], ostream) == EOF)
		{
		    fprintf (stderr, "%s error, write failure to %s\n", prog, outfile);
		    exit (12);
		}
	    }
	    failsafe = 0;
	    while (failsafe < GENERAL_LINE_LIMIT)
	    {
		if ((status = fgets (&buf1[0], MAX_LINE_LENGTH, istream)) == NULL)
		{
		    buf2[index2++] = '\0';
		    fprintf (stderr, "%s error, message not terminated, %s\n", prog, &buf2[0]);
		    exit (13);
		}
		if ((buf1[0] == '.') ||
		    (buf1[0] == '!'))
		    break;
		index1 = copy_str_but_no_indent (&buf1[0], &buf2[0]);

		/* skip blank lines
	         */
		if (strcmp (BLANK_LINE, &buf2[0]) != 0)
		{
		    if (0 > fprintf(ostream,"3%s",buf2) )
		    {
			fprintf (stderr, "%s error, write failure to %s\n", prog, outfile);
			exit (14);
		    }
		}
		failsafe++;
	    }
	    if (failsafe >= GENERAL_LINE_LIMIT)
	    {
		fprintf (stderr, "%s error, EXPLANATION not terminated, %s\n", prog, &buf2[0]);
		exit (15);
	    }
	}

	/* Finally, the USER_ACTION, if any.
         */
	if ((strncmp (&buf1[0], ".USER_ACTION", 12) == 0) ||
	    (strncmp (&buf1[0], ".user_action", 12) == 0))
	{
	    if (fputs (MSGHLP_MSG_TEXT_4, ostream) == EOF)
	    {
		fprintf (stderr, "%s error, write failure to %s\n", prog, outfile);
		exit (16);
	    }
	    index1 = find_next_word_index (&buf1[0]);
	    index1 = copy_str_but_no_indent (&buf1[index1], &buf2[0]);
	    if (strcmp (BLANK_LINE, &buf2[0]) != 0)
	    {
		if (fputs (&buf2[0], ostream) == EOF)
		{
		    fprintf (stderr, "%s error, write failure to %s\n", prog, outfile);
		    exit (17);
		}
	    }
	    failsafe = 0;
	    while (failsafe < GENERAL_LINE_LIMIT)
	    {
		if ((status = fgets (&buf1[0], MAX_LINE_LENGTH, istream)) == NULL)
		{
		    buf2[index2++] = '\0';
		    fprintf (stderr, "%s error, message not terminated, %s\n", prog, &buf2[0]);
		    exit (18);
		}
		if ((buf1[0] == '.') ||
		    (buf1[0] == '!'))
		    break;
		index1 = copy_str_but_no_indent (&buf1[0], &buf2[0]);

		/* skip blank lines
	         */
		if (strcmp (BLANK_LINE, &buf2[0]) != 0)
		{
		    if (0 > fprintf(ostream,"4%s",buf2) )
		    {
			fprintf (stderr, "%s error, write failure to %s\n", prog, outfile);
			exit (19);
		    }
		}
		failsafe++;
	    }
	    if (failsafe >= GENERAL_LINE_LIMIT)
	    {
		fprintf (stderr, "%s error, EXPLANATION not terminated, %s\n", prog, &buf2[0]);
		exit (20);
	     }
	}

	/* One final nit.  End with a blank line.
	 */
	if (fputs (BLANK_LINE, ostream) == EOF)
	{
	    fprintf (stderr, "%s error, write failure to %s\n", prog, outfile);
	    exit (23);
	}
    }    /* for names loop */

    /* Finally, terminate the file with the END text and close the output file.
     */
    if (fclose (ostream) == EOF)
    {
	fprintf (stderr, "%s error, closure failure to %s\n", prog, outfile);
	exit (22);
    }
    exit (0);
}

int copy_substr_but_single_space (char in_string[], char out_string[])
{
    int i = 0, j = 0;

    while (in_string[i] != '\0')
    {
	if (in_string[i] == '\f')
	    i++;
	else if (isspace (in_string[i]))
	{
	    out_string[j++] = ' ';
	    while ((isspace (in_string[i])) &&
		   (in_string[i] != '\0'))
		i++;
	}
	else
	    out_string[j++] = in_string[i++];
    }
    if (out_string[j-1] == ' ')
	j--;

    return j;
}

int find_next_word_index (char in_string[])
{
    int i = 0;

    while (!(isspace (in_string[i])) &&
	   (in_string[i] != '\0'))
	i++;

    while ((isspace (in_string[i])) &&
	   (in_string[i] != '\n') &&
	   (in_string[i] != '\0'))
	i++;

   return i;
}

int copy_str_but_no_indent (char in_string[], char out_string[])
{
    int i = 0, j = 0;

    while (((in_string[i] == ' ') ||
	    (in_string[i] == '\t')) &&
	   (in_string[i] != '\0'))
	i++;
 
    while (in_string[i] != '\0')
    {
	if (in_string[i] == '\f')
	    i++;
	else if (in_string[i] == '\t')
	{
	    out_string[j++] = ' ';
	    i++;
	}
	else
	    out_string[j++] = in_string[i++];
    }
    out_string[j] = '\0';

    return j;
}

int names_compare (const void *ptr1, const void *ptr2)
{
    int status;
    char *name1, *name2;

    name1 = (*(name_and_fpos **)(ptr1))->name;
    name2 = (*(name_and_fpos **)(ptr2))->name;

    return strcmp (name1, name2);
}
