/*  C_ZILL_MOST_REPORT_SUB.C
 *
 *  Subroutine to create a report of the most
 *  frequently hit numbers.
 *
 */
/*
 *      ANSI headers
 */
#include <stdlib>
#include <stdio>
#include <string>
#include <time>


/*
 *      VMS headers
 */
#include <starlet>
#include <rms>
#include <stsdef>
#include <descrip>
#include <lib$routines>

/*
 *  Our headers
 */
#pragma member_alignment save
#pragma nomember_alignment
#pragma dictionary "cdd_records.zillionare_stats_record"
#pragma member_alignment restore

#define LENGTH(descriptor)      descriptor.dsc$w_length


/*;;;;;
 *  Program constants and global data
 *;;;;;
 */
const char *DRAWING_STATS_NAME = "DRAWING_STATS";
const char *MEGA_STATS_NAME = "MEGA_STATS";
const char *MOST_REPORT_FILE_NAME = "ZILL_MOST.RPT";
const char *STAT_DEFAULT_NAME = ".IDX";

const int ELM_COUNT = 52;


/*;;;;;
 *      Function prototypes
 *;;;;;
 */

void build_stats_fab_rab( struct zillionare_stats_record *m_z, 
                         struct FAB *m_fab, struct RAB *m_rab,
                         struct XABKEY *m_xab,
                         int access,
                         const char *file_name);

void c_fill_in_logicals( void);

int desc_int_cmp( const void *e1, const void *e2);

void load_stat_array(    struct zillionare_stats_record *m_z, 
                         struct FAB *m_fab, struct RAB *m_rab,
                         struct XABKEY *m_xab,
                         int access,
                         const char *file_name);

int  open_stats_idx( struct zillionare_stats_record *m_z, 
                    struct FAB *m_fab, struct RAB *m_rab,
                    struct XABKEY *m_xab,
                    int access,
                    const char *file_name);

void write_most_report( struct zillionare_stats_record *d_stats,
                    struct zillionare_stats_record *m_stats,
                    FILE *rpt_file);

/*;;;;;
 *      Functions and subroutines
 *;;;;;
 */
void c_zill_most_report_sub( )
{
    int                     l_x, default_operations;
    char                    command_str[255];
    FILE                    *rpt_file;

    $DESCRIPTOR( command_str_desc, command_str);


    struct  FAB     mega_fab;
    struct  RAB     mega_rab;
    struct  XABKEY  mega_xab;

    //
    //  Arrays to hold stats records
    //
    struct zillionare_stats_record  d_stats[ ELM_COUNT];
    struct zillionare_stats_record  m_stats[ ELM_COUNT];

    c_fill_in_logicals();

    rpt_file = fopen( MOST_REPORT_FILE_NAME, "w", 
                      "ctx=rec","mrs=80", "rat=cr", "rfm=var");

    //
    //  Populate our working arrays.
    //
    load_stat_array( d_stats, &mega_fab, &mega_rab,
                        &mega_xab, FAB$M_GET,
                        DRAWING_STATS_NAME);

    load_stat_array( m_stats, &mega_fab, &mega_rab,
                        &mega_xab, FAB$M_GET,
                        MEGA_STATS_NAME);


    //
    //  Sort our arrays in descending order by hit count
    //
    qsort( d_stats, 
           ELM_COUNT, 
           sizeof( struct zillionare_stats_record),
           desc_int_cmp);

    qsort(  m_stats, 
            ELM_COUNT, 
            sizeof( struct zillionare_stats_record),
            desc_int_cmp);


    //
    //  Print the report
    //
    write_most_report( d_stats, m_stats, rpt_file);

    fclose( rpt_file);

    sprintf( command_str, "EDIT/READ %s", MOST_REPORT_FILE_NAME);
    LENGTH( command_str_desc) = strlen( command_str);

    l_x = lib$spawn( &command_str_desc);

}  /* end c_zill_most_report_sub */

/*;;;;;
 *  Subroutine to build the stats fab, rab, and xab structures
 *;;;;;
 */
void build_stats_fab_rab( struct zillionare_stats_record *m_stat, 
                             struct FAB *m_fab, struct RAB *m_rab,
                             struct XABKEY *m_xab, int access,
                             const char *file_name)
{
    *m_fab = cc$rms_fab;                            // fill in defaults for FAB
    m_fab->fab$l_fna = (char *)file_name;           // Pointer to file name
    m_fab->fab$b_fns = strlen(file_name);           // length of name
    m_fab->fab$l_dna = (char *)STAT_DEFAULT_NAME;   // default name 
                                                    //   really just file type
    m_fab->fab$b_dns = strlen( STAT_DEFAULT_NAME);  // length of default name
    m_fab->fab$b_org = FAB$C_IDX;                   // indexed file
    m_fab->fab$b_fac = access;                      // get, put, etc.
    m_fab->fab$w_mrs = sizeof( struct zillionare_stats_record);  
                                                    // maximum record size
    m_fab->fab$b_rat = FAB$M_FTN;                   // FORTRAN formatted records
    m_fab->fab$b_rfm = FAB$C_FIX;                   // fixed length records
    m_fab->fab$b_shr = FAB$M_NIL;                   // no file sharing
    m_fab->fab$l_xab = (void *) m_xab;              // index block address


    *m_xab = cc$rms_xabkey;             // fill in defaults for XAB
    m_xab->xab$b_dtp = XAB$C_IN4;       // signed 31 bit integer.
    m_xab->xab$b_ref = 0;               // primary key is zero
    m_xab->xab$b_nsg = 1;               // single segment key
    m_xab->xab$w_pos = 0;               // starting position zero in the record
    m_xab->xab$b_siz = 4;               // our key is 4 bytes long
    m_xab->xab$b_prolog = XAB$C_PRG3;   // prolog 3
    m_xab->xab$l_nxt = 0;               // end of chain
                                        //   we only have one key on this file.

    *m_rab = cc$rms_rab;                // fill in defaults for RAB
    m_rab->rab$l_fab = m_fab;           // pointer to FAB
    m_rab->rab$b_rac = RAB$C_KEY;       // keyed access
    m_rab->rab$b_krf = 0;               // access via primary key
    m_rab->rab$l_kbf = (void *)&m_stat->elm_no;  // key buffer
    m_rab->rab$l_ubf = (char *)m_stat;  // data buffer for record

    m_rab->rab$w_usz = sizeof( struct zillionare_stats_record);

}  /* end build_stats_fab_rab subroutine */

/*;;;;;
 *  function to be used with qsort as the compare.
 *  Normally a sort compare routine returns the following:
 *      -1  e1 < e2
 *      0   e1 = e2
 *      1   e1 > e2
 *  We need the opposite to create descending sort.
 *;;;;;
 */
int desc_int_cmp( const void *e1, const void *e2)
{
    int     ret_val;

    struct zillionare_stats_record *z_1, *z_2;

    z_1 = (struct zillionare_stats_record *)e1;
    z_2 = (struct zillionare_stats_record *)e2;


    ret_val = 0;

    if ( z_1->hit_count < z_2->hit_count)
    {
        ret_val = 1;
    }
    else 
    {
      if ( z_1->hit_count > z_2->hit_count)
            ret_val = -1;
    }

    return(ret_val);

}  /* end desc_int_cmp function */

/*;;;;;
 *  Subroutine to load all of the stats records into a working
 *  array.
 *;;;;;
 */
void load_stat_array(    struct zillionare_stats_record *m_stat, 
                         struct FAB *m_fab, struct RAB *m_rab,
                         struct XABKEY *m_xab,
                         int access,
                         const char *file_name)
{
    int     l_x, l_sub;

    //
    //  load the drawing stats
    //
    l_x = open_stats_idx( m_stat, m_fab, 
                         m_rab, m_xab, FAB$M_GET,
                         file_name);

    if ( !(l_x & 1))
    {
        puts( "Unable to open drawing stats file");
        return;
    }  /* end test for failed open */


    //
    //  Empty array
    //
    memset( m_stat, 0, sizeof( struct zillionare_stats_record) * ELM_COUNT);

    //
    //  turn off all keyed access flags
    //
    m_rab->rab$l_rop = RAB$M_WAT | RAB$M_RLK;  
    m_rab->rab$b_rac = RAB$C_SEQ;  

    l_sub = 0;

    while (l_x & 1  &&  l_sub < ELM_COUNT)
    {
        m_rab->rab$l_ubf = (void *)&m_stat[l_sub];

        l_x = sys$get( m_rab);
        l_sub++;

    }  /* end while l_x loop */

    l_x = sys$close( m_fab);

}  /* end load_drawing_stats subroutine */

/*;;;;;
 *  Function to open the statistics file
 *;;;;;
 */
int  open_stats_idx( struct zillionare_stats_record *m_stat, 
                    struct FAB *m_fab, struct RAB *m_rab,
                    struct XABKEY *m_xab, int access,
                    const char *file_name)
{
    int         l_x;


    build_stats_fab_rab( m_stat, m_fab, m_rab, m_xab, access, file_name);

    //
    //      See if we can open the file.
    //
    l_x = sys$open( m_fab);

    if (!( l_x & 1))
    {
        puts( "Unable to open input file");
        printf( "Result code %d\n", l_x);
        return l_x;
    }  /* end test for failed open */


    //
    //  once the file is open we can connect our
    //  Record Access Block to it.
    //
    l_x = sys$connect( m_rab);

    if (!(l_x & 1))
    {
        puts( "Unable to connect RAB to index");
        printf( "Result code %d\n", l_x);
        l_x = sys$close( m_fab);
        return l_x;

    }  /* end test for failed connection */

    return l_x;

}  /* end open_stats_idx */

/*;;;;;
 *  Subroutine to write the report from our sorted working arrays.
 *;;;;;
 */
void write_most_report( struct zillionare_stats_record *d_stats,
                    struct zillionare_stats_record *m_stats,
                    FILE *rpt_file)
{
    int     l_x;
    time_t  the_time;
    char    work_str[255];
    struct  zillionare_stats_record *work_stats;


    //
    //  Format system date into a string for display
    //
    the_time = time(0);
    memset( work_str, '\0', sizeof( work_str));

    strftime( work_str, sizeof( work_str)-1, "%x", 
                localtime( &the_time));
    
    //
    //  First heading
    //
    fprintf( rpt_file, "%s %22.22sMost Hits Report\n\n", work_str, " ");
    fprintf( rpt_file, "%29.29sRegular Drawing Numbers\n\n\n", " ");
    
    //
    //  Column headings
    //
    fprintf( rpt_file, "No   Hits   Since   Pct_hits   Ave_btwn\n");
    fprintf( rpt_file, "--   ----   -----   --------   --------\n");


    work_stats = d_stats;

    for( l_x=0; l_x < 10; l_x++)
    {
        fprintf( rpt_file, "%2d   %4d   %4d     %5.3f    %5.3f\n",
                    work_stats[l_x].elm_no,
                    work_stats[l_x].hit_count,
                    work_stats[l_x].since_last,
                    work_stats[l_x].pct_hits,
                    work_stats[l_x].ave_btwn);

        

    }  /* end for l_x loop */



    //
    //  Second heading
    //
    fprintf( rpt_file, "\n\n\n%30.30sMega Drawing Numbers\n\n\n", " ");
    
    //
    //  Column headings
    //
    fprintf( rpt_file, "No   Hits   Since   Pct_hits   Ave_btwn\n");
    fprintf( rpt_file, "--   ----   -----   --------   --------\n");


    work_stats = m_stats;

    for( l_x=0; l_x < 10; l_x++)
    {

        fprintf( rpt_file, "%2d   %4d   %4d     %5.3f    %5.3f\n",
                    work_stats[l_x].elm_no,
                    work_stats[l_x].hit_count,
                    work_stats[l_x].since_last,
                    work_stats[l_x].pct_hits,
                    work_stats[l_x].ave_btwn);

    }  /* end for l_x loop */

}  /* end write_most_report subroutine */
