/* DRAWING_RECORD_CLASS.CXX                                                 */
/*                                                                          */
/*  C++ class designed to handle all of our IO requirements for the         */
/*  drawing data indexed file.  This file is part of the Mega Zillionare    */
/*  example application.                                                    */
/*                                                                          */
#include "drawing_record_class.hxx"


//;;;;;
//      Constructor
//;;;;;
Drawing_Record_Class::Drawing_Record_Class()
{
    short int     b_x;
    
    strcpy( default_file_name, "MY_MEGA_FILE");

    memset( &local_rec, '\0', sizeof( local_rec));

    local_fab = cc$rms_fab;                                    // fill in defaults for FAB
    local_fab.fab$l_fna = default_file_name;                   // Pointer to file name
    local_fab.fab$b_fns = strlen(default_file_name);           // length of name
    local_fab.fab$l_dna = ".IDX";                              // default name, really just file type
    local_fab.fab$b_dns = 4;                                   // length of default name
    local_fab.fab$b_org = FAB$C_IDX;                           // indexed file
    local_fab.fab$b_fac = FAB$M_GET;                           // get, put, etc.
    local_fab.fab$w_mrs = sizeof( struct drawing_record);      // maximum record size
    local_fab.fab$b_rat = FAB$M_FTN;                           // FORTRAN formatted records
    local_fab.fab$b_rfm = FAB$C_FIX;                           // fixed length records
    local_fab.fab$b_shr = FAB$M_NIL;                           // no file sharing
    local_fab.fab$l_xab = (void *) &local_xab;                 // index block address


    local_xab = cc$rms_xabkey;                                 // fill in defaults for XAB
    local_xab.xab$b_ref = 0;                                   // primary key is zero
    local_xab.xab$b_nsg = 1;                                   // single segment key
    local_xab.xab$w_pos[0] = 0;                                // starting position zero in the record
    local_xab.xab$b_siz[0] = 8;                                // our key is 8 bytes long
    local_xab.xab$b_prolog = XAB$C_PRG3;                       // prolog 3
    local_xab.xab$l_nxt = 0;                                   // end of chain   we only have one key on this file.


    local_rab = cc$rms_rab;                                    // fill in defaults for RAB
    local_rab.rab$l_fab = &local_fab;                          // pointer to FAB
    local_rab.rab$b_rac = RAB$C_KEY;                           // keyed access
    local_rab.rab$b_krf = 0;                                   // access via primary key
    local_rab.rab$l_kbf = local_rec.draw_dt;                   // key buffer
                                                               // 
    local_rab.rab$b_ksz = sizeof( local_rec.draw_dt);          // in our C++
                                                               // version we
                                                               // fill in the
                                                               // key size.
    local_rab.rab$l_ubf = (char *)&local_rec;                  // data buffer for record
    local_rab.rab$l_rbf = (char *)&local_rec;                  // data buffer for write
    local_rab.rab$w_usz = sizeof( struct drawing_record);
    local_rab.rab$w_rsz = sizeof( struct drawing_record);

}  /* end constructor */

//;;;;;
//      Destructor
//;;;;;
Drawing_Record_Class::~Drawing_Record_Class()
{
    close();

}  /* end destructor */

//;;;;;
//      Open method
//;;;;;
int Drawing_Record_Class::open( int access, char *f_name)
{
    int     l_x;
    
    if ( f_name != NULL)
        strcpy( file_name, f_name);
    else
        strcpy( file_name, default_file_name);

    local_fab.fab$l_fna = file_name;
    local_fab.fab$b_fns = strlen( file_name);
    local_fab.fab$b_fac = access;

    l_x = sys$open( &local_fab);

    if ( !(l_x & 1))
    {
        cout << "Unable to open file " << file_name << endl;
        cout << "Error " << l_x << endl;
        return l_x;
    }  /* end test for successfull open */

    l_x = sys$connect( &local_rab);
    if ( !(l_x & 1))
    {
        cout << "Unable to connect RAB to index file" << file_name << endl;
        cout << "Error " << l_x;
    }  /* end test for successful connection */

    return l_x;
    
}  /* end open method */

//;;;;;
//      Close routine
//;;;;;
void Drawing_Record_Class::close()
{
    sys$close( &local_fab);
    
}  /* end close method */

//;;;;;
//      Keyed retrieval functions
//;;;;;
int Drawing_Record_Class::get_via_k0( char *key_val, struct drawing_record *d)
{
    int     l_x;

    if (strlen( key_val) > sizeof( local_rec.draw_dt))
        l_x = sizeof( local_rec.draw_dt);
    else
        l_x = strlen( key_val);
        
    memcpy( local_rec.draw_dt, key_val, l_x);

    //
    //  Be certain we have the correct options.
    //  
    local_rab.rab$l_rop = RAB$M_KGE | RAB$M_WAT | RAB$M_RLK;
    local_rab.rab$b_rac = RAB$C_KEY;

    l_x = sys$find( &local_rab);

    if ( l_x & 1)
    {
        l_x = sys$get( &local_rab);
        if ( d != NULL)
            get_record_data( d);
    }

    return l_x;
}  /* end get_via_k0 method */


int Drawing_Record_Class::get_next_key( struct drawing_record *d)
{
    int     l_x;

    //
    //  Be certain we have the correct options.
    //  
    local_rab.rab$l_rop = RAB$M_NXT | RAB$M_WAT | RAB$M_RLK;
    local_rab.rab$b_rac = RAB$C_KEY;

    l_x = sys$get( &local_rab);
    if ( d != NULL)
        get_record_data( d);

    return l_x;
}  /* end get_next_key method */



int Drawing_Record_Class::get_next_key_rev( struct drawing_record *d)
{
    int     l_x;

    //
    //  Be certain we have the correct options.
    //  
    local_rab.rab$l_rop = RAB$M_REV | RAB$M_NXT | RAB$M_WAT | RAB$M_RLK;
    local_rab.rab$b_rac = RAB$C_KEY;

    l_x = sys$get( &local_rab);
    if ( d != NULL)
        get_record_data( d);

    return l_x;
}  /* end get_next_key method */


//;;;;;
//  Plain old sequential read
//;;;;;
int Drawing_Record_Class::get_seq( struct drawing_record *d)
{
    int     l_x;

    //
    //  Be certain we have the correct options.
    //  
    local_rab.rab$l_rop = RAB$M_WAT | RAB$M_RLK;
    local_rab.rab$b_rac = RAB$C_SEQ;

    l_x = sys$get( &local_rab);
    if ( d != NULL)
        get_record_data( d);

    return l_x;
}  /* end get_seq method */

//;;;;;
//  Write a record to the file
//;;;;;
int Drawing_Record_Class::put_record( struct drawing_record *d)
{
    int     l_x;
    
    if ( d != NULL)
        set_record_data( d);

    l_x = sys$put( &local_rab);

    return l_x;
    
}  /* end put_record method */

//;;;;;
//  Update the current record
//;;;;;
int Drawing_Record_Class::update_record( struct drawing_record *d)
{
    int     l_x;
    
    if ( d != NULL)
        set_record_data( d);

    l_x = sys$update( &local_rab);

    return l_x;
    
}  /* end update_record method */

//;;;;;
//  Method to delete the current record
//;;;;;
int Drawing_Record_Class::delete_record()
{
    int     l_x;

    l_x = sys$delete( &local_rab);

    return l_x;
    
}  /* end delete_record method */


//;;;;;
//      Transfer record data into and out of buffer
//;;;;;
void Drawing_Record_Class::get_record_data( struct drawing_record *d)
{
    *d = local_rec;
    
}  /* end get_record_data method */


void Drawing_Record_Class::set_record_data( struct drawing_record *d)
{
    local_rec = *d;
    
}  /* end set_record_data method */
