/*
 * Copyright (c) 1983, 1988, 1993, 1994
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright (c) 1983, 1993\n\
	The Regents of the University of California.  All rights reserved.\n";
#endif /* not lint */

#ifndef lint
static char sccsid[] = "@(#)logger.c	8.1 (Berkeley) 6/6/93";
#endif /* not lint */

/*
**++
**  FACILITY:  syslog
**
**  MODULE DESCRIPTION:
**
**      logger for OpenVMS.  This program is used to send messages to a 
**  syslogd process.
**
**  AUTHORS:
**
**      John Vottero
**
**  CREATION DATE:  05-May-1995
**
**  DESIGN ISSUES:
**
**      {@tbs@}
**
**  [@optional module tags@]...
**
**  MODIFICATION HISTORY:
**
**      {@tbs@}...
**--
*/


/*
**
**  INCLUDE FILES
**
*/

#include <stddef.h>
#include <ctype.h>
#include <assert.h>
#include <limits.h>
#include <signal.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <unixio.h>
#include <file.h>
#include <time.h>

#define SYSLOG_NAMES
#include "syslog.h"

/*
**  TCP/IP (UCX) Includes
*/
#include <in.h>
#include <inet.h>
#include <netdb.h>
#include <socket.h>

/*
**  VMS Specific Includes
*/
#include <descrip.h>
#include <jpidef.h>
#include <lib$routines.h>
#include <prvdef.h>
#include <starlet.h>

#define	dprintf		if (debugging) printf
#define bitmask(the_bit) (1 << (the_bit - 1))
#define MAXHOSTNAMELEN 64
#define TIMERINTVL 300

static int	debugging;		/* debug flag */
static int	priv_user;		/* User has priv for commands */

/*
**  Static Function Prototypes
*/
static void log_line(
    char *raw_line,
    struct sockaddr_in *from,
    int from_len);

static void log_error(
    char *error_msg);

static int decode_fac_pri(
    char *fac_pri_txt);

int decode(
    char *name,
    CODE *codetab);

void print_usage(void);


/*
**  MAIN - Main entry point for logger
*/
int main(
    int argc,
    char **argv)
{
    int argi;
    int option_ch;
    int sd_syslog;		/* The syslog Socket Descriptor */
    int send_testmsg;
    int send_msg_len;
    int bytes_sent;
    int fac_pri_int;
    int send_command;
    unsigned int priv_bits[2];
    char the_command[32];
    char *message_tail;
    char message_str[1024];
    char send_hostname[MAXHOSTNAMELEN + 1];
    char send_msg[2048];
    char time_str[32];
    struct sockaddr_in to_addr;
    struct hostent *hp;
    time_t now;
    struct servent *service_ent;
    struct sockaddr_in socket_name;

    /*
    **  Check the users privledges to see if they are allowed to
    **  issue commands to the SYSLOGD process
    */
    lib$getjpi (
    	&JPI$_CURPRIV, 
    	0, 
    	0, 
    	priv_bits, 
    	0, 
    	0);
    if (priv_bits[0] & PRV$M_SYSPRV)
	{
	priv_user = TRUE;
	}
    else
	{
	priv_user = FALSE;
	}

    /*
    **  Parse the command line options
    */
    send_command = FALSE;
    debugging = FALSE;
    (void) gethostname(send_hostname, sizeof send_hostname);
    fac_pri_int = LOG_USER | LOG_NOTICE;
    for (argi=1; argi<argc; argi++)
	{
	if (argv[argi][0] != '-')
	    {
	    /*
	    **  This must be the start of the message
	    */
	    break;
	    }
	else if (strcmp(argv[argi], "-d") == 0)
	    {
	    debugging = TRUE;
	    }
	else if (strcmp(argv[argi], "-h") == 0)
	    {
	    argi++;
	    if (argi<argc)
		{
		strcpy(send_hostname, argv[argi]);
		}
	    else
		{
		(void)fprintf(stderr,
		    "-h requires a host name to follow\n");
		print_usage();
		}
	    }
	else if (strcmp(argv[argi], "-p") == 0)
	    {
	    argi++;
	    if (argi<argc)
		{
		fac_pri_int = decode_fac_pri(argv[argi]);
		}
	    else
		{
		(void)fprintf(stderr,
		    "-p requires [facility.]priority to follow\n");
		print_usage();
		}
	    }
	else if (strcmp(argv[argi], "-c") == 0)
	    {
	    /*
	    **  A command is being issued
	    */
	    argi++;
	    if (priv_user)
		{
		if (argi<argc)
		    {
		    send_command = TRUE;
		    strncpy(the_command, argv[argi], sizeof(the_command));
		    the_command[(sizeof(the_command) - 1)] = '\0';
		    }
		else
		    {
		    (void)fprintf(stderr,
			"-c requires a command to follow\n");
		    print_usage();
		    }
		}
	    else
		{
		(void)fprintf(stderr,
		    "Not authorized to issue commands (-c)\n");
		print_usage();
		}
	    }
	else if (strcmp(argv[argi], "-?") == 0)
	    {
	    print_usage();
	    }
	else
	    {
	    (void)fprintf(stderr,
		"%s is an invalid option\n", argv[argi]);
	    print_usage();
	    } 
	}

    /*
    **  Get the message
    */
    message_tail = message_str;
    if (argi < argc)
	{
	while(argi < argc)
	    {
	    strcpy(message_tail, argv[argi]);
	    message_tail += strlen(argv[argi]);
	    *message_tail = ' ';
	    message_tail++;
	    argi++;
	    }

	/*
	**  Backup to get rid of the last space
	*/
	message_tail--;
	}
    *message_tail = '\0';

    /*
    **  Create a socket
    */
    sd_syslog = socket(AF_INET, SOCK_DGRAM, 0);
    if (sd_syslog < 0)
	{
	log_error("Unable to create socket");
	}

    /*
    **  Get the syslogd service name
    */
    service_ent = getservbyname("syslog", "udp");
    if (service_ent == NULL)
	{
	errno = 0;
	log_error("syslog/udp: unknown service");
	}

    /*
    **  See if we are sending a command
    */
    if (send_command)
	{
	/*
	**  Figure out what the command is
	*/
	switch (the_command[0])
	    {
	    case 'S':
	    case 's':
		{
		/*
		**  Shutdown
		*/
		fac_pri_int = 0x00FF0000 | 1;
		break;
		}
	    case 'R':
	    case 'r':
		{
		/*
		**  Reopen the log
		*/
		fac_pri_int = 0x00FF0000 | 2;
		break;
		}
	    default:
		{
		/*
		**  A bogus command
		*/
		(void)fprintf(stderr,
		    "valid commands are s (shutdown) and r (reopen log)\n");
		print_usage();
		}
	    }

	/*
	**  Change the message to <n>USERNAME at TERMINAL
	*/
	sprintf(send_msg,
	    "<%d>%s at %s",
	    fac_pri_int,
	    cuserid(NULL),
	    ctermid(NULL));
	}
    else
	{
	/*
	**  Format the message
	*/
	(void) time(&now);
	strcpy(time_str, ctime(&now));
	sprintf(
	    send_msg,
	    "<%d>%.16s%s",
	    fac_pri_int,
	    &time_str[4],
	    message_str);
	}

    /*
    **  Convert the host name to a number
    */
    hp = gethostbyname(send_hostname);
    if (hp == NULL)
	{
	printf("Unknown host, %s\n",
	    send_hostname);
	close (sd_syslog);
	return;
	}

    dprintf("Sending message >%s<\nto %s\n",
	send_msg,
	send_hostname);

    /*
    **  Setup socket address structure
    */
    to_addr.sin_family = AF_INET;
    to_addr.sin_port = service_ent->s_port;
    memcpy(&to_addr.sin_addr, hp->h_addr_list[0], hp->h_length);
    memset(to_addr.sin_zero, 0, sizeof(to_addr.sin_zero));

    /*
    **  Send the message
    */
    send_msg_len = strlen(send_msg);
    bytes_sent = sendto(
		    sd_syslog,
		    send_msg,
		    send_msg_len,
		    0,
		    (struct sockaddr *) &to_addr,
		    sizeof(to_addr));

    /*
    **  Check bytes sent vs. expected
    */
    if (bytes_sent != send_msg_len)
	{
	printf("Sent bytes does not match expected, sent %d, expected %d\n",
	    bytes_sent,
	    send_msg_len);
	}

    /*
    **  Close the socket and return
    */
    close (sd_syslog);

    return;
}



/*
**  LOG_ERROR - Log an error message
*/
static void log_error(
    char *error_msg)
{
    printf("%s\n", error_msg);
    exit (0);
}


/*
**  DECODE_FAC_PRI - Decode a facility/priority from ASCII to integer
**		     ASCII form is [facility.]priority
**		     If the ASCII text looks like a number, return the number
*/
static int decode_fac_pri(
    char *fac_pri_txt)
{
    char *fac_ptr;
    char *pri_ptr;
    int fac_int;
    int pri_int;

    fac_ptr = fac_pri_txt;

    /*
    **  Find the . separating the facility from the priority
    */
    for (pri_ptr = fac_pri_txt;
	 ((*pri_ptr) && (*pri_ptr != '.'));
	 pri_ptr++);

    /*
    **  If the pri_ptr is pointing to a ., we have facility.priority
    */
    if (*pri_ptr == '.')
	{
	*pri_ptr = '\0';
	fac_int = decode(fac_ptr, facilitynames);
	if (fac_int < 0)
	    {
	    (void)fprintf(stderr,
		"logger: unknown facility name: %s.\n", fac_ptr);
	    exit(1);
	    }
	*pri_ptr = '.';
	pri_ptr++;
	}
    else
	{
	/*
	**  No facility, try to decode the whole thing as a priority
	*/
	fac_int = 0;
	pri_ptr = fac_pri_txt;
	}

    /*
    **  Now decode the priority
    */
    pri_int = decode(pri_ptr, prioritynames);
    if (pri_int < 0)
	{
	(void)fprintf(stderr,
	    "logger: unknown priority name: %s.\n", pri_ptr);
	exit(1);
	}

    /*
    **  Return the facility and priority
    */
    return ((fac_int & LOG_FACMASK) | (pri_int & LOG_PRIMASK));
}


/*
**  DECODE - Convert from a name to a number via a CODE table
*/
int decode(
    char *name,
    CODE *codetab)
{
    int i;
    char lower_name[32];
    CODE *c;

    /*
    **  If it starts with a digit, convert the whole thing to a number
    */
    if (isdigit(*name))
	return (atoi(name));

    /*
    **  Make sure that the input is lowercase
    */
    for(i=0; name[i]; i++)
	{
	lower_name[i] = tolower (name[i]);
	if (i > 30) break;
	}
    lower_name[i] = '\0';

    /*
    **  Look it up in the table
    **  Do a caseless compare
    */
    for (c = codetab; c->c_val >= 0; c++)
	{
	if (!strcmp(lower_name, c->c_name))
		return (c->c_val);
	}

    /*
    **  If we got this far, we didn't find it
    */
    return (-1);
}


/*
**
*/
void print_usage(void)
{
	(void)fprintf(stderr,
	    "logger [-p [facility.]priority] [-h host] [ message ... ]\n");
	if (priv_user)
	    {
	    (void)fprintf(stderr,
		"Or, to send a command to SYSLOG\nlogger -c (s | r)\n(s means shutdown, r means reopen logs)");
	    }
	exit(1);
}
