/*       deltree

         Author
           Jonathan C. Baker
         Date
            1 Oct 92
           24 Nov 92  -  Jon Baker  - Fix error message, close files after
                                      reading file access block (FAB) to
                                      fix log error, modify some return
                                      status's.
            8 Dec 92  -  Jon Baker  - Replace delete command from C with
                                      system service LIB$DELETE_FILE to
                                      check return status codes.
           15 Dec 92  -  Jon Baker  - When using /CONFIRM, allow inputs that
                                      begin with Y, T or 1 to translate as
                                      yes, and N, F or 0 to translate as
                                      no.  Requestion all other inputs.
           10 Fef 93  -  Jon Baker  - When using /CONFIRM, allow ALL as valid
                                      input.  Strip leading and trailing
                                      blanks from /CONFIRM input.
            6 Jul 93  -  Jon Baker  - Handle identifiers and ACLs.
            2 Sep 93  -  Jon Baker  - Display file that is locked or user does
                                      not have privilege to instead of the
                                      directory that file resides in.

         Purpose
           Delete directory structures
         Description
           Delete directory structures by using recursive call to clean out 
           files beneath a directory
         Routines
     F     blocks       : determines allocated blocks of file
     F     check_dir    : check directory for validity
     F     command_line : get command line parameters and qualifier switches
     F     del_files    : recursive routine to delete files
     S     msg          : message routine
         Global Variables & Constants
     C     false        : boolean false
     C     max_file_len : maximum file length
     C     null         : ASCII null character
     C     true         : boolean true
     G     about        : about deltree
     G     block_size   : file block size
     G     confirm      : switch to confirm deletion of files
     G     file_count   : total files deleted
     G     log          : switch to display deletion of files
     G     total_blocks : total blocks deleted
         
         include c modules
*/       
#include <clidef.h>
#include <climsgdef.h>
#include <descrip.h>
#include <iodef.h>
#include <lmfdef.h>
#include <rms.h>
#include <rmsdef.h>
#include <ssdef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
/*
         define constants
*/
#define  true            1
#define  false           0
#define  max_file_len    256
#define  null            0
/*
         define some VMS error codes
*/
#define  RMS$_OK1        98962
#define  RMS$_OK2        99018
/*
         define global variables
*/
         long               log, confirm, about, block_size, file_count, 
                            total_blocks;
/*
***********************************************************************
*/
 long    *blocks (filename)
/*
         author
           jonathan c. baker
         date
           8 october 1992
         purpose
           find the allocated blocks to a file
         description
           use RMS calls
         communications
           calls from
             main
             del_files
           calls to
             none

         parameters
           filename     : file to retrieve allocated blocks
         local glossary
           file_fab     : file information structure
           status       : return variable
           status1      : return variable

         declare parameters
*/
         char      *filename;
{
/*
         declare variables
*/
         struct FAB file_fab;

         long       status, status1;
/*
         initialize variables
*/
         block_size = 0;
/*
         initialize the FAB and NAM
*/
         file_fab = cc$rms_fab;
         file_fab.fab$l_fna = filename;
         file_fab.fab$b_fns = strlen (filename);
         file_fab.fab$b_fac = FAB$M_GET;
         file_fab.fab$l_fop = FAB$V_NAM;
/*
         open for file information
*/
         status = SYS$OPEN(&file_fab);
         status1 = SYS$CLOSE(&file_fab);
         if (status != RMS$_NORMAL)
         {
           return 18;
         }
/*
         extract information
*/
         block_size = file_fab.fab$l_alq;
         file_count = file_count + 1;
         total_blocks = total_blocks + block_size;

         return 0;
}
/*
         end of blocks

***********************************************************************
*/
 long    *check_dir (dir)
/*
         author
           jonathan c. baker
         date
           4 september 1992
         purpose
           see if valid directory to delete
         description
           test with LIB$FIND_FILES
         communications
           calls from
             main
           calls to
             (none)

         parameters
           dir         : directory to test
         local glossary
           command     : command to execute
           context     : file context
           current     : current directory
           ent_desc    : entry descriptor
           filename    : filename
           flag        : flag variable
           flag_desc   : temporary descriptor
           i           : control variable
           j           : control variable
           k           : control variable
           len         : length of variable
           ret_desc    : return descriptor
           status      : return variable
           status1     : return variable

         declare parameters
*/
         char     *dir;
{
/*
         declare local variables
*/
         struct    dsc$descriptor_s flag_desc, ent_desc, ret_desc;
         long      status, len, context, status1, i, j, k;
         char      command[max_file_len], flag[10], 
                   filename[max_file_len], current[max_file_len];
/*
         initialize variables
*/
         ent_desc.dsc$b_dtype = DSC$K_DTYPE_T;
         ent_desc.dsc$b_class = DSC$K_CLASS_S;
         ret_desc.dsc$b_dtype = DSC$K_DTYPE_T;
         ret_desc.dsc$b_class = DSC$K_CLASS_S;
         flag_desc.dsc$b_dtype = DSC$K_DTYPE_T;
         flag_desc.dsc$b_class = DSC$K_CLASS_S;
/*
         get local directory 
*/
         ent_desc.dsc$a_pointer = current;
         ent_desc.dsc$w_length = max_file_len;
         status = sys$setddir (0, &len, &ent_desc);
/*
         set up dir for proper format if needed
*/
         if (strstr(dir, "[]") == dir)
         {
           strcpy (dir, current);
         }

         if (strstr(dir, "[.") == dir)
         {
           strcpy (filename, dir);
           len = strlen (current) - 1;
           strncpy (dir, current, len);
           j = len;
           len = strlen (filename);
           for (i = 1; i < len; i = i + 1)
           {
             k = j + i - 1;
             dir[k] = filename[i];
           }
           k = j + i - 1;
           strcpy (&dir[k], "\0");
         }
/*
         perform logical tests
*/
         if (strstr (dir, "]") == '\0')
         {
           strncat (dir, ":\0", 1);
         }
/*
         test for directory existence and proper structure
*/
         strcpy (command, dir);
         strncat (command, "*.*.*\n", 5);
         ent_desc.dsc$a_pointer = command;
         ent_desc.dsc$w_length = strlen (command);
         ret_desc.dsc$a_pointer = filename;
         ret_desc.dsc$w_length = max_file_len;
         strcpy (flag, "MULTIPLE\0");
         flag_desc.dsc$a_pointer = flag;
         flag_desc.dsc$w_length = strlen (flag);
         context = 0;
         status = lib$find_file (&ent_desc, &ret_desc, 
                                 &context, 0, 0, 0, &flag_desc);
         status1 = lib$find_file_end (&context);
         if ((status != RMS$_NORMAL) && 
             (status != RMS$_NMF) &&
             (status != RMS$_OK1) &&
             (status != RMS$_OK2))
         {
/*
         return appropriate error codes
*/
           if (status == 100052)
           {
             return 10;
           }
           if (status == 114730)
           {
             return 11;
           }
           if (status == 114762)
           {
             return 12;
           }
           if (status == 98970)
           {
             return 13;
           }
           return 15;                 /* all purpose error code */
         }

         return 0;
};
/*
         end of check_dir

***********************************************************************
*/
 char    *command_line ()
/*
         author
           jonathan c. baker
         date
           1 jun 1992
         purpose
           to set command line variables
         description
           read command line through cli$ routines
         communications
           calls from
             main
           calls to
             (none)

         parameters
           command_line  : returns directory to delete
           confirm       : confirm qualifier switch
           log           : log qualifier switch
         local glossary
           ent_desc      : entity descriptor
           param         : parameter
           parameter_des : parameter descriptor
           ret_desc      : return descriptor
           ret_len       : return length
           status        : status variable
           tempdir       : temporary directory

         declare parameters
*/
{
/*
         declare local variables
*/
         struct    dsc$descriptor_s parameter_des, ent_desc, ret_desc;
         char      param[max_file_len], tempdir[max_file_len],
                   temp[max_file_len];
         short     ret_len;
         long      *status, i;
/*
         initialize variables
*/
         ent_desc.dsc$b_dtype = DSC$K_DTYPE_T;
         ent_desc.dsc$b_class = DSC$K_CLASS_S;
         ret_desc.dsc$b_dtype = DSC$K_DTYPE_T;
         ret_desc.dsc$b_class = DSC$K_CLASS_S;
         parameter_des.dsc$b_dtype = DSC$K_DTYPE_T;
         parameter_des.dsc$b_class = DSC$K_CLASS_S;
/*
         set cli calls
*/
         strcpy (param, "ABOUT");
         parameter_des.dsc$a_pointer = param;
         parameter_des.dsc$w_length = strlen (param);
         if (cli$present(&parameter_des) == CLI$_PRESENT)
         {
           about = true;
         }
         else
         {
           about = false;

           strcpy (param, "CONFIRM");
           parameter_des.dsc$a_pointer = param;
           parameter_des.dsc$w_length = strlen (param);
           if (cli$present(&parameter_des) == CLI$_PRESENT)
           {
             confirm = true;
           }
           else
           {
             confirm = false;
           }

           strcpy (param, "LOG");
           parameter_des.dsc$a_pointer = param;
           parameter_des.dsc$w_length = strlen (param);
           if (cli$present(&parameter_des) == CLI$_PRESENT)
           {
             log = true;
           }
           else
           {
             log = false;
           }
/*
         get the directory to delete
*/
           ret_desc.dsc$a_pointer = tempdir;
           ret_desc.dsc$w_length = max_file_len;
           strcpy (param, "P1");
           ent_desc.dsc$a_pointer = param;
           ent_desc.dsc$w_length = strlen (param);
           status = cli$get_value (&ent_desc, &ret_desc, &ret_len);
           tempdir[ret_len] = '\0';
           while (strlen(tempdir) == 0)
           {
             printf("Enter directory to delete:  ");
             gets (&tempdir);
             ent_desc.dsc$a_pointer = tempdir;
             ent_desc.dsc$w_length = strlen (tempdir);
             ret_desc.dsc$a_pointer = tempdir;
             ret_desc.dsc$w_length = max_file_len;
             str$upcase (&ent_desc, &ret_desc);
             ent_desc.dsc$a_pointer = tempdir;
             ent_desc.dsc$w_length = strlen (tempdir);
             ret_desc.dsc$a_pointer = tempdir;
             ret_desc.dsc$w_length = max_file_len;
             str$trim (&ret_desc, &ent_desc, &i);
             tempdir[i] = '\0';
/*
         remove preceeding spaces
*/
             strcpy (&temp, &tempdir);
             for (i = 0; temp[i] == ' '; i = i + 1);
             strcpy (&tempdir, &temp[i]);
             i = strstr (tempdir, " \0");
             if (i != 0)
             {
               tempdir[i] = '\0';
             }
           }
           return tempdir;
         }
};
/*
         end command_line

***********************************************************************
*/
 long    *del_files (dir)
/*
         author
           jonathan c. baker
         date
           3 september 1992
         purpose
           to delete the files beneath the subdirectory.
         description
           delete files with the appropriate flags provided.
         communications
           calls from
             main
             del_files
           calls to
             del_files
             msg

         parameters
           dir             : directory to delete
         local glossary
           answer          : confirmation answer
           check_confirm   : check confirm status
           command         : delete command
           context         : for closing call to lib$find_file
           current         : current directory
           ent_desc        : entity descriptor
           filename        : filename to delete
           flag            : search type for lib$find_file
           flag_desc       : flag descriptor
           len             : length of current
           new_dir         : directory file to delete
           ptr             : pointer variable
           ret_desc        : return descriptor
           sent_desc       : entity descriptor
           sret_desc       : return descriptor
           status          : return variable
           status1         : return variable

         declare parameters
*/
         char      *dir;
{
/*
         declare local variables
*/
         struct    dsc$descriptor_s flag_desc, ent_desc, ret_desc, del_desc,
                   sent_desc, sret_desc;
         long      status, len, context, status1, check_confirm, i;
         char      answer[max_file_len], command[max_file_len], flag[10], 
                   filename[max_file_len], current[max_file_len], *ptr,
                   new_dir[max_file_len], temp[max_file_len];
/*
         initialize variables
*/
         del_desc.dsc$b_dtype = DSC$K_DTYPE_T;
         del_desc.dsc$b_class = DSC$K_CLASS_S;
         sent_desc.dsc$b_dtype = DSC$K_DTYPE_T;
         sent_desc.dsc$b_class = DSC$K_CLASS_S;
         sret_desc.dsc$b_dtype = DSC$K_DTYPE_T;
         sret_desc.dsc$b_class = DSC$K_CLASS_S;
         ent_desc.dsc$b_dtype = DSC$K_DTYPE_T;
         ent_desc.dsc$b_class = DSC$K_CLASS_S;
         ret_desc.dsc$b_dtype = DSC$K_DTYPE_T;
         ret_desc.dsc$b_class = DSC$K_CLASS_S;
         flag_desc.dsc$b_dtype = DSC$K_DTYPE_T;
         flag_desc.dsc$b_class = DSC$K_CLASS_S;
/*
         delete files in the directory
*/
         strcpy (command, dir);
         strncat (command, "*.*.*\n", 5);
         ent_desc.dsc$a_pointer = command;
         ent_desc.dsc$w_length = strlen (command);
         ret_desc.dsc$a_pointer = filename;
         ret_desc.dsc$w_length = max_file_len;
         strcpy (flag, "MULTIPLE\0");
         flag_desc.dsc$a_pointer = flag;
         flag_desc.dsc$w_length = strlen (flag);
         context = 0;
         status = lib$find_file (&ent_desc, &ret_desc, 
                                 &context, 0, 0, 0, &flag_desc);
         if (status != RMS$_NORMAL)
         {
           if ((status == RMS$_NMF) ||
               (status != RMS$_OK1) ||
               (status != RMS$_OK2))
           {
             return 0;
           }
           if (status == 114730)
           {
             return 11;
           }
           if (status == 98970)
           {
             strcpy (dir, filename);
             return 13;
           }
           dir = filename;
           return 16;                 /* all purpose error code */
         }

         while (status == RMS$_NORMAL)
         {
           ptr = strchr(filename, ' ');
           *ptr = '\0';
/*
         test for a directory
*/
           ptr = strstr(filename, ".DIR;1");
           if (ptr != null)
           {
             strcpy (new_dir, filename);
             ptr = strstr(new_dir, ".DIR;1");
             *ptr = ']';
             ptr = ptr + 1;
             *ptr = '\0';
             ptr = strchr(new_dir, ']');
             *ptr = '.';

             status = del_files (new_dir);
             if (status != 0)
             {
/*
         return error code
*/
               strcpy (dir, new_dir);
               return status;
             }
           }
/*
         delete file
*/
           if (confirm == true)
           {
             check_confirm = true;
             while (check_confirm)
             {
               printf ("\nDELETE  %s  ? [N]:  ", filename);
               gets (&answer);
               sent_desc.dsc$a_pointer = answer;
               sent_desc.dsc$w_length = strlen (answer);
               sret_desc.dsc$a_pointer = answer;
               sret_desc.dsc$w_length = max_file_len;
               str$upcase (&sent_desc, &sret_desc);
/*
         remove preceeding spaces
*/
               strcpy (&temp, &answer);
               for (i = 0; temp[i] == ' '; i = i + 1);
               strcpy (&answer, &temp[i]);
               strcpy (&temp, &answer);
               for (i = 0; (temp[i] != ' ' && temp[i] != '\0'); i = i + 1);
               temp[i] = '\0';
               strcpy (&answer, &temp);
/*
         set up alternate answers
*/
               if ((answer[0] == '1') || (answer[0] == 'T'))
               {
                 answer[0] = 'Y\0';
               }
               if ((answer[0] == '0') || (answer[0] == 'F') ||
                   (strlen(answer) == 0))
               {
                 answer[0] = 'N\0';
               }

               if (strcmp(answer, "ALL\0") == 0)
               {
                 check_confirm = false;
                 confirm = false;
               }
               else if (answer[0] == 'Y')
               {
                 check_confirm = false;
               }
               else if (answer[0] == 'N')
               {
/*
         stop with non-confirm status
*/
                 return 20;
               }
               else
               {
                 status1 = 21;
                 msg (&status1, answer);
               }
             }
           }
/*
         attempt setting protection
*/
           status1 = chmod (filename, 0006);

           if (log == true)
           {
             status = blocks (filename);
             if (status != 0)
             {
               strcpy (dir, filename);
               return status;
             }
             del_desc.dsc$a_pointer = filename;
             del_desc.dsc$w_length = strlen (filename);
             status = lib$delete_file (&del_desc);
/*
         set appropriate status code and return messages
*/
             if (status == SS$_NORMAL)
             {
               status = 2;
               msg (&status, filename);
             }
             else
             {
               strcpy (dir, filename);
               if (status == 98970)
               {
                 return 13;
               }
               else
               {
                 return 18;
               }
             }
           }
           else
           {
             del_desc.dsc$a_pointer = filename;
             del_desc.dsc$w_length = strlen (filename);
             status = lib$delete_file (&del_desc);
/*
         set appropriate status codes and return messages
*/
             if (status != SS$_NORMAL)
             {
               strcpy (dir, filename);
               if (status == 98970)
               {
                 return 13;
               }
               else
               {
                 return 18;
               }
             }
           }
/*
         notify the deletion of directory file
*/
           ptr = strstr(filename, ".DIR;1");
           if (ptr != null)
           {
             status = 1;
             msg (&status, filename);
           }
/*
         get the next file to delete
*/
           status = lib$find_file (&ent_desc, &ret_desc, 
                                   &context, 0, 0, 0, &flag_desc);
         }
         status1 = lib$find_file_end (&context);
/*
         return appropriate error codes
*/
         if ((status != RMS$_NORMAL) && (status != RMS$_NMF) &&
             (status != RMS$_OK1) && (status != RMS$_OK2))
         {
           strcpy (dir, filename);
           if (status == 98970)
           {
             return 13;
           }
           return 16;                 /* all purpose error code */
         }

         return 0;
};
/*
         end of del_files

***********************************************************************
*/
         msg (num, str)
/*
         author
           jonathan c. baker
         date
           4 september 1992
         purpose
           write out message codes.
         description
           use number reference to message code.
         communications
           calls from
             main
             del_files
           calls to
             (none)

         parameters
           str        : string used in informational message
           num        : message number
         local glossary
           (none)

         declare parameters
*/
         char       *str;
         long       *num;
{
         if (*num == 2)
         {
           printf ("%%DELTREE-I-FILDEL, %s deleted (%d blocks)\n", str, 
                   block_size);
         }

         else if (*num == 1)
         {
           printf ("%%DELTREE-S-DIRDEL, directory file %s successfully", str);
           printf (" deleted\n"); 
         }

         else if (*num == 0)
         {
           printf ("%%DELTREE-I-SUCCESS, %s successfully deleted\n", str);
         }

         else if (*num == 3)
         {
           printf ("%%DELTREE-I-TOTALDEL, %d total files deleted", file_count);
           printf (" (%d total blocks)\n", total_blocks);
         }

         else if (*num == 10)
         {
           printf ("%%DELTREE-F-INVDIR, Invalid directory syntax, check");
           printf (" brackets and other delimeters\n");
           printf (" \\%s\\\n", str);
         }

         else if (*num == 11)
         {
           printf ("%%DELTREE-F-INVDIRFILE, Invalid directory file");
           printf (" within directory structure\n");
           printf (" \\%s\\\n", str);
         }

         else if (*num == 12)
         {
           printf ("%%DELTREE-F-NOEXIST, Nonexistant directory %s\n", str);
         }

         else if (*num == 13)
         {
           printf ("%%DELTREE-F-NOPRIV, No privilege to delete %s\n", str);
         }

         else if (*num == 15)
         {
           printf ("%%DELTREE-F-DIRERR, Error in directory parameter\n");
         }

         else if (*num == 16)
         {
           printf ("%%DELTREE-F-FILERR, Error in file %s, no ");
           printf ("message\n", str);
         }

         else if (*num == 18)
         {
           printf ("%%DELTREE-F-FILEOPEN, File %s is currently locked", str);
           printf (" by another user\n");
         }

         else if (*num == 20)
         {
           printf ("%%DELTREE-S-NOCONFIRM, Execution stopped by ");
           printf ("non-confirmed file deletion\n");
         }

         else if (*num == 21)
         {
           printf ("%%DELTREE-W-INVALANS, %s is an invalid answer", str);
         }
};
/*
         end of msg

***********************************************************************
         main program module
*/
         main ()
/*
         author
           jonathan c. baker
         date
           1 june 1992
         purpose
           to delete directory structures.
         description
           delete structure from bottom up.
         language
           c
         routines
     s     command_line  : retrieve command line qualifiers/parameter
     s     next_file     : get the next file

         local glossary
           answer        : answer to delete [/CONFIRM]
           check_confirm : check for confirmation
           command       : system routine command
           context       : LIB$FIND_FILE context
           dir           : directory to delete
           ent_desc      : entity descriptor
           file          : return filename
           flag          : flag variable
           flag_desc     : flag descriptor
           i             : control variable
           ptr           : pointer varible
           ret_desc      : return descriptor
           status        : return variable
           status1       : return variable
           temp          : temporary holding variable
/*
         define local variables
*/
{
         struct             dsc$descriptor_s ret_desc, ent_desc, 
                            del_desc, flag_desc;
         char               dir[max_file_len], *answer[max_file_len], *ptr, 
                            flag[10], file[max_file_len], *ptr1,
                            command[max_file_len], temp[max_file_len];
         long               status, context, status1, check_confirm, i;
/*
         initialize variables
*/
         file_count = 0;
         total_blocks = 0;
         del_desc.dsc$b_dtype = DSC$K_DTYPE_T;
         del_desc.dsc$b_class = DSC$K_CLASS_S;
         ret_desc.dsc$b_dtype = DSC$K_DTYPE_T;
         ret_desc.dsc$b_class = DSC$K_CLASS_S;
         ent_desc.dsc$b_dtype = DSC$K_DTYPE_T;
         ent_desc.dsc$b_class = DSC$K_CLASS_S;
         flag_desc.dsc$b_dtype = DSC$K_DTYPE_T;
         flag_desc.dsc$b_class = DSC$K_CLASS_S;
/*
         get command line information
*/
         strcpy (dir, command_line() );
/*
         write out about information
*/
         if (about)
         {
           printf("\n");
           printf("             DELTREE 1.2\n");
           printf("\n");
           printf(" The original DELTREE was inspired by Claude\n");
           printf(" W. Thompson of DIGITAL using a DCL procedure.\n");
           printf(" This DELTREE was written by Jonathan C. Baker\n");
           printf(" with added options using the C programming language.\n");
           printf(" Questions, problems or suggestions can be addressed\n");
           printf(" to Jon by mail, phone or his INTERNET address:\n");
           printf("        Naval Surface Warfare Center\n");
           printf("        Code N83\n");
           printf("        Dahlgren, VA 22448\n");
           printf("        540-653-8705\n");
           printf("        INTERNET:  jbaker@nswc.navy.mil\n");
           printf("\n");
           exit (0);
         }
/*
         check directory to delete for validity
*/
         status = check_dir (dir);
         if (status != 0)
           {
           msg (&status, dir);
           }
         else
           {
/*
         recursive deletion of files
*/
           status = del_files (dir);
           if (status != 0)
           {
             msg (&status, dir);
           }
           else
           {
             if (confirm == true)
             {
               check_confirm = true;
               while (check_confirm == true)
               {
                 printf ("\nDELETE  %s ? [N]:  ", dir);
                 gets (&answer);
                 ent_desc.dsc$a_pointer = answer;
                 ent_desc.dsc$w_length = strlen (answer);
                 ret_desc.dsc$a_pointer = answer;
                 ret_desc.dsc$w_length = max_file_len;
                 str$upcase (&ent_desc, &ret_desc);
/*
         remove preceeding spaces
*/
                 strcpy (&temp, &answer);
                 for (i = 0; temp[i] == ' '; i = i + 1);
                 strcpy (&answer, &temp[i]);
                 strcpy (&temp, &answer);
                 for (i = 0; (temp[i] != ' ' && temp[i] != '\0'); i = i + 1);
                 temp[i] = '\0';
                 strcpy (&answer, &temp);
/*
         set up alternate answers
*/
                 if ((answer[0] == '1') || (answer[0] == 'T'))
                 {
                   answer[0] = 'Y\0';
                 }
                 if ((answer[0] == '0') || (answer[0] == 'F') ||
                     (strlen(answer) == 0))
                 {
                   answer[0] = 'N\0';
                 }

                 if (strcmp(answer, "ALL\0") == 0)
                 {
                   check_confirm = false;
                   confirm = false;
                 }
                 else if (answer[0] == 'Y')
                 {
                   check_confirm = false;
                 }
                 else if (answer[0] == 'N')
                 {
/*
         stop with non-confirm status
*/
                   status1 = 20;
                   msg (&status1, dir);
                   exit (0);
                 }
                 else
                 {
                   status1 = 21;
                   msg (&status1, answer);
                 }
               } 
             }
/*
         translate if logical name
*/
             strcpy (temp, dir);
             if (strrchr (dir, ':') != null)
             {
               ret_desc.dsc$a_pointer = file;
               ret_desc.dsc$w_length = max_file_len;
               strcpy (command, dir);
               strncat (command, "*.*.*\0", 5);
               ent_desc.dsc$a_pointer = command;
               ent_desc.dsc$w_length = strlen (command);
               strcpy (flag, "MULTIPLE\0");
               flag_desc.dsc$a_pointer = flag;
               flag_desc.dsc$w_length = strlen (flag);
               context = 0;
               status = lib$find_file (&ent_desc, &ret_desc, 
                                       &context, 0, 0, 0, &flag_desc);
               status1 = lib$find_file_end (&context);

               if ((status == RMS$_NORMAL) || 
                   (status == RMS$_NMF) ||
                   (status == RMS$_OK1) ||
                   (status == RMS$_OK2))
               {
                 ptr = strchr (file, '*');
                 *ptr = '\0';
                 strcpy (dir, file);
               }
             }
/*
         put directory root in proper file format
*/
             ptr = strrchr (dir, '.');
             if (ptr == null)
             {
               ptr = strrchr (dir, '[');
               ptr1 = strrchr (dir, ']');
               ptr = ptr + 1;
               strcpy (file, "\0");
               strcat (file, ptr);
               ptr = ptr - 1;
               *ptr = '\0';
               strncat (dir, "[000000]\0", 8);
               strcat (dir, file); 
             }
             else
             {
               *ptr = ']';
             }
             ptr = strrchr (dir, ']');
             *ptr = '.';
             strncat (dir, "DIR;1\n", 5);
/*
         attempt setting protection
*/
             status = chmod (dir, 0006);
/*
         delete the entered directory
*/
             if (log == true)
             {
               status = blocks (dir);
               if (status != 0)
               {
                 msg (&status, dir);
                 exit (18);
               }
               del_desc.dsc$a_pointer = dir;
               del_desc.dsc$w_length = strlen (dir);
               status = lib$delete_file (&del_desc);
/*
         set appropriate status code and return messages
*/
               if (status == SS$_NORMAL)
               {
                 status = 2;
                 msg (&status, dir);
                 printf ("\n");
                 status = 3;
                 msg (&status, dir);
                 status = 0;
               }
               else
               {
                 if (status == 98970)
                 {
                   status = 13;
                 }
                 else
                 {
                   status = 18;
                 }
               }
             }
             else
             {
               del_desc.dsc$a_pointer = dir;
               del_desc.dsc$w_length = strlen (dir);
               status = lib$delete_file (&del_desc);
/*
         set appropriate status codes and return messages
*/
               if (status == SS$_NORMAL)
               {
                 status = 0;
               }
               else
               {
                 if (status == 98970)
                 {
                   status = 13;
                 }
                 else
                 {
                   status = 18;
                 }
               }
             }
             strcpy (dir, temp);
             msg (&status, dir);
           }
         }
/*
         end of main

****************************************************************
*/
};
