#include	<ssdef.h>
#include	<lib$routines.h>

#include <stdio.h>
#include <stdlib.h>		/* should have atof & getopt */
#include "tiffiop.h"

#ifndef BINMODE
#define	BINMODE
#endif

#ifndef EXIT_SUCCESS
#define EXIT_SUCCESS	0
#endif
#ifndef EXIT_FAILURE
#define EXIT_FAILURE	1
#endif

TIFF	*inp_fax;
#define XSIZE		1728
char	rowbuf[TIFFhowmany(XSIZE,8)];
char	refbuf[TIFFhowmany(XSIZE,8)];

int	verbose;
int	stretch;
uint16	badfaxrun;
uint32	badfaxlines;

int	copyFaxFile(TIFF* tifin, TIFF* tifout);
static	void usage(void);

static tsize_t
DummyReadProc(thandle_t fd, tdata_t buf, tsize_t size)
{
	(void) fd; (void) buf; (void) size;
	return (0);
}

static tsize_t
DummyWriteProc(thandle_t fd, tdata_t buf, tsize_t size)
{
	(void) fd; (void) buf; (void) size;
	return (size);
}

int
main(int argc, char* argv[])
{
TIFF	*out_tif = NULL;
TIFFErrorHandler whandler;
int compression = COMPRESSION_CCITTFAX3,
	fillorder = FILLORDER_LSB2MSB,
	photometric = PHOTOMETRIC_MINISWHITE,
	mode = FAXMODE_CLASSF;
uint32 group3options = GROUP3OPT_FILLBITS|GROUP3OPT_2DENCODING;
int	rows,c,pn, npages;
extern	int optind;
extern	char* optarg;

	/* smuggle a descriptor out of the library */
	if ( !(inp_fax = TIFFClientOpen("(FakeInput)", "w", (thandle_t) -1,
				 DummyReadProc, DummyWriteProc,
				 NULL, NULL, NULL, NULL, NULL)) )
		return	EXIT_FAILURE;

	inp_fax->tif_mode = O_RDONLY;

	TIFFSetField(inp_fax, TIFFTAG_IMAGEWIDTH,	XSIZE);
	TIFFSetField(inp_fax, TIFFTAG_SAMPLESPERPIXEL,	1);
	TIFFSetField(inp_fax, TIFFTAG_BITSPERSAMPLE,	1);
	TIFFSetField(inp_fax, TIFFTAG_FILLORDER,	FILLORDER_LSB2MSB);
	TIFFSetField(inp_fax, TIFFTAG_PLANARCONFIG,	PLANARCONFIG_CONTIG);
	TIFFSetField(inp_fax, TIFFTAG_PHOTOMETRIC,	PHOTOMETRIC_MINISWHITE);
	TIFFSetField(inp_fax, TIFFTAG_YRESOLUTION,	196.);
	TIFFSetField(inp_fax, TIFFTAG_RESOLUTIONUNIT,	RESUNIT_INCH);

	inp_fax->tif_scanlinesize = TIFFScanlineSize(inp_fax);

	while ((c = getopt(argc, argv, "R:2BLMW14cflmpsvwz")) != -1)
		{
		switch (c) {
			/* input-related options */
		case '2':		/* input is 2d-encoded */
			TIFFSetField(inp_fax,
			    TIFFTAG_GROUP3OPTIONS, GROUP3OPT_2DENCODING);
			break;
		case 'B':		/* input has 0 mean black */
			TIFFSetField(inp_fax,
			    TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_MINISBLACK);
			break;
		case 'L':		/* input has lsb-to-msb fillorder */
			TIFFSetField(inp_fax,
			    TIFFTAG_FILLORDER, FILLORDER_LSB2MSB);
			break;
		case 'M':		/* input has msb-to-lsb fillorder */
			TIFFSetField(inp_fax,
			    TIFFTAG_FILLORDER, FILLORDER_MSB2LSB);
			break;
		case 'R':		/* input resolution */
			TIFFSetField(inp_fax,
			    TIFFTAG_YRESOLUTION, atof(optarg));
			break;
		case 'W':		/* input has 0 mean white */
			TIFFSetField(inp_fax,
			    TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_MINISWHITE);
			break;

			/* output-related options */
		case '1':		/* generate 1d-encoded output */
			group3options &= ~GROUP3OPT_2DENCODING;
			break;
		case '4':		/* generate g4-encoded output */
			compression = COMPRESSION_CCITTFAX4;
			break;
		case 'c':		/* generate "classic" g3 format */
			mode = FAXMODE_CLASSIC;
			break;
		case 'f':		/* generate Class F format */
			mode = FAXMODE_CLASSF;
			break;
		case 'm':		/* output's fillorder is msb-to-lsb */
			fillorder = FILLORDER_MSB2LSB;
			break;
		case 'p':		/* zero pad output scanline EOLs */
			group3options &= ~GROUP3OPT_FILLBITS;
			break;
		case 's':		/* stretch image by dup'ng scanlines */
			stretch = 1;
			break;
		case 'w':		/* undocumented -- for testing */
			photometric = PHOTOMETRIC_MINISBLACK;
			break;

		case 'z':		/* undocumented -- for testing */
			compression = COMPRESSION_LZW;
			break;

		case 'v':		/* -v for info */
			verbose++;
			break;
		case '?':
			usage();
			/*NOTREACHED*/
		}
	}

	npages = argc - optind;
	if ( npages < 1 )
		usage();

	if ( 0 >= (inp_fax->tif_fd = open(argv[optind],O_RDONLY,0,"rfm=fix", "mrs=512", "bls=512")) )
		{
		perror("open output file");
		return	EXIT_FAILURE;
		}

	if ( !(out_tif = TIFFOpen(argv[optind+1], "w")) )
		{
		perror("open input file");
		return	EXIT_FAILURE;
		}

	inp_fax->tif_readproc = out_tif->tif_readproc;	/* XXX */
	inp_fax->tif_writeproc = out_tif->tif_writeproc;	/* XXX */
	inp_fax->tif_seekproc = out_tif->tif_seekproc;	/* XXX */
	inp_fax->tif_closeproc = out_tif->tif_closeproc;	/* XXX */
	inp_fax->tif_sizeproc = out_tif->tif_sizeproc;	/* XXX */
	inp_fax->tif_mapproc = out_tif->tif_mapproc;	/* XXX */
	inp_fax->tif_unmapproc = out_tif->tif_unmapproc;	/* XXX */

	TIFFSetField(inp_fax, TIFFTAG_COMPRESSION, COMPRESSION_CCITTFAX3);

	inp_fax->tif_clientdata = (thandle_t) inp_fax->tif_fd;
	inp_fax->tif_name = argv[optind];
	TIFFSetField(out_tif, TIFFTAG_IMAGEWIDTH, XSIZE);
	TIFFSetField(out_tif, TIFFTAG_BITSPERSAMPLE, 1);
	TIFFSetField(out_tif, TIFFTAG_COMPRESSION, compression);
	TIFFSetField(out_tif, TIFFTAG_PHOTOMETRIC, photometric);
	TIFFSetField(out_tif, TIFFTAG_ORIENTATION, ORIENTATION_TOPLEFT);
	TIFFSetField(out_tif, TIFFTAG_SAMPLESPERPIXEL, 1);
	if (compression == COMPRESSION_CCITTFAX3)
		{
		TIFFSetField(out_tif, TIFFTAG_GROUP3OPTIONS, group3options);
		TIFFSetField(out_tif, TIFFTAG_FAXMODE, mode);
		}
	if (compression == COMPRESSION_CCITTFAX3 ||
		    compression == COMPRESSION_CCITTFAX4)
			TIFFSetField(out_tif, TIFFTAG_ROWSPERSTRIP, -1L);
	else	TIFFSetField(out_tif, TIFFTAG_ROWSPERSTRIP,
			    TIFFDefaultStripSize(out_tif, 0));

	TIFFSetField(out_tif, TIFFTAG_PLANARCONFIG, PLANARCONFIG_CONTIG);
	TIFFSetField(out_tif, TIFFTAG_FILLORDER, fillorder);
	TIFFSetField(out_tif, TIFFTAG_SOFTWARE, "fax2tiff");
	TIFFSetField(out_tif, TIFFTAG_XRESOLUTION, 204.0);
		if (!stretch) {
			float yres;
			TIFFGetField(inp_fax, TIFFTAG_YRESOLUTION, &yres);
			TIFFSetField(out_tif, TIFFTAG_YRESOLUTION, yres);
		} else
			TIFFSetField(out_tif, TIFFTAG_YRESOLUTION, 196.);
		TIFFSetField(out_tif, TIFFTAG_RESOLUTIONUNIT, RESUNIT_INCH);
		TIFFSetField(out_tif, TIFFTAG_PAGENUMBER, pn+1, npages);

		rows = copyFaxFile(inp_fax, out_tif);
		close(inp_fax->tif_fd);

		TIFFSetField(out_tif, TIFFTAG_IMAGELENGTH, rows);

			fprintf(stderr, "%s:\n", argv[optind]);
			fprintf(stderr, "%d rows in input\n", rows);
			fprintf(stderr, "%ld total bad rows\n",
			    (long) badfaxlines);
			fprintf(stderr, "%d max consecutive bad rows\n", badfaxrun);
		if (compression == COMPRESSION_CCITTFAX3 &&
		    mode == FAXMODE_CLASSF) {
			TIFFSetField(out_tif, TIFFTAG_BADFAXLINES, badfaxlines);
			TIFFSetField(out_tif, TIFFTAG_CLEANFAXDATA, badfaxlines ?
			    CLEANFAXDATA_REGENERATED : CLEANFAXDATA_CLEAN);
			TIFFSetField(out_tif, TIFFTAG_CONSECUTIVEBADFAXLINES, badfaxrun);
		}
		TIFFWriteDirectory(out_tif);

	TIFFClose(out_tif);
	return (EXIT_SUCCESS);
}

int
copyFaxFile(TIFF* tifin, TIFF* tifout)
{
	uint32 row;
	uint16 badrun;
	int ok;

	tifin->tif_rawdatasize = TIFFGetFileSize(tifin);
	tifin->tif_rawdata = _TIFFmalloc(tifin->tif_rawdatasize);

	if (!ReadOK(tifin, tifin->tif_rawdata, tifin->tif_rawdatasize))
		{
		perror(read);
		return (0);
		}
	tifin->tif_rawcp = tifin->tif_rawdata;
	tifin->tif_rawcc = tifin->tif_rawdatasize;

	(*tifin->tif_setupdecode)(tifin);
	(*tifin->tif_predecode)(tifin, (tsample_t) 0);
	tifin->tif_row = 0;
	badfaxlines = 0;
	badfaxrun = 0;

	_TIFFmemset(refbuf, 0, sizeof (refbuf));
	row = 0;
	badrun = 0;		/* current run of bad lines */
	while (tifin->tif_rawcc > 0) {
		ok = (*tifin->tif_decoderow)(tifin, (tdata_t) rowbuf, 
					     sizeof (rowbuf), 0);
		if (!ok) {
			badfaxlines++;
			badrun++;
			/* regenerate line from previous good line */
			_TIFFmemcpy(rowbuf, refbuf, sizeof (rowbuf));
		} else {
			if (badrun > badfaxrun)
				badfaxrun = badrun;
			badrun = 0;
			_TIFFmemcpy(refbuf, rowbuf, sizeof (rowbuf));
		}
		tifin->tif_row++;

		if (TIFFWriteScanline(tifout, rowbuf, row, 0) < 0) {
			printf("%s: Write error at row %ld.\n",
			    tifout->tif_name, (long) row);
			break;
		}
		row++;
		if (stretch) {
			if (TIFFWriteScanline(tifout, rowbuf, row, 0) < 0) {
				printf("%s: Write error at row %ld.\n",
				    tifout->tif_name, (long) row);
				break;
			}
			row++;
		}
	}
	if (badrun > badfaxrun)
		badfaxrun = badrun;
	_TIFFfree(tifin->tif_rawdata);
	return (row);
}

char* stuff[] = {
"usage: fax2tiff [options] input.gf3 output.tiff",
"where options are:",
" -2		input data is 2d encoded",
" -B		input data has min 0 means black",
" -L		input data has LSB2MSB bit order (default)",
" -M		input data has MSB2LSB bit order",
" -W		input data has min 0 means white (default)",
" -R #		input data has # resolution (lines/inch) (default is 196)",
"",
" -1		generate 1d-encoded output (default is G3 2d)",
" -4		generate G4-encoded output (default is G3 2D)",
" -c		generate \"classic\" TIFF format (default is TIFF/F)",
" -f		generate TIFF Class F (TIFF/F) format (default)",
" -m		output fill order is MSB2LSB (default is LSB2MSB)",
" -p		do not byte-align EOL codes in output (default is byte-align)",
" -s		stretch image by duplicating scanlines",
" -v		print information about conversion work",
NULL
};

static void
usage(void)
{
	char buf[BUFSIZ];
	int i;

	setbuf(stderr, buf);
	for (i = 0; stuff[i] != NULL; i++)
		fprintf(stderr, "%s\n", stuff[i]);
	exit(EXIT_FAILURE);
}
