[ident ('V3.5-1; Copyright  1993 Anglia Polytechnic University'),
inherit ('sys$library:starlet')]
module tscon_display;

{ Note: RTTDRIVER at VMS V4-V5.0 does not support TRM$_ESCTRMOVR }
{ so this adjusts its terminal read scheme accordingly }

const
	cr = chr(13);
	ctrl_z = chr(26);
	esc = chr(27);
	bs = chr(8);
	max_text_size = 1500;
	max_term_items = 3; { in item list }

	read_escapes = false; { Whether to have escape string handling }

	oob_timeout_cs = 90; { Time allowed for noticing an OOB char. }
	remote_tx_delay_secs = 3; { Leeway allowed for remote links }

	min_poll_cs = 1;
	max_poll_cs = 100;
        default_term_poll_cs = 10;
	default_batch_poll_cs = 2;

	%include 'appl_vmsdef:smgsizes' { Missing from starlet }

type
	%include 'appl_vmsdef:datatypes'

	text_type = varying [max_text_size] of char;
	fixed_text_type = packed array [1..max_text_size] of char;
	string_desc = record
			    length : integer;
			    addr : ^fixed_text_type;
		      end;

	{ An input stream }
	class_type = (uninit, smg, terminal, command);
	input_ptr = ^input_entry;
	input_entry = record
			prev : input_ptr;
			keyboard : unsigned;
			attr : smg$type;
			terminal_device : boolean;
			class : class_type;
			file_spec : text_type;
		      end;

	{ Terminal $qio item list entry }
	term_item = record
			dsize : _uword;
			code : _uword;
			value : unsigned;
			retlen : unsigned;
		    end;

var
	dbg_force_batch : [global] unsigned; { For Debugger access }
	terminal_oob_only, { Only used for out-of-band char. checks }
	force_batch, remote_terminal, entirely_batch_mode : [volatile] boolean;
	poll_cs : [volatile] integer;
	timeout_efn : [volatile, static] unsigned := 0;

	data_logging : boolean;

	disconnect_char : [volatile] char;

	cur_input : [volatile] input_ptr := nil;

	{ Command specified on command line }
	c_line_used : boolean; { Whether it's been used }
	command_line_command : text_type;

	{ Direct-terminal-I/O data }
	term_iosb : [volatile, unsafe] record
			status : _uword;
			data_size : _uword;
			term_char : _ubyte;
			term_char2 : char;
			term_size : _uword;
		    end;
	term_efn : [volatile] unsigned;
	term_itmlst : array [1..max_term_items] of term_item;
	timeout_index, esctrmovr_index : integer;
	oob_timer_efn : [volatile] unsigned;
	term_page_size : unsigned;
	term_chan : [volatile] _uword;
	oob_timer_running, oob_ast_set, oob_ast_triggered,
	timer_expired, timed_out : [volatile] boolean;

	{ General I/O data }
	last_input_line, pending_text, pending_log_text : text_type := '';
	ansi_output,
	last_input_null, server_waiting, write_done, last_write_done,
	prompting, prompt_pw, last_prompt_pw : boolean := false;
	
	smg$_eof,
	tscon_disconnecting, tscon_ast_error, tscon_normal,
	tscon_write_error : [external, value] unsigned;

%include 'appl_vmsdef:rtldef'
%include 'appl_vmsdef:smgrtndef'

[asynchronous] function ccat_setimer
			(centisecs : unsigned;
			 var efn : [volatile] unsigned;
			 %immed [asynchronous, unbound] procedure astrtn
			): unsigned; external;
[asynchronous] function ccat_sys_fao
			(%descr ctrstr : [readonly] varying [csize] of char;
			 %ref reslen : unsigned := %immed 0;
			 %descr resstr : varying [rsize] of char;
			 %immed p1 : [list, unsafe] unsigned
			): unsigned; external;
function ccat_term_get_char (var char1, char2, char3 : unsigned
			    ): unsigned; external;
function ccat_term_set_char (char1, char2, char3 : unsigned
			    ): unsigned; external;

function tscon_note_connected: boolean; external;
function tscon_data_log (line : varying [lsize] of char
		        ): unsigned; external;

{******************************************************************************}
procedure signal_error (sts1, sts2 : unsigned);
			
  begin
	lib$signal (sts1, 0, sts2, 0);
  end; { signal_error }

{******************************************************************************}
function translate_lnm (lnm : varying [lsize] of char): text_type;
{ Translate a logical name. }
{ Return null string if there's no translation. }
  const
	num_items = 2;
  type
	%include 'appl_vmsdef:itmlst'
  var
	i : integer;
	sts : unsigned;
	result : text_type;
	item_list : array [1..num_items] of itmlst_type;
  begin
	result.length := 0;

	item_list := zero;
	i := 1;
	with item_list[i] do
	  begin
		code := lnm$_string;
		size := max_text_size;
		dataaddr := iaddress (result.body);
		sizeaddr := iaddress (result.length);
		i := i + 1;
	  end;

	sts := $trnlnm (tabnam := 'LNM$FILE_DEV',
			lognam := lnm,
			itmlst := item_list);
	if not odd (sts) then result.length := 0;

	translate_lnm := result;	    
  end; { translate_lnm }

{******************************************************************************}
procedure add_pending_log_text (str : varying [ssize] of char);
  var
	copy : boolean;
	idx : integer;
  begin
	idx := 1;
	while idx <= ssize do
	  begin
		copy := true;

		{ Edit out <BS><SP><BS> i.e. echo of <DEL> }
		if (str.body[idx] = bs)
		and (idx+2 <= ssize) then
		  begin
			if (str.body[idx+1] = ' ')
			and (str.body[idx+2] = bs) then
			  begin
				copy := false;
				idx := idx + 2; { Skip the BS/SP }
				if pending_log_text.length > 0 then
				    pending_log_text.length := 
					pending_log_text.length - 1;
			  end;
		  end;

		if copy then
		    pending_log_text := pending_log_text + str.body[idx];
			
		idx := idx + 1;
	  end; { while }
  end; { add_pending_log_text }

{******************************************************************************}
[asynchronous, unbound] procedure terminal_timeout_ast;
  begin
	timer_expired := true;
	if term_iosb.status = 0 then { i/o still uncompleted }
	  begin
		$cancel (chan := term_chan);
		$setef (efn := term_efn);
		if term_iosb.status = 0 then term_iosb.status := ss$_timeout;
		oob_ast_set := false; { Now cancelled }
		timed_out := true;
	  end;
  end; { terminal_timeout_ast }

{******************************************************************************}
[asynchronous, unbound] procedure timeout_ast;
  begin
	timer_expired := true;
  end; { timeout_ast }

{******************************************************************************}
procedure set_timer (period : integer; just_wait : boolean);
  begin
	timer_expired := false;
	timed_out := false;
	if just_wait then ccat_setimer (period, timeout_efn, timeout_ast)
		     else ccat_setimer (period, timeout_efn, terminal_timeout_ast);
  end; { set_timer }

{******************************************************************************}
procedure cancel_timer;
  begin
	if not timer_expired then $cantim (reqidt := timeout_efn);
  end; { cancel_timer }

{******************************************************************************}
procedure wait_for_timer;
  begin
	$waitfr (efn := timeout_efn);
  end; { wait_for_timer }

{******************************************************************************}
procedure delete_input (var block : input_ptr);
  begin
	with block^ do
	  begin
		if keyboard <> 0 then
		    smg$delete_virtual_keyboard (keyboard_id := keyboard);
	  end;
	dispose (block);
  end; { delete_input }

{******************************************************************************}
procedure unwind_input;
{ Unwind input stack one level }
  var
	prev_input : input_ptr;
  begin
	prev_input := cur_input;
	cur_input := prev_input^.prev;
	delete_input (prev_input);
  end; { unwind_input }

{******************************************************************************}
function setup_input (var input : [volatile] input_ptr;
		      input_spec : varying [isize] of char;
		      signal_error : boolean
		     ): unsigned;
  var
	sts : unsigned;
	def_spec : text_type;
	tscon_erropen, tscon_create_kb : [value, external] unsigned;
  begin
	{ Find site-supplied default file-spec }
	def_spec := translate_lnm ('TSCON_DEFAULT_FILE_SPEC');
	if def_spec.length = 0 then def_spec := 'TSCON.DAT';
	
	input^ := zero;
	with input^ do
	  begin
		sts := smg$create_virtual_keyboard
			    (new_keyboard_id := keyboard,
			     filespec := input_spec,
			     resultant_filespec := file_spec,
			     default_filespec := def_spec);
		if odd (sts) then
		  sts := smg$get_keyboard_attributes
			    (keyboard_id := keyboard,
			     p_kit := attr,
			     p_kit_size := smg$s_keyboard_info_block)
		else if signal_error then
		  begin
			{ If facility = SS or RMS, handle differently }
			if uand (sts, %x0fff0000) <= %x00010000 then
			  begin
				lib$signal (tscon_erropen, 2,
					    input_spec.length,
					    iaddress (input_spec.body),
					    sts);
			  end
			else
			  begin
				lib$signal (tscon_create_kb, 0, sts);
			  end;
			$exit (uor (sts,sts$m_inhib_msg));
		  end;

		if odd (sts) then
		  begin
			class := smg;
			{ VMS V5.0: devchar seems not to be set }
			if (uand (attr.smg$l_devchar, dev$m_trm) <> 0)
			or (attr.smg$b_devclass = dc$_term) then
			  begin
				terminal_device := true;
				if not force_batch then
				    class := terminal;
			  end;
		  end;
	  end;
	setup_input := sts;
  end; { setup_input }

{******************************************************************************}
procedure setup_terminal (block : input_ptr);
  var
	i : integer;
	temp, sts : unsigned;
	tscon_assignfail : [external, value] unsigned;
  begin
	if term_chan = 0 then
	  begin
		sts := $assign (devnam := block^.file_spec,
				chan := term_chan);
		if not odd (sts) then
			lib$signal (tscon_assignfail,
				    2,
				    block^.file_spec.length,
				    iaddress (block^.file_spec.body),
				    sts);
		lib$get_ef (term_efn);

		lib$getdvi (item_code := dvi$_devdepend,
			    chan := term_chan,
			    out_value := temp);
		term_page_size := temp div %x1000000; { High byte }

		{ Test if this is a remote terminal (SET HOST) }
		lib$getdvi (item_code := dvi$_devchar2,
			    chan := term_chan,
			    out_value := temp);
		temp := uand (temp, dev$m_rtt);
		remote_terminal := (temp <> 0);

		{ Set up item list }
		term_itmlst := zero;
		i := 1;
		with term_itmlst[i] do
		  begin
			code := trm$_modifiers;
			value :=  trm$m_tm_noecho
				+ trm$m_tm_noedit
				+ trm$m_tm_nofiltr;
			if read_escapes
			and (not remote_terminal) then
			    value := value + trm$m_tm_escape;
			i := i + 1;
		  end;
		with term_itmlst[i] do
		  begin
			timeout_index := i; { Needed in read_term }
			code := trm$_timeout;
			{ Value will be set in read_term }
			i := i + 1;
		  end;
		if not remote_terminal then { Not RTTDRIVER }
		  with term_itmlst[i] do
		    begin
			esctrmovr_index := i; { Needed in read_term }
			code := trm$_esctrmovr;
			{ Value will be set in read_term }
			i := i + 1;
		    end;
	end;
  end; { setup_term }

{******************************************************************************}
[global] function tscon_display_init (input_spec : varying [isize] of char;
				      poll_interval : integer;
				      idle_timeout_interval : integer;
				      command_str : varying [csize] of char;
				      force : boolean;
				      chars : boolean;
				      disc_char : char;
				      logging : boolean
				     ): boolean;
{ Set up display module. }
{ Return boolean for entirely-batch mode. }
{ Input_spec = "" => use sys$command. }
{ Poll = 0 => use default value. }
{ Idle_timeout parameter not used at the moment. }
{ Command is a command to be sent to the server. }
  var
	sts : unsigned;
	file_input, cline_input, sys$command, sys$input : input_ptr;
  begin
	{ Clear all input streams }
	sys$command := nil;
	sys$input := nil;
	file_input := nil;
	cline_input := nil;
	cur_input := nil;

	disconnect_char := disc_char;

	{ Remember if data logging is required }
	data_logging := logging;

	force_batch := odd (dbg_force_batch); { Debugger-override flag? }
	if force then { Forcing a mode }
	  if not chars then force_batch := true;

	if command_str.length > 0 then
	  begin { Command-line command specified }
		new (cline_input);
		cline_input^ := zero;
		cline_input^.class := command;
		command_line_command := command_str; { Save command }
		c_line_used := false; { Not yet sent to server }
		cur_input := cline_input;
	  end;

	new (sys$command);
	sts := setup_input (sys$command, 'SYS$COMMAND', true);
	if not odd (sts) then
	  begin
		dispose (sys$command);
		sys$command := nil;
	  end;

	terminal_oob_only := true; { Assume no terminal input/output }

	if input_spec.length > 0 then
	  begin { Input script file given }
	  	new (file_input);
		sts := setup_input (file_input, input_spec, true);

		if odd (sts) then
		  begin
			{ Add to input stack, pushing any prev. stream }
			if cur_input <> nil then file_input^.prev := cur_input;
			cur_input := file_input;
		  end
		else dispose (file_input);
	  end
	else if (cline_input = nil)
	    and (sys$command <> nil) then
	  begin { No command-line input - use sys$command instead }

		{ Add to input stack, pushing any prev. stream }
		if cur_input <> nil then sys$command^.prev := cur_input;
		cur_input := sys$command;

		{ Check terminal usage mode }
		if sys$command^.terminal_device
		and (sys$command^.class = terminal) then
		  begin
			terminal_oob_only := false;
			     { It'll be used for input & output }
		  end;
	  end;

	{ Test if sys$input <> current input and isn't a terminal }
	if cur_input <> nil then { There must be a stream }
	  begin
		new (sys$input);
		sts := setup_input (sys$input, 'SYS$INPUT', false);
		if odd (sts) 
		and (sys$input^.file_spec <> cur_input^.file_spec)
		and (sys$input^.terminal_device <> cur_input^.terminal_device)
		and (sys$input^.class = smg) then
		  begin { Push the input stack & make sys$input the current stream }
			sys$input^.prev := cur_input;
			cur_input := sys$input;
		  end
		else delete_input (sys$input);
	  end;
	
	if sys$command <> nil then
	  if sys$command^.terminal_device then
		setup_terminal (sys$command);

	{ Entirely-batch mode if there's no terminal i/o }
	entirely_batch_mode := (term_chan = 0)
			    or terminal_oob_only;

	{ Keyboard polling interval }
	if poll_interval = 0 then
	  begin { Set default }
		if entirely_batch_mode then poll_cs := default_batch_poll_cs
				       else poll_cs := default_term_poll_cs;
	  end;
	poll_cs := max (min_poll_cs, poll_cs);
	poll_cs := min (max_poll_cs, poll_cs);

	tscon_display_init := entirely_batch_mode;
  end; { tscon_display_init }

{******************************************************************************}
procedure pasthru_mode (state : boolean);
{ Set/clear pasthru mode. }
{ This also sets an image exit handler to ensure reset. }
  var
	cur_state : boolean;
	char1, char2, char3 : unsigned;
	sts, temp : unsigned;
  begin
	sts := ccat_term_get_char (char1, char2, char3);
	if not odd (sts) then lib$stop (sts);

	temp := uand (char3, tt2$m_pasthru);
	cur_state := (temp <> 0);

	if state <> cur_state then
	  begin { Change pasthru mode }
		char3 := uxor (char3, tt2$m_pasthru); { Flip the bit }
	    	sts := ccat_term_set_char (char1, char2, char3);
		if not odd (sts) then lib$stop (sts);
	  end;
  end; { pasthru_mode }

{******************************************************************************}
[asynchronous, unbound] procedure oob_timeout_ast;
  begin
	oob_timer_running := false;
	lib$signal (tscon_disconnecting, 0);
  end; { oob_timeout_ast }

{******************************************************************************}
[asynchronous, unbound] procedure oob_ast;
{ An out-of-band key was pressed. }
{ Set timer for detecting lack of response to OOB key. }
  begin
	oob_ast_triggered := true;
	if not terminal_oob_only then
	  begin
		oob_timer_running := true;
		ccat_setimer (oob_timeout_cs, oob_timer_efn, oob_timeout_ast);
	  end;
  end; { oob_ast }

{******************************************************************************}
procedure set_oob_ast (on : boolean);
{ Toggle out-of-band key recognition. }
  var
	func, sts : unsigned;
	oob_mask : _uquad;

  [asynchronous] procedure lib$insv (src : unsigned;
				     pos : unsigned;
				     size : unsigned;
				     var value : unsigned); external;

  [asynchronous,external(sys$qiow)] function oob_qiow (
	%immed efn : unsigned := %immed 0;
	%immed chan : integer;
	%immed func : unsigned;
	var iosb : [volatile] _uquad := %immed 0;
	%immed [unbound, asynchronous] procedure astadr := %immed 0;
	%immed astprm : unsigned := %immed 0;
	%immed [asynchronous, unbound] procedure p1 := %immed 0;
	%ref p2 : _uquad := %immed 0;
	%immed p3 : integer := %immed 0;
	%immed p4 : integer := %immed 0;
	%immed p5 : integer := %immed 0;
	%immed p6 : integer := %immed 0) : unsigned; external;

  begin
	oob_ast_triggered := false;

	if on then 
	  begin
		oob_ast_set := false;

		{ Set out-of-band AST control char mask }
		oob_mask := zero;
		lib$insv (1, ord (disconnect_char), 1, oob_mask.l1);

		{ If this is the only use for the terminal, watch for ^z too }
		if terminal_oob_only then
		  begin
			lib$insv (1, ord (ctrl_z), 1, oob_mask.l1);
			func := io$_setmode+io$m_outband;
				    { Don't keep in data stream }
		  end
		else func := io$_setmode+io$m_outband+io$m_include;
				{ Pass the keystroke on for further processing }
				{ e.g. as a terminator }

		sts := oob_qiow (
			   chan   := term_chan,
			   efn    := term_efn,
			   func   := func,
			   iosb   := term_iosb,
			   p1	  := oob_ast,
		    	   p2	  := oob_mask
			  );
		if odd (sts) then oob_ast_set := true;
	  end
	else
	  begin
		sts := oob_qiow (
			   chan   := term_chan,
			   efn    := term_efn,
		           func   := io$_setmode+io$m_outband+io$m_include,
		           iosb   := term_iosb
			  );
		oob_ast_set := true;
	  end;
  end; { set_oob_ast }

{******************************************************************************}
function read_term (var buffer : varying [bsize] of char;
		    var break_flag : boolean;
		    var terminator : smg$terminator_code
		   ): unsigned;
{ Read from terminal. }
{ Significant returns:  sts = smg$_eof  }
{		        sts = tscon_disconnecting }
{ Buff contains data + terminator }
  var
	buff_size, itmlst_size : integer;
	sts : unsigned;
	temp : [unsafe] smg$terminator_code;
  begin
	buff_size := bsize;

	with term_itmlst[timeout_index] do
	  begin
		value := 0; { Assume immediate timeout - read just typeahead }
		if server_waiting then { It's ready for input }
		  begin
			{ Specify a timeout that's longer than the poll time. }
			{ (See the I/O Ref. Man. notes on timeout secs.) }
			value := (max_poll_cs+199) div 100;
			if remote_terminal then value := value + remote_tx_delay_secs;
			    { Heuristic: allow for transmission delays }
		  end;
	  end;

	if not remote_terminal then
	  begin
		with term_itmlst[esctrmovr_index] do
		  begin
			value := bsize - 1;
			    { Reserve 1st byte for the char. to be read }
		  end;
		itmlst_size := max_term_items * size(term_item);
	  end
	else { Using RTTDRIVER }
	  begin
		buff_size := 1; { Read a single char }
		{ Assume esctrmovr item = last entry. }
		{ Omit that item list entry. }
		itmlst_size := (max_term_items-1) * size(term_item);
	  end;

  	sts := $qiow (chan   := term_chan,
		      efn    := term_efn,
		      func   := io$_readvblk+io$m_extend,
		      iosb   := term_iosb,
		      p1     := buffer.body,
		      p2     := buff_size,
		      p5     := iaddress (term_itmlst),
		      p6     := itmlst_size);

  	if odd (sts) then sts := term_iosb.status;
  	temp := term_iosb.term_char;
  	terminator := temp;
  	buffer.length := term_iosb.data_size + term_iosb.term_size;

	if odd (sts) then
	  begin
		if terminator = smg$k_trm_ctrlb then
		  begin
			break_flag := true;
			sts := tscon_normal;
		  end;

		if (buffer.length > 0)
		and (buffer.body[buffer.length] = disconnect_char) then
			sts := tscon_disconnecting;

		if terminator = ord (disconnect_char) then
			sts := tscon_disconnecting;
	  end;

	read_term := sts;
  end; { read_term }

{******************************************************************************}
function screen_input (var res_str : string_desc;
		       var res_len : _uword;
		       var break_flag : boolean
		      ): unsigned;
{ Read a character without echo from the current stream with a timeout. }
  var
	sts : unsigned;
	buff : text_type;
	terminator : smg$terminator_code;
  begin
	if not oob_ast_set then set_oob_ast (true);

	set_timer (poll_cs, false);

	pasthru_mode (true);

	if oob_timer_running then
	  begin
		$cantim (reqidt := oob_timer_efn);
		oob_timer_running := false;
	  end;

	sts := tscon_normal; { Keep compiler /usage happy }
	buff.length := 0;
	terminator := smg$k_trm_ctrlm;

	sts := read_term (buff, break_flag, terminator);

        cancel_timer;

	res_len := buff.length;
	res_str.addr^ := buff;

	if (sts = ss$_cancel)
	or (sts = ss$_abort)
	or (sts = ss$_timeout) then { timed-out }
	  begin
		sts := tscon_normal;
	  end
	else if not odd (sts) then
	  begin
		pasthru_mode (false);
		if oob_ast_set then set_oob_ast (false);
	  end;

	screen_input := sts;
  end; { screen_input }

{******************************************************************************}
function write_term (str : varying [ssize] of char;
		     newline : boolean): unsigned;
  var
	sts : unsigned;
	carcon : [unsafe] integer;
  begin
	if newline then carcon := %x01000000 { Postfix 1 CR/LF }
		   else carcon := 0;
	sts := $qiow (
		   chan   := term_chan,
		   efn    := term_efn,
		   func   := io$_writevblk,
		   iosb   := term_iosb,
		   p1     := str.body,
		   p2     := str.length,
		   p4     := carcon);
	if odd (sts) then sts := term_iosb.status;
	write_term := sts;
  end; { write_term }

{******************************************************************************}
procedure end_of_page;
{ Move the cursor to end-of-page on an ANSI terminal }
  var
	line : text_type;
  begin
	ccat_sys_fao ('[!UB;1H',, line, term_page_size);
	write_term (esc+line, false);
  end; { end_of_page }

{******************************************************************************}
[global] procedure tscon_purge_typeahead;
  var
	buff : fixed_text_type;
  begin
	if term_chan <> 0 then
	    $qiow (chan   := term_chan,
		   efn    := term_efn,
		   func   := io$_readvblk+io$m_noecho+io$m_purge+io$m_timed,
		   iosb   := term_iosb,
		   p1     := iaddress (buff),
		   p2     := size(buff),
		   p3	  := 0);
  end; { tscon_purge_typeahead }

{******************************************************************************}
function read_line (var buffer : varying [bsize] of char;
		    var break_flag : boolean;
		    var terminator : smg$terminator_code
		   ): unsigned;
{ Read from current keyboard using SMG. }
{ Significant returns:  sts = smg$_eof  }
{		        sts = tscon_disconnecting }
  var
	sts : unsigned;
  begin
	sts := smg$read_string (keyboard_id := cur_input^.keyboard,
			        received_text := buffer,
			        terminator_code := terminator,
			        prompt_string := pending_text);

	if odd (sts) then
	  begin
		if terminator = smg$k_trm_ctrlb then
		  begin
			break_flag := true;
		  end;

		if terminator = smg$k_trm_f10 then
		  begin
			terminator := smg$k_trm_ctrlz;
			sts := smg$_eof;
		  end;

		if terminator = ord (disconnect_char) then
		  begin
			buffer.length := 0;
			if cur_input^.class = terminal then
			  begin
				sts := smg$_eof;
				terminator := smg$k_trm_ctrlm;
			  end
			else
			  begin
				sts := tscon_disconnecting;
			  end;
		  end;
	  end;

	read_line := sts;
  end; { read_line }

{******************************************************************************}
procedure indirect_file (var line : varying [lsize] of char);
{ Line is "@<fspec>" }
  var
	sts : unsigned;
	input_file : text_type;
	temp_input : input_ptr;
	tscon_erropen : [external, value] unsigned;
  begin
	input_file := substr (line, 2, line.length-1); { Skip '@' }

	new (temp_input);
	sts := setup_input (temp_input, input_file, false);
	if not odd (sts) then
	  begin
		dispose (temp_input);
		lib$signal (tscon_erropen,
			    2,
			    input_file.length,
			    iaddress (input_file.body),
			    sts);
	  end
	else
	  begin
		temp_input^.prev := cur_input;
		cur_input := temp_input;
	  end;
  end; { indirect_file }

{******************************************************************************}
function get_unwind (var retry : boolean): unsigned;
{ Unwind input stack one level, test if there's more }
  begin
	unwind_input;

	if cur_input <> nil then { More stacked input }
	  begin
		retry := true; { Go read again }
		get_unwind := tscon_normal;
	  end
	else { End of all input }
	  begin
		retry := false;
		get_unwind := tscon_disconnecting;
	  end;
  end; { get_unwind }

{******************************************************************************}
function get_line (var buffer : varying [bsize] of char;
		   var break_flag : boolean;
		   var retry : boolean
		  ): unsigned;
  label 11;
  var
	sts : unsigned;
	terminator : smg$terminator_code;
	pos : integer;
  begin
	retry := false;
	buffer.length := 0;
	break_flag := false;
	terminator := zero;

	if cur_input = nil then
	  begin { Input was previously finished }
		sts := tscon_disconnecting;
		goto 11;
	  end;

	if cur_input^.class = command then
	  begin { Current input is a command line }
		if c_line_used then
		  begin { Already sent the command line }
			sts := get_unwind (retry);
			goto 11;
		  end
		else { Send the command line }
		  begin
			buffer := command_line_command;
			terminator := smg$k_trm_ctrlm;
			c_line_used := true;
			sts := tscon_normal;
			last_input_line := pending_text + buffer;
				{ Save <prompt><response> }
			goto 11;
		  end;
	  end;

	if cur_input^.class = terminal then
	  begin { Use terminal input logic }
		{ Force break out of non-terminal-class logic loop }
		sts := tscon_normal;
		goto 11;
	  end;

	{ Normal case - read from file }
	sts := read_line (buffer, break_flag, terminator);

	if sts = smg$_eof then
	  begin
		sts := get_unwind (retry);
		goto 11;
	  end;

	if not odd (sts) then goto 11; { Unexpected error }

	if buffer.length > 0 then
	  begin
		pos := index (buffer, '!');
		if pos > 0 then { '!' found - check for '!!' }
		  begin
			if pos < buffer.length then
			  if buffer.body[pos+1] = '!' then
			    begin { Convert '!!" to '!' }
				buffer := substr (buffer, 1, pos-1)
					+ substr (buffer, pos+1,
						  buffer.length-pos);
				str$trim (buffer,buffer);
				pos := 0; { Show '!' now dealt with }
			    end;
		  end;
		if pos > 0 then { handle '!' }
		  begin
			if pos = 1 then { Line begins with '!' }
			  begin
				buffer.length := 0;
				retry := true;
			  end
			else
			  begin
				buffer := substr (buffer, 1, pos-1);
				str$trim (buffer,buffer);
			  end;
		  end;
	  end;

	if buffer.length > 0 then 
	  if buffer.body[1] = '@' then { indirect file }
	    begin
		indirect_file (buffer);
		retry := true;
	    end;

	{ Save <prompt><response> }
	last_input_line := pending_text + buffer;

    11:
	if odd (sts) then
	  begin
		{ Append terminator character }
		if (terminator > 0)
		and (terminator < 256) then
			buffer := buffer + chr(terminator);
	  end;

	get_line := sts;
  end; { get_line }

{******************************************************************************}
[global, unbound] function tscon_input (var res_str : string_desc;
				        var res_len : _uword;
				        var break_flag : boolean
				       ): unsigned;
{ General input routine }
  label 11;
  var
	sts : unsigned;
	retry : boolean;
	buffer : text_type;

  function tscon_mop_test_ds90: boolean; external;

  begin
	{ Must have connected to console to arrive here so declare it }
	if tscon_note_connected then { This is the first time }
	  begin { Supply the first CR which wakes up the server }
		last_input_null := false;
		if tscon_mop_test_ds90 then res_len := 0
				       else res_len := 1;
		res_str.addr^[1] := cr;
		tscon_input := tscon_normal;
		goto 11;
	  end;

	{ Ensure that the terminal is being watched for the disconnect char. }
	if terminal_oob_only
	and (term_chan <> 0) then
	  begin
		if not oob_ast_set then set_oob_ast (true);

		if oob_ast_triggered then
		  begin { It's time to go }
			if ansi_output then end_of_page;
			res_len := 0;
			tscon_input := tscon_disconnecting;
			goto 11;
		  end;
	  end;

	server_waiting := false;
	if last_input_null { Server wasn't waiting last time }
	and (not write_done) { No write since last call }
	and (not last_write_done) { No write since prev call }
	and (prompting) then { There's a prompt }
	    server_waiting := true; { Waiting for input }

	{ If the server is ready for input, forget about any ANSI output }
	if server_waiting then ansi_output := false;

	if cur_input <> nil then
	  if cur_input^.class = terminal then
	    begin { Use terminal input logic }
		if pending_text.length > 0 then
		  begin
			if data_logging then
			  begin { Save the pending frag. for data logging }
				add_pending_log_text (pending_text);
			  end;

			{ Ensure any pending text (prompt) is shown. }
			{ This typically occurs at change-over from }
			{ non-terminal to terminal input mode. }
			write_term (pending_text, false);
			pending_text.length := 0;
			{ Keep the prompting flag however }
		  end;
		tscon_input := screen_input (res_str, res_len, break_flag);
		goto 11;
	    end;

	set_timer (poll_cs, true);

	if server_waiting then
	  begin
		last_prompt_pw := prompt_pw; { For detection of login pw failure }
		retry := true;
		sts := tscon_normal;
		buffer.length := 0;

		while (retry) and (odd (sts)) do
		    sts := get_line (buffer, break_flag, retry);

		{ Set caller's buffer }
		res_len := buffer.length;
		res_str.addr^ := substr (buffer, 1, buffer.length);
		tscon_input := sts;
	  end
	else { Server not waiting for input }
	  begin
		res_len := 0;
		tscon_input := tscon_normal;
	  end;

	wait_for_timer; { This ensures similar behaviour in terminal }
		        { and batch mode with respect to timings. }

     11:
	last_input_null := (res_len = 0);
	last_write_done := write_done; { Remember prev. setting }
	write_done := false; { Set to detect when next write occurs }
  end; { tscon_input }

{******************************************************************************}
[global, unbound] function tscon_put_chars (var out_str : string_desc
					   ): unsigned;
{ Put a partial line. }
  var
	pos : integer;
	str : text_type;
	tscon_servpwfail : [value, external] unsigned;
  begin
	write_done := true;

	str := substr (out_str.addr^, 1, out_str.length);

	if str.length > 0 then
	  begin
		if str.body[1] = esc then ansi_output := true;

		prompt_pw := false;
		if str.length > 1 then
		  for pos := 1 to str.length-1 do
		    begin
			if (str.body[pos] = '#')
			and (str.body[pos+1] = ' ') then prompt_pw := true;
		    end;

		if last_prompt_pw
		and prompt_pw
		and entirely_batch_mode then lib$signal (tscon_servpwfail);

	        pending_text := pending_text + str;
	  end;

	{ Flag whether server may be prompting }
	prompting := (pending_text.length > 0);

	if cur_input <> nil then
	  if cur_input^.class = terminal then
	    begin
		if data_logging then
		  begin { Save the pending frag. for data logging }
			add_pending_log_text (pending_text);
		  end;

		write_term (pending_text, false);
		pending_text.length := 0;
		{ Keep the prompting flag however }
	    end;

	tscon_put_chars := tscon_normal;
  end; { tscon_put_chars }

{******************************************************************************}
function put_line (istr : varying [isize] of char): unsigned;
{ Put a completed line. }
  var
	str : text_type;
  begin
	if istr.length > 0 then
	  if istr.body[1] = esc then ansi_output := true;

        str := pending_text + istr;
	pending_text.length := 0;
	prompting := false; { Server isn't prompting }

	if not entirely_batch_mode then write_term (str, true)
	else lib$put_output (str);

	if data_logging then
	  begin
		tscon_data_log (pending_log_text + str);
		pending_log_text.length := 0;
	  end;

	write_done := true;
	put_line := tscon_normal;
  end; { put_line }

{******************************************************************************}
[global, unbound] function tscon_put_line (var out_str : string_desc
					  ): unsigned;
{ Put a completed line. }
  begin
        tscon_put_line := put_line (substr (out_str.addr^, 1, out_str.length));
  end; { tscon_put_line }

{******************************************************************************}
[global, unbound] procedure tscon_put_blank;
  begin
	put_line ('');
  end; { tscon_put_blank }

{******************************************************************************}
[global, unbound] function tscon_put_message (var out_str : string_desc
					     ): unsigned;
{ $PUTMSG action routine }
  begin
        put_line (substr (out_str.addr^, 1, out_str.length));
	tscon_put_message := 0; { Suppress output by $putmsg }
  end; { tscon_put_message }

end.
