/***********************************************************************/
/* TOPPROC.C                                                           */
/*                                                                     */
/* Set of routines to determine the current top cpu user on the system.*/
/* These can be called from another application or linked as a         */
/* standalone exe (see comments by main() routine).                    */
/*                                                                     */
/* Created by J.Kazimierczyk  14-FEB-1993                              */
/***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <jpidef.h>
#include <ssdef.h>
#include <starlet.h>
#include <syidef.h>

int initialize_topproc();
int get_topproc();
char *format_topproc(int);
int topproc_getPID(int);

/* Struct for creating VMS item lists */
typedef struct _vms_item_list {
	short	 buffer_length;
	short	 item_code;
	void	*buffer;
	int	*returned_buffer_length;
	} VMS_ItemList;

/* Struct to contain the info we need to track the top cpu user */
typedef struct _proclist {
	int	pid;
	int	cputime;
	int	change;
	} Proc_List;

/* Pointer to our array of processes on the system */
Proc_List *ProcList;

static int timelast;		/* last value from sys$gettim */
static int timediff;		/* last change in system time */


/**********************************************************************/
/* This main() routine demonstrates how these routines can be tested, */
/* or used for some other application.  To create a standalone        */
/* executable for testing, do:                                        */
/*           $ CC TOPPROC.C/DEFINE=TEST                               */
/*           $ LINK TOPPROC,SYS$INPUT/OPT                             */
/*                       SYS$LIBRARY:VAXCRTL.EXE/SHARE                */
/* This will print out some information about the current top process */
/* every 3 seconds.                                                   */
/**********************************************************************/

#ifdef TEST
main ()
    {
    int topprocIndex;
    char *text;

    initialize_topproc();			/* only do this once */
    while (1)
        {
        topprocIndex = get_topproc();		/* ret index into array */
        text = format_topproc(topprocIndex);	/* canned display format */
        printf ("%s\n", text);
        lib$wait (&(float)3);			/* wait 3 seconds */
        }
    }
#endif


/**********************************************************************/
/* initialize_topproc is called to initialize the data structure that */
/* we need to track cpu utilization of each process.  This returns    */
/* the value of MAXPROCESSCNT or 0 if we fail.  This should only be   */
/* called once per application.                                       */
/*                                                                    */
/* The basic idea here is to create and init the array ProcList with  */
/* MAXPROCESSNCT elements.  This is what we need to track all         */
/* processes on the system.                                           */
/**********************************************************************/

int initialize_topproc()
    {
    int status, i;
    unsigned int timequad[2];

    static int maxproc;
    static VMS_ItemList SYI_itemlist[2] = {
        sizeof(maxproc), SYI$_MAXPROCESSCNT, &maxproc, 0,
        0, 0, 0, 0};


    status = sys$getsyiw (0, 0, 0, &SYI_itemlist, 0, 0, 0);
    if (status != SS$_NORMAL) return 0;

    /* Create the array to hold all processes on the system */
    ProcList = calloc( (maxproc+1), sizeof(Proc_List) );

    sys$gettim (&timequad[0]);		/* To calc % cpu used */
    timelast = timequad[0];

    return maxproc;
    }

/**********************************************************************/
/* get_topproc scans through all processes on the system, and finds   */
/* which one is the top cpu user.  initialize_topproc must have been  */
/* called first to set up the ProcList array.  We return an index     */
/* into the ProcList array, pointing to our top user of the moment.   */
/* This index can then be used by the other routines to identify the  */
/* top process.                                                       */
/**********************************************************************/

int get_topproc()
   {
    int status, idx, biggest_idx, biggest_change;
    unsigned int timequad[2];

    static int pid, cputime, proc_index;
    static VMS_ItemList itemlist[4] = {
        sizeof(pid),        JPI$_PID,        &pid,        0,
        sizeof(cputime),    JPI$_CPUTIM,     &cputime,    0,
        sizeof(proc_index), JPI$_PROC_INDEX, &proc_index, 0,
        0, 0, 0, 0};


    biggest_idx = 0;
    biggest_change = -1;
    idx = -1;			/* wildcard for initial $getjpiw call */

    while (1)
        {
        status = sys$getjpiw (0, &idx, 0, &itemlist, 0, 0, 0);
        if (status == SS$_NOMOREPROC) break;

        if (ProcList[proc_index].pid == pid)
	    {
	    /* We already saw this process, so we have enough info to */
            /* calculate the change in cputime since the last interval */

            ProcList[proc_index].change = cputime-ProcList[proc_index].cputime;
	    ProcList[proc_index].cputime = cputime;

	    /* Also keep track of the biggest change (top user) */
            if (ProcList[proc_index].change > biggest_change)
                {
                biggest_idx = proc_index;
	        biggest_change = ProcList[proc_index].change;
	        }
	    }
	else
	    {
	    /* There is a new process in this slot - initialize it */
	    ProcList[proc_index].pid = pid;
	    ProcList[proc_index].cputime = cputime;
	    ProcList[proc_index].change = 0;
	    }
        }

    /* We need the change in system time to calc the % cpu used */
    /* Note: using half of the system datetime quadword is not  */
    /* 100% accurate, but it's easier and doesnt seem to be a problem */
    sys$gettim (&timequad[0]);
    timediff = timequad[0] - timelast;
    timelast = timequad[0];

    return biggest_idx;		/* return our top user's index */
    }


/**********************************************************************/
/* format_topproc formats the top process information for display.    */
/* The integer top is the index into ProcList which tells us who we   */
/* need to format.  We return a pointer to a character string which   */
/* contains the formatted text.                                       */
/**********************************************************************/


char *format_topproc(int top)
    {
    int i, status, usrlen, prcstrt, imgstrt, tmp;
    static char text[304];

    static char imgname[256], usrnam[13], prcnam[16];
    static int imglen;
    static VMS_ItemList JPI_dsp_itemlist[4] = {
        sizeof(imgname)-1, JPI$_IMAGNAME, &imgname, &imglen,
        sizeof(usrnam)-1,  JPI$_USERNAME, &usrnam,  0,
        sizeof(prcnam)-1,  JPI$_PRCNAM,   &prcnam,  0,
        0, 0, 0, 0};

    status = sys$getjpiw (0, &ProcList[top].pid, 0, &JPI_dsp_itemlist, 0,0,0);
    if ( status != SS$_NORMAL  &&  status != SS$_SUSPENDED ) {
        /* These must be valid values because they are still printed out below*/
        usrnam[0] = imgname[0] = prcnam[0] = '?';
        usrnam[1] = imgname[1] = prcnam[1] = '\0';
        usrlen = imglen = 1;
        imgstrt = prcstrt = 0;
    } else {
        /* Trim the trailing spaces from username */
        for (usrlen=1;usrlen<=sizeof(usrnam)&&usrnam[usrlen]!=' ';usrlen++);
        /* Trim leading spaces from process name */
        for (prcstrt=0;prcstrt<=sizeof(prcnam)&&prcnam[prcstrt]==' ';prcstrt++);
        if ( status == SS$_SUSPENDED ) {
          imgname[0] = '?'; /* Can't rely on image name for susp process */
          imgname[1] = '\0';
          imgstrt = 0;
          imglen = 1;
        } else {
	  /* Trim off the dev:[dir] (or dev:<dir>) part of the image name */
          imgname[imglen] = '\0';
          for (i=imglen; i>=0 && imgname[i]!=']' && imgname[i]!='>'; i--);
          imgstrt = i+1;
        }
    }
                     
    tmp = timediff / 100000;
    sprintf (text, "%08X %.*s %.*s %s (%d%%)",
	ProcList[top].pid,
	usrlen, usrnam,
	(sizeof(prcnam)-prcstrt), (prcnam+prcstrt),
	(imgname+imgstrt),
	((ProcList[top].change)*100+50)/(tmp ? tmp : 1));

    return text;	/* pointer to the formatted top proc info */
    }

/**********************************************************************/
/* topproc_getPID() simply takes the ProcList array index, and returns*/
/* the PID of the process pointed to.   The caller can then use this  */
/* PID for whatever it likes.                                         */
/**********************************************************************/

int topproc_getPID (int top)
    {
    return ProcList[top].pid;
    }

