/*****************************************************************************/
/* Copyright (c) 2003 Matjaz Rihtar <matjaz@eunet.si>                        */
/*                                                                           */
/* MORE - simple more/less for OpenVMS (tested on OpenVMS V7.3-1, DECC V6.5) */
/*                                                                           */
/* Works on ANSI, VT100, xterm and similar terminals. Whole file is read     */
/* into memory, so you'll have to wait for pipe to finish (max. 8M lines     */
/* of 65Kb length), but it's very fast after that. Unprintable characters    */
/* are shown as dot (".").                                                   */
/*                                                                           */
/* $ more -?                                                                 */
/* Usage: more [-c] <file(s)>                                                */
/*   -c  case sensitive search (default: ignore case)                        */
/*                                                                           */
/* $ more file.txt                                                           */
/* $ more *.txt                                                              */
/* $ pipe dir | more                                                         */
/*                                                                           */
/* When inside, use the following commands:                                  */
/* <space>, <Page Down>, <^D>  next page                                     */
/* <Page Up>, <^B>             previous page                                 */
/* <^M>, <Down Arrow>, j       next line                                     */
/* <Up Arrow>, k               previous line                                 */
/* <Home>, 0G, 1G              beginning of file                             */
/* <End>, G                    end of file                                   */
/* <^G>                        show info about the file                      */
/* <^W>, <^R>, <^L>            refresh screen                                */
/* q, <^Z>, <^C>               quit viewing this file and view next file     */
/*                             (exit if last file)                           */
/* Q, <^Y>                     quit viewing files and exit                   */
/* /                           find string in file (forwards, up to EOF)     */
/* ?                           find string in file (backwards, up to BOF)    */
/*                             (you can use *, ?, % and [a-z] DEC regexp)    */
/* n                           find next occurence of string (up to EOF)     */
/* N                           find previous occurence of string (up to BOF) */
/* w                           write current buffer to disk (asks for name)  */
/*                             (file written has tabs converted to spaces!)  */
/*****************************************************************************/
#pragma module more "V1.1-2"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <unixlib.h>
#include <errno.h>

#include <string.h>
#include <ctype.h>

#include <starlet.h>
#include <ssdef.h>
#include <iosbdef.h>
#include <lib$routines.h>
#include <libdef.h>
#include <descrip.h>

#include <rms.h>
#include <jpidef.h>
#include <dvidef.h>
#include <iodef.h>
#include <ttdef.h>
#include <tt2def.h>

#define MAXL 65535    // maximum line length
#define MAXN 255      // maximum file name length (system limitation)
#define MAXR 8388607  // maximum number of records in file

#define CTRLB '\002'
#define CTRLC '\003'
#define CTRLD '\004'
#define CTRLL '\014'
#define CTRLR '\022'
#define CTRLW '\027'
#define CTRLY '\031'
#define CTRLZ '\032'
#define CR    '\015'
#define ESC   '\033'
#define CTRLG '\007'

#define EXIT   1
#define QUIT   2
#define NLINE  3
#define PLINE  4
#define NPAGE  5
#define PPAGE  6
#define HOME   7
#define END    8
#define FINDF  9
#define FINDB 10
#define FNEXT 11
#define FPREV 12
#define INFO  13
#define WRITE 14
#define RFRSH 15

typedef struct itemlist2 {
  unsigned short len;
  unsigned short code;
  union {
    void *addr;
    unsigned int data;
  } p2;
} IL2;

typedef struct itemlist3 {
  unsigned short len;
  unsigned short code;
  union {
    void *addr;
    unsigned int data;
  } p2;
  unsigned short *buflen;
} IL3;

typedef struct ttychar {
  unsigned char class;
  unsigned char type;
  unsigned short width;
  unsigned int btc;
  unsigned int xtc;
} TTC;

typedef struct exithandler {
  int forward_link;
  int (*exit_routine)();
  int arg_count;
  int *status_addr;
  int exit_status;
} EXH;

char *prog;
int igncase;
int sfn;  // short file name
int fileshown;

int rows, cols;
unsigned short tty;
TTC old_ttychar;

char *file[MAXR];

struct FAB fblock;
struct NAM nblock;
char search_name[MAXN+1];


int opentty(void)
{
  int status;
  iosb iostatus;
  unsigned int mpid, ppid;
  char phyn[9];
  $DESCRIPTOR(phyn_d, phyn);
  IL3 itemlist[3];

  rows = 24; cols = 80;
  tty = 0;

  mpid = 0; ppid = 0xFFFFFFFF;
  for ( ; ; ) {
    itemlist[0].len = sizeof(mpid);
    itemlist[0].code = JPI$_MASTER_PID;
    itemlist[0].p2.addr = &mpid;
    itemlist[0].buflen = NULL;
    memset(&itemlist[1], 0, sizeof(IL3));
    status = sys$getjpiw(0, &mpid, 0, &itemlist, &iostatus, 0, 0);
    if (status & 1) status = iostatus.iosb$l_getxxi_status;
    if (status != SS$_NORMAL && status != LIB$_NORMAL) {
      fprintf(stderr, "$GETJPI: %s\n", strerror(EVMSERR, status));
    }
    if (mpid == ppid) break;
    ppid = mpid;
  }

  itemlist[0].len = sizeof(phyn)-1;
  itemlist[0].code = JPI$_TT_PHYDEVNAM;
  itemlist[0].p2.addr = &phyn;
  itemlist[0].buflen = &phyn_d.dsc$w_length;
  memset(&itemlist[1], 0, sizeof(IL3));
  status = sys$getjpiw(0, &mpid, 0, &itemlist, &iostatus, 0, 0);
  if (status & 1) status = iostatus.iosb$l_getxxi_status;
  if (status != SS$_NORMAL && status != LIB$_NORMAL) {
    fprintf(stderr, "$GETJPI: %s\n", strerror(EVMSERR, status));
  }

  itemlist[0].len = sizeof(rows);
  itemlist[0].code = DVI$_TT_PAGE;
  itemlist[0].p2.addr = &rows;
  itemlist[0].buflen = NULL;
  itemlist[1].len = sizeof(cols);
  itemlist[1].code = DVI$_DEVBUFSIZ;
  itemlist[1].p2.addr = &cols;
  itemlist[1].buflen = NULL;
  memset(&itemlist[2], 0, sizeof(IL3));
  status = sys$getdviw(0, 0, &phyn_d, &itemlist, &iostatus, 0, 0, 0);
  if (status & 1) status = iostatus.iosb$l_getxxi_status;
  if (status != SS$_NORMAL && status != LIB$_NORMAL) {
    fprintf(stderr, "$GETDVI: %s\n", strerror(EVMSERR, status));
  }
/*
  printf("MPid %04.4X, %s %d x %d\n", mpid, phyn, rows, cols);
*/
  status = sys$assign(&phyn_d, &tty, 0, 0);
  if (status != SS$_NORMAL && status != LIB$_NORMAL) {
    fprintf(stderr, "$ASSIGN: %s\n", strerror(EVMSERR, status));
    return(-1);
  }

  status = sys$qiow(0, tty, IO$_SENSEMODE, &iostatus, 0, 0,
                    &old_ttychar, sizeof(old_ttychar), 0, 0, 0, 0);
  if (status & 1) status = iostatus.iosb$w_status;
  if (status != SS$_NORMAL && status != LIB$_NORMAL) {
    fprintf(stderr, "$QIO(SENSEMODE): %s\n", strerror(EVMSERR, status));
    return(-1);
  }

  return(0);
} /* opentty */


int stty(int pasthru, int echo)
{
  int status;
  iosb iostatus;
  TTC new_ttychar;

  memcpy(&new_ttychar, &old_ttychar, sizeof(TTC));
  new_ttychar.btc |= TT$M_EIGHTBIT;
  if (!echo) new_ttychar.btc |= TT$M_NOECHO;
  new_ttychar.xtc |= TT2$M_XON | TT2$M_EDITING | TT2$M_INSERT;
  if (pasthru) new_ttychar.xtc |= TT2$M_PASTHRU;

  status = sys$qiow(0, tty, IO$_SETMODE, &iostatus, 0, 0,
                    &new_ttychar, sizeof(new_ttychar), 0, 0, 0, 0);
  if (status & 1) status = iostatus.iosb$w_status;
  if (status != SS$_NORMAL && status != LIB$_NORMAL) {
    fprintf(stderr, "$QIO(SETMODE): %s\n", strerror(EVMSERR, status));
    return(-1);
  }

  return(0);
} /* stty */


int closetty(void)
{
  int status;
  iosb iostatus;
  char buf[MAXL+1];

  sprintf(buf, "\r\033[m");
  status = sys$qiow(0, tty, IO$_WRITEVBLK | IO$M_NOFORMAT | IO$M_CANCTRLO,
                    &iostatus, 0, 0, buf, strlen(buf), 0, 0, 0, 0);
  if (status & 1) status = iostatus.iosb$w_status;
  if (status != SS$_NORMAL && status != LIB$_NORMAL) {
    fprintf(stderr, "$QIO(WRITEVBLK): %s\n", strerror(EVMSERR, status));
    return(-1);
  }

  status = sys$qiow(0, tty, IO$_SETMODE, &iostatus, 0, 0,
                    &old_ttychar, sizeof(old_ttychar), 0, 0, 0, 0);
  if (status & 1) status = iostatus.iosb$w_status;
  if (status != SS$_NORMAL && status != LIB$_NORMAL) {
    fprintf(stderr, "$QIO(SETMODE): %s\n", strerror(EVMSERR, status));
    return(-1);
  }

  status = sys$dassgn(tty);
  if (status != SS$_NORMAL && status != LIB$_NORMAL) {
    fprintf(stderr, "$DASSGN: %s\n", strerror(EVMSERR, status));
    return(-1);
  }

  return(0);
} /* closetty */


int init_search(char *wanted_name)
{
  int len, status;
  static char default_name[] = "*.*";
  char *sn;

  fblock = cc$rms_fab;
  fblock.fab$l_dna = default_name;
  len = strlen(default_name);
  if (len > MAXN) len = MAXN;
  fblock.fab$b_dns = len;
  fblock.fab$l_fna = wanted_name;
  len = strlen(wanted_name);
  if (len > MAXN) len = MAXN;
  fblock.fab$b_fns = len;
  fblock.fab$l_nam = &nblock;

  nblock = cc$rms_nam;
  nblock.nam$l_esa = search_name;
  nblock.nam$b_ess = MAXN;
  search_name[0] = '\0';

  status = sys$parse(&fblock);
  search_name[nblock.nam$b_esl] = '\0';
  if (status != RMS$_NORMAL) {
    if (sfn) {
      if ((sn = strrchr(search_name, ']')) == NULL) sn = search_name; else sn++;
      fprintf(stderr, "%s: %s\n", sn, strerror(EVMSERR, status));
    }
    else fprintf(stderr, "%s: %s\n", search_name, strerror(EVMSERR, status));
    return(-1);
  }
/*
  printf("Searching for: |%s|\n", search_name);
*/
  /* reset for searching */
  fblock = cc$rms_fab;
  fblock.fab$l_nam = &nblock;

  return(nblock.nam$b_esl);
} /* init_search */


int search(char *fname)
{
  int status;
  char *sn;

  nblock.nam$l_rsa = fname;
  nblock.nam$b_rss = MAXN;
//fname[0] = '\0';  /* don't do this or search won't work */

  status = sys$search(&fblock);
  search_name[nblock.nam$b_esl] = '\0';
  if (status == RMS$_NMF) return(0);  // no more files
  if (status != RMS$_NORMAL) {
    if (sfn) {
      if ((sn = strrchr(search_name, ']')) == NULL) sn = search_name; else sn++;
      fprintf(stderr, "%s: %s\n", sn, strerror(EVMSERR, status));
    }
    else fprintf(stderr, "%s: %s\n", search_name, strerror(EVMSERR, status));
    return(-1);
  }
  fname[nblock.nam$b_rsl] = '\0';
/*
  printf("Found: |%s|\n", fname);
*/
  return(nblock.nam$b_rsl);
} /* search */


char rdc(void)
{
  int status;
  iosb iostatus;
  char c;
  unsigned int tmask[2] = { 0, 0 };  // read terminator mask

  status = sys$qiow(0, tty, IO$_READVBLK | IO$_TTYREADALL | IO$M_NOECHO,
                    &iostatus, 0, 0, &c, 1, 0, &tmask, 0, 0);
  if (status & 1) status = iostatus.iosb$w_status;
  if (status != SS$_NORMAL && status != LIB$_NORMAL) {
    fprintf(stderr, "$QIO(READVBLK): %s\n", strerror(EVMSERR, status));
    return(0);
  }
/*
  if (c < 32 || c > 126) printf("Read %3d (.) \n", c);
  else printf("Read %3d (%c) \n", c, c);
*/
  return(c);
} /* rdc */


int rds(char *prompt, char *buf, int buflen)
{
  int status;
  iosb iostatus;
  unsigned int tmask[2] = { 0, 0 };  // read terminator mask

  status = sys$qiow(0, tty, IO$_READPROMPT | IO$M_TRMNOECHO,
                    &iostatus, 0, 0, buf, buflen, 0, 0, prompt, strlen(prompt));
  if (status & 1) status = iostatus.iosb$w_status;
  if (status != SS$_NORMAL && status != LIB$_NORMAL) {
    fprintf(stderr, "$QIO(READVBLK): %s\n", strerror(EVMSERR, status));
    return(-1);
  }
  buf[iostatus.iosb$w_bcnt] = '\0';
/*
  printf("Read %3d |%s| \n", iostatus.iosb$w_bcnt, buf);
*/
  return(iostatus.iosb$w_bcnt);
} /* rds */


int wrs(char *buf)
{
  int status;
  iosb iostatus;

  status = sys$qiow(0, tty, IO$_WRITEVBLK | IO$M_NOFORMAT | IO$M_CANCTRLO,
                    &iostatus, 0, 0, buf, strlen(buf), 0, 0, 0, 0);
  if (status & 1) status = iostatus.iosb$w_status;
  if (status != SS$_NORMAL && status != LIB$_NORMAL) {
    fprintf(stderr, "$QIO(WRITEVBLK): %s\n", strerror(EVMSERR, status));
    return(-1);
  }

  return(0);
} /* wrs */


int getcmd(void)
{
  char c;

  c = rdc();
again:
  if (c == ESC) {
    c = rdc();
    if (c == 'O') {
      c = rdc();
      if (c == 'A') return(PLINE);
      else if (c == 'B') return(NLINE);
      else if (c == 'H') return(HOME);
      else if (c == 'F') return(END);
      goto again;
    }
    else if (c == '[') {
      c = rdc();
      if (c == 'A') return(PLINE);
      else if (c == 'B') return(NLINE);
      else if (c == '1') {
        c = rdc();
        if (c == '~') return(HOME);
        goto again;
      }
      else if (c == '4') {
        c = rdc();
        if (c == '~') return(END);
        goto again;
      }
      else if (c == '5') {
        c = rdc();
        if (c == '~') return(PPAGE);
        goto again;
      }
      else if (c == '6') {
        c = rdc();
        if (c == '~') return(NPAGE);
        goto again;
      }
    }
    goto again;
  }
  else if (c == '0') {
    c = rdc();
    if (c == 'G') return(HOME);
    goto again;
  }
  else if (c == '1') {
    c = rdc();
    if (c == 'G') return(HOME);
    goto again;
  }
  else if (c == CR) return(NLINE);
  else if (c == ' ') return(NPAGE);
  else if (c == 'j') return(NLINE);
  else if (c == 'k') return(PLINE);
  else if (c == 'G') return(END);
  else if (c == 'q') return(QUIT);
  else if (c == 'Q') return(EXIT);
  else if (c == '/') return(FINDF);
  else if (c == '?') return(FINDB);
  else if (c == 'n') return(FNEXT);
  else if (c == 'N') return(FPREV);
  else if (c == 'w') return(WRITE);
  else if (c == CTRLC) return(QUIT);
  else if (c == CTRLZ) return(QUIT);
  else if (c == CTRLY) return(EXIT);
  else if (c == CTRLB) return(PPAGE);
  else if (c == CTRLD) return(NPAGE);
  else if (c == CTRLG) return(INFO);
  else if (c == CTRLL) return(RFRSH);
  else if (c == CTRLR) return(RFRSH);
  else if (c == CTRLW) return(RFRSH);

  wrs("\007");
  return(0);
} /* getcmd */


void trim(char *s)
{
  int len;

  len = strlen(s);
  while (len > 0 && isspace(s[len-1]))
    s[--len] = 0;
} /* trim */


char *filter(char *s, int detab)
{
  static char line[MAXL+1];
  int ii, pos;

  pos = 1;
  for (ii = 0; *s; s++) {
#if 0
    if (!isprint(*s)) {
#else
    if (*s < 32 || *s > 126) {
#endif
      if (*s == 9 && detab) {
        line[ii++] = ' '; pos++;
        for ( ; pos%8 != 1; pos++)
          line[ii++] = ' ';
      }
      else {
        line[ii++] = '.'; pos++;
      }
    }
    else {
      line[ii++] = *s; pos++;
    }
  }
  line[ii] = '\0';

  return(line);
} /* filter */


void lower(char *s)
{
  for ( ; *s; s++)
    *s = tolower(*s);
} /* lower */


int more(char *fname)
{
  FILE *inp, *out;
  char *fn, line[MAXL+1], *nl;
  char oname[MAXL+1], *on;
  int ii, n, rn, len, mrs, cl, pl, cmd;
  int stsline, pos, rcl;
  char tmp[MAXL+1], pat[MAXL+1];
  int found, rc;

  if ((fn = strrchr(fname, ']')) == NULL) fn = fname; else fn++;

  if (strncasecmp(fname, "SYS$INPUT", 9) == 0) inp = stdin;
  else if (access(fname, R_OK) < 0) {
    if (sfn) fprintf(stderr, "%s: %s\n", fn, strerror(EVMSERR, vaxc$errno));
    else fprintf(stderr, "%s: %s\n", fname, strerror(EVMSERR, vaxc$errno));
    return(-1);
  }
  else inp = fopen(fname, "r",
#if 0
    "ctx=rec",
#endif
    "mbc=32","mbf=8","rop=rah");
  if (inp == NULL) {
    if (sfn) fprintf(stderr, "%s: %s\n", fn, strerror(EVMSERR, vaxc$errno));
    else fprintf(stderr, "%s: %s\n", fname, strerror(EVMSERR, vaxc$errno));
    return(-1);
  }

  rn = 0; mrs = 0;
  for ( ; ; ) {
#if 1
    fgets(line, MAXL, inp); line[MAXL] = '\0';
    if (ferror(inp)) {
#else
    n = decc$record_read(inp, line, MAXL);
    if (n >= 0) line[n] = '\0';
    else {
#endif
      if (sfn) fprintf(stderr, "%s: %s\n", fn, strerror(EVMSERR, vaxc$errno));
      else fprintf(stderr, "%s: %s\n", fname, strerror(EVMSERR, vaxc$errno));
      if (inp != stdin) fclose(inp);
      return(-1);
    }
    if (feof(inp)) break;

    /* read file */
    trim(line);
    strcpy(line, filter(line, 1));  // filter non-printable, detab

    len = strlen(line);
    if (len > mrs) mrs = len;
    nl = malloc(len + 1);
    if (nl != NULL && rn < MAXR) {
      strcpy(nl, line);
      file[rn++] = nl;
    }
  } /* while !eof */
  if (inp != stdin) fclose(inp);

  nl = malloc(6);
  if (nl != NULL && rn < MAXR) {
    strcpy(nl, "[EOF]");
    file[rn++] = nl;
  }

  stty(1, 0);  // pasthru, noecho
  pat[0] = '\0';  // search pattern

  /* display file xxx */
  if (fileshown) wrs("\033[H\033[J");
  stsline = 0;

  cl = 0;
  for (ii = 0; ii < rows && cl < rn; ii++) {
    if (strlen(file[cl]) >= cols)
      sprintf(line, "%-.*s\r", cols, file[cl]);
    else
      sprintf(line, "%-.*s\033[K\r", cols, file[cl]);
    wrs(line); cl++;
    if (ii < rows-1 && cl < rn) wrs("\n");
  }
  pl = cl - rows - 1;

  rc = 0;
  for ( ; ; ) {
    cmd = getcmd();
    if (stsline) {
      if (strlen(file[cl-1]) >= cols)
        sprintf(line, "\r\033[m%-.*s\r", cols, file[cl-1]);
      else
        sprintf(line, "\r\033[m%-.*s\033[K\r", cols, file[cl-1]);
      wrs(line); stsline = 0;
    }
    if (cmd == QUIT) {
      wrs("\033[J");
      break;
    }
    else if (cmd == EXIT) {
      wrs("\033[J");
      rc = 1;
      break;
    }
    else if (cmd == NLINE) {
      if (cl < rn) {
        sprintf(line, "\n%-.*s\r", cols, file[cl]); wrs(line);
        cl++; pl++;
      }
      else wrs("\007");
    }
    else if (cmd == PLINE) {
      if (pl >= 0) {
        wrs("\0337\033[H\033M");
        sprintf(line, "%-.*s\r", cols, file[pl]); wrs(line);
        wrs("\0338");
        cl--; pl--;
      }
      else wrs("\007");
    }
    else if (cmd == NPAGE) {
      if (cl < rn) {
        for (ii = 0; ii < rows && cl < rn; ii++) {
          sprintf(line, "\n%-.*s\r", cols, file[cl]); wrs(line);
          cl++;
        }
        pl = cl - rows - 1;
      }
      else wrs("\007");
    }
    else if (cmd == PPAGE) {
      if (rn > rows) {
        wrs("\033[H");
        cl = cl - 2*rows; if (cl < 0) cl = 0;
        for (ii = 0; ii < rows && cl < rn; ii++) {
          if (strlen(file[cl]) >= cols)
            sprintf(line, "%-.*s\r", cols, file[cl]);
          else
            sprintf(line, "%-.*s\033[K\r", cols, file[cl]);
          wrs(line); cl++;
          if (ii < rows-1 && cl < rn) wrs("\n");
        }
        pl = cl - rows - 1;
      }
      else wrs("\007");
    }
    else if (cmd == HOME) {
      if (rn > rows) {
        wrs("\033[H");
        cl = 0;
        for (ii = 0; ii < rows && cl < rn; ii++) {
          if (strlen(file[cl]) >= cols)
            sprintf(line, "%-.*s\r", cols, file[cl]);
          else
            sprintf(line, "%-.*s\033[K\r", cols, file[cl]);
          wrs(line); cl++;
          if (ii < rows-1 && cl < rn) wrs("\n");
        }
        pl = cl - rows - 1;
      }
      else wrs("\007");
    }
    else if (cmd == END) {
      if (rn > rows) {
        wrs("\033[H");
        cl = rn - rows;
        for (ii = 0; ii < rows && cl < rn; ii++) {
          if (strlen(file[cl]) >= cols)
            sprintf(line, "%-.*s\r", cols, file[cl]);
          else
            sprintf(line, "%-.*s\033[K\r", cols, file[cl]);
          wrs(line); cl++;
          if (ii < rows-1 && cl < rn) wrs("\n");
        }
        pl = cl - rows - 1;
      }
      else wrs("\007");
    }
    else if (cmd == FINDF) {
      stty(0, 1); // nopasthru, echo
      n = rds("\033[7mSearch for: \033[K", tmp, 255);
      stty(1, 0); // pasthru, noecho
      if (strlen(file[cl-1]) >= cols)
        sprintf(line, "\r\033[m%-.*s\r", cols, file[cl-1]);
      else
        sprintf(line, "\r\033[m%-.*s\033[K\r", cols, file[cl-1]);
      wrs(line);

      if (n == 0) goto fnext;

      sprintf(pat, "*%s*", tmp);
      if (igncase) lower(pat);

      goto fnext;
    }
    else if (cmd == FNEXT) {
fnext:
      found = 0;
      ii = pl + 2; if (ii < 0) ii = 0;
      if (igncase) {
        for ( ; ii < rn; ii++) {
          strcpy(line, file[ii]); lower(line);
          if (decc$match_wild(line, pat)) {
            found = 1; break;
          }
        }
      }
      else {
        for ( ; ii < rn; ii++) {
          if (decc$match_wild(file[ii], pat)) {
            found = 1; break;
          }
        }
      }
      if (found) {
        wrs("\033[H\033[J");
        cl = ii;
        for (ii = 0; ii < rows && cl < rn; ii++) {
          sprintf(line, "%-.*s\r", cols, file[cl]); wrs(line);
          cl++;
          if (ii < rows-1 && cl < rn) wrs("\n");
        }
        pl = cl - ii - 1;
      }
      else wrs("\007");
    }
    else if (cmd == FINDB) {
      stty(0, 1); // nopasthru, echo
      n = rds("\033[7mSearch for: \033[K", tmp, 255);
      stty(1, 0); // pasthru, noecho
      if (strlen(file[cl-1]) >= cols)
        sprintf(line, "\r\033[m%-.*s\r", cols, file[cl-1]);
      else
        sprintf(line, "\r\033[m%-.*s\033[K\r", cols, file[cl-1]);
      wrs(line);

      if (n == 0) goto fprev;

      sprintf(pat, "*%s*", tmp);
      if (igncase) lower(pat);

      goto fprev;
    }
    else if (cmd == FPREV) {
fprev:
      found = 0;
      ii = pl;
      if (igncase) {
        for ( ; ii >= 0; ii--) {
          strcpy(line, file[ii]); lower(line);
          if (decc$match_wild(line, pat)) {
            found = 1; break;
          }
        }
      }
      else {
        for ( ; ii >= 0; ii--) {
          if (decc$match_wild(file[ii], pat)) {
            found = 1; break;
          }
        }
      }
      if (found) {
        wrs("\033[H\033[J");
        cl = ii;
        for (ii = 0; ii < rows && cl < rn; ii++) {
          sprintf(line, "%-.*s\r", cols, file[cl]); wrs(line);
          cl++;
          if (ii < rows-1 && cl < rn) wrs("\n");
        }
        pl = cl - ii - 1;
      }
      else wrs("\007");
    }
    else if (cmd == INFO) {
      rn--; if (cl > rn) rcl = rn; else rcl = cl;
      if (rn != 0) pos = (rcl*100)/rn; else pos = 100;
      sprintf(line, "\033[7m%s, %d/%d lines (%d%%)\033[K", fn, rcl, rn, pos);
      wrs(line); stsline = 1;
      rn++;
    }
    else if (cmd == WRITE) {
      stty(0, 1); // nopasthru, echo
      n = rds("\033[7mWrite to file: \033[K", oname, 255);
      stty(1, 0); // pasthru, noecho

      if (n == 0) {
        if (strlen(file[cl-1]) >= cols)
          sprintf(line, "\r\033[m%-.*s\r", cols, file[cl-1]);
        else
          sprintf(line, "\r\033[m%-.*s\033[K\r", cols, file[cl-1]);
        wrs(line);
        continue;
      }

      out = fopen(oname, "w",
#if 1
        "rfm=var","rat=cr",
#endif
        "mbc=32","mbf=8","rop=wbh");
      if (out != NULL) {
        fgetname(out, oname);
        if ((on = strrchr(oname, ']')) == NULL) on = oname; else on++;

        for (ii = 0; ii < rn-1; ii++) {
          fprintf(out, "%s\n", file[ii]);
          if (ferror(out)) goto ferr;
        }

        fclose(out);
        sprintf(line, "\r\033[7m%s, %d lines (mrs=%d) written\033[K",
                on, rn-1, mrs);
        wrs(line); stsline = 1;
      }
      else {
ferr:
        sprintf(tmp, "%s: %s", oname, strerror(EVMSERR, vaxc$errno));
        if (strlen(tmp) >= cols)
          sprintf(line, "\r\033[7m\007%-.*s", cols, tmp);
        else
          sprintf(line, "\r\033[7m\007%-.*s\033[K", cols, tmp);
        wrs(line); stsline = 1;
      }
    }
    else if (cmd == RFRSH) {
      wrs("\033[H\033[J");
      cl = cl - rows; if (cl < 0) cl = 0;
      for (ii = 0; ii < rows && cl < rn; ii++) {
        sprintf(line, "%-.*s\r", cols, file[cl]); wrs(line);
        cl++;
        if (ii < rows-1 && cl < rn) wrs("\n");
      }
      pl = cl - rows - 1;
    }
  } /* for getcmd xxx */

  for (ii = 0; ii < rn; ii++)
    free(file[ii]);

  return(rc);
} /* more */


int main(int argc, char *argv[])
{
  int ii, ac, err, status;
  char *s, fname[MAXN+1];
  EXH exh = { 0, NULL, 1, &exh.exit_status, 0 };

  if ((prog = strrchr(argv[0], ']')) == NULL) prog = argv[0];
  else prog++;
  if ((s = strstr(prog, ".exe")) != NULL) *s = '\0';

  ac = 0; igncase = 1; sfn = 0;
  for (ii = 1; ii < argc; ii++) {
    if (strcmp(argv[ii], "-c") == 0) {
      argv[ii][0] = '\0'; igncase = 0;
    }
    else if (strcmp(argv[ii], "-s") == 0) {
      argv[ii][0] = '\0'; sfn = 1;
    }
    else if (strcmp(argv[ii], "-?") == 0) {
      fprintf(stderr, "Usage: %s [-c] <file(s)>\n", prog);
      exit(EXIT_FAILURE);
    }
    else ac++;
  }

  opentty();
  exh.exit_routine = (int (*)())closetty;
  sys$dclexh(&exh);

  if (ac == 0) {
    more("SYS$INPUT");
    exit(EXIT_SUCCESS);
  }

  err = 0; fileshown = 0;
  for (ii = 1; ii < argc; ii++) {
    if (strlen(argv[ii]) == 0) continue;
    if (strncasecmp(argv[ii], "SYS$INPUT", 9) == 0) {
      status = more("SYS$INPUT"); fileshown++;
      if (status < 0) err++;
      else if (status) goto done;
      continue;
    } /* if stdin */

    if (init_search(argv[ii]) <= 0) { err++; continue; }
    for ( ; ; ) {
      status = search(fname);
      if (status < 0) { err++; break; }
      else if (status == 0) break;  // no more files
      else {
        status = more(fname); fileshown++;
        if (status < 0) err++;
        else if (status) goto done;
      }
    } /* for fname */
  } /* for argc */
done:
  if (err) return(EXIT_FAILURE);
  else return(0);
} /* main */
