/*------------------------------------------------------------------------------
 *   PERF_METER.C - Performance Meter display for DECwindows Motif, OpenVMS
 *
 *   History:
 *
 *	V01-001		Robert Heller			08-Nov-1989
 *		Created on Wed Nov  8 10:39:17 1989
 *	V01-002		Mark Pizzolato			22-Nov-1989
 *		mark@infopiz.uucp (uunet!lupine!infopiz!mark)
 *	V01-003		Robert Heller			27-Nov-1989
 *		Fixed for invert displays
 *		Modified to use set_label in place of certain set_something
 *		calls and new version of init_cpu_times.
 *
 *	V01-004		Phil Ottewell/Tony Davis	19-OCT-1992
 *		Modified to use UIL/Motif toolkit instead of DECwindows
 *
 *	V02-000		Phil Ottewell			27-NOV-1992
 *		Modified to use different colours for CPU modes,
 *		use a decent icon, and allow more user configuration.
 *
 *	V02-001		Joe Kazimierczyk (kaz@bms.com)	14-FEB-1993 
 *		Plugged memory leak caused by calls to DwtLatin1String without
 *		corresponding XtFree().
 *		Added 'top process' info to bottom of main window
 *		Added toggle button to create a 'show process/cont' type
 *		display for the current top process.
 *		Moved exit button to a popup menu instead of the menu bar,
 *		to save display space.
 *
 *	V02-000-A	Phil Ottewell			20-JAN-1993
 *		Modified V2.0 to work on Alpha
 *
 *	V02-002		Phil Ottewell			01-MAY-1995
 *		Modified to work with DEC C and Motif 1.2
 *		without having to specify /STANDARD=VAXC
 *		Merged Joe Kazimierczyk V2.1 changes in to V02-000-A
 *
 *      v02-002-a       Dean Schamberger                20-may-1995
 *              Modified to protect against setting update interval to 0 secs.
 *              Also updated .UIL to allow up to 4 devices again,
 *              and set the default to a blank line.  Added fix to device
 *              name string to keep columns aligned, and forced TOPcpu
 *              process to stay a fixed length.  This fixed some strange
 *              redraws on some x-terminals.  Finally added the option to
 *              compile a non-priv version that can still see CPU users
 *              (ie.  does not need CMKRNL privs like in version 2.0).
 *              The default is not to need CMKRNL.  To return to the
 *		previous mode, comment out the define NON_PRIV_USERS.
 *		(no longer quite true - see V02-003 below - P.O.)
 *		This only effects CPU states and usage, one still needs
 *		privs to see max cpu user.
 *	V02-003		Phil Ottewell			01-MAY-1995
 *		Fix memory leak caused byXmStringLtoRCreate (Motif pre 1.2-3)
 *		by using XmStringCreateSimple instead. The NON_PRIV_USER option
 *		has been taken out, and the VAX version builds without calling
 *		sys$cmkrnl, whereas the Alpha version HAS to get the CPU info
 *		with kernel mode access. An informational message to this effect
 *		is now issued by the MAKE_PERF_METER.COM procedure.
 *	V02-004		Phil Ottewell			22-MAR-1997
 *		Fix memory leak in TOPPROC.C, set icon properly, remove some
 *              redundant variables. Remove more global variables,
 *              add a free memory/total memory display
 *	V02-005		Phil Ottewell			21-MAR-2000
 *			Chris Sharman <Chris.Sharman@ccagroup.co.uk>
 *		Fix rounding errors in CPU graph.
 *              
 *----------------------------------------------------------------------------*/

/* ANSI C headers */
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

/* VMS headers */
#include <descrip.h>
#include <dvidef.h>
#include <jpidef.h>
#include <ssdef.h>
#include <starlet.h>
#include <syidef.h>

/* Local headers */
#ifdef __ALPHA
#include "cpudef_alpha.h"
#else
#include "cpudef.h"
#endif
#include "perf_meter_const.h"

/* X Toolkit header */
#include <X11/Intrinsic.h>
/* Motif general header (includes most Xlib stuff too) */
#include <Xm/Xm.h>
/* Other Motif headers */
#include <Xm/RowColumn.h>
/* Motif Resource Manager*/
#include <Xm/MrmAppl.h>

/* #defines go here */

#define PROGRAM_TITLE_FORMAT "%s, %s, VMS %s, %d Meg"
#define PROGRAM_NAME         "PerfMeter"
#define PROGRAM_VERSION      "2.5"

#define MX_STATES 7

#define PARAM_STRING 0
#define PARAM_INTEGER 1
#ifdef __ALPHA
#define K_TIME 0
#define E_TIME 2
#define S_TIME 4
#define U_TIME 6
#define I_TIME 8
#define C_TIME 0
#define SPIN_TIME 10
#define IDLE_TIME 12
#else
#define K_TIME 0
#define E_TIME 1
#define S_TIME 2
#define U_TIME 3
#define I_TIME 4
#define C_TIME 5
#define SPIN_TIME 6
#define IDLE_TIME 7
#endif

/* Global references */
#ifdef __DECC
#  pragma message save
#  pragma message disable (GLOBALEXT)
#endif
globalref long   EXE$GL_ABSTIM_TICS;
globalref long   SMP$GL_ACTIVE_CPUS;
globalref $CPU*  SMP$GL_CPU_DATA[1];
globalref long   PMS$GL_FAULTS;
globalref long   PMS$GL_PREADS;
globalref long   PMS$GL_PREADIO;
globalref long   PMS$GL_PWRITES;
globalref long   PMS$GL_PWRITIO;
globalref long   PMS$GL_DZROFLTS;
globalref long   SCH$GL_FREECNT;
globalref long   SCH$GL_MFYCNT;
#if defined(__VAX) && (__VMS_VER < 60000000) 
globalref long   MMG$GL_MAXMEM;
#endif
#ifdef __DECC
#  pragma message restore
#endif
#define CURRENT_FAULTS     (PMS$GL_FAULTS)
#define CURRENT_PAGEIO     (PMS$GL_PREADIO + PMS$GL_PWRITIO)
#define CURRENT_FREEPAGES  (SCH$GL_FREECNT)
#define CURRENT_MODPAGES   (SCH$GL_MFYCNT)

/* Function prototypes */
void s_error(char* problem_string);
void get_something(Widget w, char* resource, XtArgVal value);
void set_something(Widget w, char* resource, XtArgVal value);

void activate_proc(Widget w, XtPointer client_data, XmAnyCallbackStruct *cb_s);
void create_proc(Widget w, XtPointer client_data, XmAnyCallbackStruct *cb_s);
void cancel_proc(Widget w, XtPointer client_data, XmAnyCallbackStruct *cb_s);
void update( XtPointer data, XtIntervalId *id );

void create_proc(Widget w, XtPointer client_data, XmAnyCallbackStruct *cb_s);
void exposure_proc(Widget w, XtPointer client_data, XmAnyCallbackStruct *cb_s);
void update_contproc( XtPointer data, XtIntervalId *id );
void toggle_contproc(Widget w, XtPointer client_data, XmToggleButtonCallbackStruct *toggle);
void contproc_on(void);
void contproc_off(void);
void contproc_jpi_status(int status);
void doMainPopupMenu(Widget wdgt,XtPointer client_data,
                     XEvent *event, Boolean *propogate);

void DrawStateGraph( Display *dpy, Window window );
void DrawStateBar( Display *dpy, Window window, GC gc,
       int iWindowWidth, int iWindowHeight,
       int iLValue, int iLX, int iLBarWidth, int iLWidth, int iLHeight );

int initialize_topproc(void);
int get_topproc(long* pri0);
char *format_topproc(int top);
int topproc_getPID(int top);

void format_cputime (int t, char *s);

void get_initial_kernelly_stuff();
void get_new_kernelly_stuff();

void init_cpu_times();
void get_cpu_times(long* null, long* user, long* super, long* exec,
                   long* kernel, long* interrupt, long* compat);
void get_disk_opcnt(register long devnum, register long* count);
void update_disk_name_label(Widget w, long devnum);

GC MakeGC(Display* disp, Window wind, long fun, Pixel background, Pixel foreground);

Window GetIconWindow( void );
Window MakeIconWindow( Widget shell, Pixmap pxmap );
void SetIconPixmap( Widget shell, Pixmap pxmap, Bool want_icon_window );

/* Macros */
#define FAILED(status) (((status)&1) == 0)
#define SIGNAL_ERROR(status)  \
    {                         \
      int ___status = status; \
      if (FAILED(___status))  \
      lib$signal(___status);  \
    }

#define MAX_WIDGETS (MAX_WIDGET + 1)

/* Macro to load up an element of a resource list for a resource enquiry */
#define PFM_LOAD_RESOURCE(res,rname,rclass,rtype,rsize,roffset,\
                          defrtype,defraddr) \
{ \
XtResource *_resptr = &(res); /* Only dereference actual resource once */ \
_resptr->resource_name = (rname); \
_resptr->resource_class = (rclass); \
_resptr->resource_type = (rtype); \
_resptr->resource_size = (rsize); \
_resptr->resource_offset = (roffset); \
_resptr->default_type = (defrtype); \
_resptr->default_addr = (defraddr); \
}

/* Global variables */

typedef struct _vms_item_list {		/* Useful for creating item lists*/
    short	 buffer_length;		/* for system service calls */
    short	 item_code;
    void	*buffer;
    int	*returned_buffer_length;
    } VMS_ItemList;

static XtAppContext app_context;

static Pixel state_colour_pxl[MX_STATES];
static char *state_colour_suggestion[MX_STATES] = {
  "YellowGreen",/* Null */
  "Lightgrey",  /* Compatibility - never seen */
  "Black",      /* User mode */
  "Blue",       /* Supervisor mode */
  "Yellow",     /* Executive mode */
  "Orange",     /* Kernel mode */
  "Red"         /* Interrupt mode */
};
static char *state_colour_resource_name[MX_STATES] = {
  "nullColour",     /* Null */
  "compColour",     /* Compatibility - never seen */
  "userColour",     /* User mode */
  "superColour",    /* Supervisor mode */
  "execColour",     /* Executive mode */
  "kernelColour",   /* Kernel mode */
  "intColour"       /* Interrupt mode */
};
static char *state_colour_resource_class[MX_STATES] = {
  "NullColour",     /* Null */
  "CompColour",     /* Compatibility - never seen */
  "UserColour",     /* User mode */
  "SuperColour",    /* Supervisor mode */
  "ExecColour",     /* Executive mode */
  "KernelColour",   /* Kernel mode */
  "IntColour"       /* Interrupt mode */
};

static char  *good_colour_suggestion = "ForestGreen";
static char  *bad_colour_suggestion = "Red";
static char  *neutral_colour_suggestion = "Yellow";
static Pixel good_colour_pxl, bad_colour_pxl, neutral_colour_pxl;

static Widget main_popup_wdgt;       /* The popup exit box */
static Widget contproc_wdgt;         /* The show proc/cont display */
static Widget generic_message_wdgt;  /* generic message box     */
static Widget generic_caution_wdgt;  /* generic caution box     */
static Widget generic_wip_wdgt;      /* generic WIP box         */
static Widget widget_array[MAX_WIDGETS]; /* Place to keep all other widget IDs*/

static GC        cpu_graph_gc[MX_STATES];
static GC        state_graph_gc[MX_STATES];
static GC        good_colour_gc, bad_colour_gc, neutral_colour_gc;

static XtIntervalId contproc_IntervalId;  /* timeout interval id for contproc */
static int topproc;                       /* index into top process array */
static int topPID;                        /* PID of the top process */
static unsigned int timelast;             /* sys time for contproc update */

static Position contproc_x =  50;    /* Initial location of.. */
static Position contproc_y =  50;    /*  ..cont proc window */
static int contproc_update_int;      /* Contproc updates every .. millisecs */

static Position main_xpos = 856;     /* Initial location of.. */
static Position main_ypos =  36;     /*  ..the main window */
static int main_update_secs = 5;     /* Update secs from resource file */
static int main_update_int;          /* Main window updates every .. millisecs */

static long  last_user, last_super, last_exec, last_kernel, last_interrupt;
static long  last_compat, last_spin, last_idle;

static long  new_null;
static long  new_user, new_super, new_exec, new_kernel, new_interrupt;
static long  new_compat, new_idle, new_spin;

static long total_time,NullTime,UserTime,SupeTime,ExecTime,KernTime,
            InteTime,CompTime,DeltaIO;

/* Used for the contproc routines */
static long NewDirIO, DirIO, DeltaDirIO, LastDeltaDirIO;
static long NewBufIO, BufIO, DeltaBufIO, LastDeltaBufIO;
static long NewPpgcnt, Ppgcnt, DeltaPpgcnt, LastDeltaPpgcnt;
static long NewGpgcnt, Gpgcnt, DeltaGpgcnt, LastDeltaGpgcnt;
static long NewPageflts, Pageflts, DeltaPageflts, LastDeltaPageflts;
static long NewCputim, Cputim, DeltaCputim, LastDeltaCputim;
static long NewWSSize, WSSize, DeltaWSSize, LastDeltaWSSize;
static long NewPgfluse, Pgfluse, DeltaPgfluse, LastDeltaPgfluse;

static int CPUPercent[MX_STATES][101];
static int CPUbase[MX_STATES][101];

static long Device_0_IO, Device_1_IO, Device_2_IO, Device_3_IO;

static long PageFaults, PageIO, FreePages, ModPages, TotalPages;

/* The names and addresses of things that MrM has to bind. The names do */
/* not have to be in alphabetical order */
static MrmRegisterArg reglist[] = {
  {"activate_proc",  (XtPointer) activate_proc},
  {"create_proc",    (XtPointer) create_proc},
  {"cancel_proc",    (XtPointer) cancel_proc},
  {"exposure_proc",  (XtPointer) exposure_proc},
  {"toggle_contproc",(XtPointer) toggle_contproc}
};
static int reglist_num = (sizeof reglist/sizeof reglist[0]);

/*----------------------------------------------------------------------------*/
/* Main Program                                                               */
/*----------------------------------------------------------------------------*/
#ifdef VMS
#  ifdef __DECC
#    pragma message save
#    pragma message disable (MAINPROGEXT)
#  endif

PerfMeter(int argc, char *argv[])
MAIN_PROGRAM

#  ifdef __DECC
#    pragma message restore
#  endif
#else

int main(int argc, char *argv[]) /* Standard C version */

#endif
{
    Bool IsColour, want_icon_window;
    static char string[128];
    static unsigned short nlen, mlen, vlen;
    XIconSize *icon_size_list;
    int i, icon_size_count, x_hot, y_hot;
    int update_delay = 2; /* Contproc updates every 2 sec */
    GC gctemp;
    Pixel icon_fg_pxl, icon_bg_pxl;
    Pixmap icon_bitmap, icon_pixmap;
    unsigned int icon_width,icon_height;
    static char vmsversion[8], model[32], nodename[16];
    static long PageSize;
    long status, MegaBytes;
    struct {
      unsigned short bufLen;
      unsigned short itemCode;
      void *bufAddr;
      unsigned short *returnLen;
    } syi_itmlst[] = { {sizeof(nodename)-1,SYI$_SCSNODE,nodename,&nlen},
#if defined(ALPHA) || (__VMS_VER >= 60000000) 
                       {sizeof(TotalPages),SYI$_MEMSIZE,&TotalPages,NULL},
#endif
                       {sizeof(vmsversion)-1,SYI$_VERSION,vmsversion,&vlen},
                       {sizeof(model)-1,SYI$_HW_NAME,model,&mlen},
                       {sizeof(PageSize),SYI$_PAGE_SIZE,&PageSize,NULL},
                       {0, 0, NULL, NULL} };
    union {
      char *string_ptr;
      Bool bool;
    } urs;
    XtResource ursl[MX_STATES+1];
    MrmHierarchy  s_MrmHierarchy;   /* MrM database hierarchy ID */
    MrmType dummy_class;            /* and class variable. */
    char* db_filename_vec[] =             /* MrM heirachy file list. */
               {"PFM$DATA:perf_meter.uid" /* There is only one UID file for */
               };                         /* this application. */
    int db_filename_num = sizeof(db_filename_vec)/sizeof(db_filename_vec[0]);
    Window root_win;
    Display *dpy;
    Screen  *screen;
    Widget toplevel_wdgt, main_window_wdgt;
/*  End of declarations ... */

/*  Get nodename and Total System Memory*/
    status = sys$getsyiw(0,0,0,syi_itmlst,0,0,0);
    if ( FAILED(status) ) PageSize = 512;
    for (i = nlen; (i >= 0) && (nodename[i] == ' ' || !nodename[i] ); i--);
    nodename[++i] = '\0';
    for (i = mlen; (i >= 0) && (model[i] == ' ' || !model[i] ); i--);
    model[++i] = '\0';
    for (i = vlen; (i >= 0) && (vmsversion[i] == ' ' || !vmsversion[i] ); i--);
    vmsversion[++i] = '\0';
    
#if defined(__VAX) && (__VMS_VER < 60000000) 
/*  Following is a kludge to avoid doing any hard work on 5.5-2 systems */
/*  The MMG$GL_MAXMEM under reads a small amount */
/*  but if we assume this is less than 1 Meg we can just add 1 to the answer */
    TotalPages = MMG$GL_MAXMEM;
    MegaBytes = (TotalPages*PageSize) >> 20;
    MegaBytes += 1;
#else
    MegaBytes = (TotalPages*PageSize) >> 20;
#endif

/*  Initialize MrM before initializing the X Toolkit */
    MrmInitialize();

/*  If we had user-defined widgets, we would register them with MrM here */

/*  Initialize the X Toolkit. We get back a top level shell widget */
    toplevel_wdgt = XtAppInitialize(&app_context,PROGRAM_NAME,
                                    (XrmOptionDescList)NULL,0,
                                    &argc,(String*)argv,(String*)NULL,  NULL,0);

/*  Get the X server display */
    dpy = XtDisplay(toplevel_wdgt);

/*  Force synchronization of ALL X calls (use ONLY for DEBUG - very slow) */
/*    XSynchronize(dpy,TRUE);*/

/*  Find screen for this display */
    screen = XtScreen(toplevel_wdgt);

/*  Find the root window for this screen */
    root_win = XRootWindowOfScreen(screen);

/*  Check if we are on a colour or grey-scale system, or B+w */
    if ( ( (XDefaultVisualOfScreen(screen)->class != PseudoColor)
        && (XDefaultVisualOfScreen(screen)->class != DirectColor)
        && (XDefaultVisualOfScreen(screen)->class != TrueColor)
        && (XDefaultVisualOfScreen(screen)->class != GrayScale) ) ||
        (argc > 1) ) {
      IsColour = FALSE;
    } else {
      IsColour = TRUE;
    }

    if (IsColour) {
/*    Create the colours for positive and negative increments */
      PFM_LOAD_RESOURCE(ursl[0],"goodColour","GoodColour",
                          XtRPixel,sizeof(Pixel),0,XtRString,
                          good_colour_suggestion);
      XtGetApplicationResources(toplevel_wdgt,&good_colour_pxl,ursl,1,NULL,0);

      PFM_LOAD_RESOURCE(ursl[0],"badColour","BadColour",
                          XtRPixel,sizeof(Pixel),0,XtRString,
                          bad_colour_suggestion);
      XtGetApplicationResources(toplevel_wdgt,&bad_colour_pxl,ursl,1,NULL,0);
      PFM_LOAD_RESOURCE(ursl[0],"neutralColour","NeutralColour",
                          XtRPixel,sizeof(Pixel),0,XtRString,
                          neutral_colour_suggestion);
      XtGetApplicationResources(toplevel_wdgt,&neutral_colour_pxl,ursl,1,NULL,0);

/*    Get colour for icon pixmap foreground */
      PFM_LOAD_RESOURCE(ursl[0],"iconForeground","IconForeground",
                          XtRPixel,sizeof(Pixel),0,XtRString,
                          "LightGrey");
      XtGetApplicationResources(toplevel_wdgt,&icon_fg_pxl,ursl,1,NULL,0);
      PFM_LOAD_RESOURCE(ursl[0],"iconBackground","IconBackground",
                          XtRPixel,sizeof(Pixel),0,XtRString,
                          "Linen");
      XtGetApplicationResources(toplevel_wdgt,&icon_bg_pxl,ursl,1,NULL,0);
    } else {
      good_colour_pxl = XBlackPixelOfScreen(screen);
      bad_colour_pxl = XBlackPixelOfScreen(screen);
      neutral_colour_pxl = XBlackPixelOfScreen(screen);
    }

/*  Find the colours for different CPU modes */
    for (i = 0; i < MX_STATES; i++) {
      if (IsColour) {
        PFM_LOAD_RESOURCE(ursl[i],state_colour_resource_name[i],
                          state_colour_resource_class[i],
                          XtRPixel,sizeof(Pixel),(i*sizeof(Pixel)),XtRString,
                          state_colour_suggestion[i]);
      } else {
        state_colour_pxl[i] = XBlackPixelOfScreen(screen);
      }
    }
    if (IsColour) XtGetApplicationResources(toplevel_wdgt,state_colour_pxl,
                                            ursl,MX_STATES,NULL,0);

/*  Set title bar */
    sprintf(string,PROGRAM_TITLE_FORMAT,nodename,model,vmsversion,MegaBytes);
    set_something(toplevel_wdgt,XmNtitle,(XtArgVal)string);

/*  Set icon name */
    sprintf(string,"%s %s V%s",nodename,PROGRAM_NAME,PROGRAM_VERSION);
    set_something(toplevel_wdgt,XmNiconName,(XtArgVal)string);

/*  Get preferred icon size and pick biggest of 48, 50 or 64 width pictures */
    if ( !XGetIconSizes( dpy, root_win, &icon_size_list, &icon_size_count) ) {
/*    No preferred size */
      sprintf(string,"PFM$DATA:PERF_METER_ICON64.XBM");
    } else {
/*    Use preferred size */
      if ( icon_size_list->max_width < 64 ) {
        if ( icon_size_list->max_width < 50 ) {
          sprintf(string,"PFM$DATA:PERF_METER_ICON48.XBM");
        } else {
          sprintf(string,"PFM$DATA:PERF_METER_ICON50.XBM");
        }
      } else {
        sprintf(string,"PFM$DATA:PERF_METER_ICON64.XBM");
      }
    }
/*  Create and set an icon bitmap */
    if ( BitmapSuccess == XReadBitmapFile(dpy,root_win,string,
                                          &icon_width,&icon_height,
                                          &icon_bitmap,&x_hot,&y_hot) ) {
      gctemp = MakeGC( dpy, root_win, GXcopy, icon_bg_pxl, icon_fg_pxl);
      icon_pixmap = XCreatePixmap( dpy, root_win, icon_width,icon_height,
                                   XDefaultDepthOfScreen(screen) );
/*    Now copy 1-plane pixmap (icon_bitmap) to normal pixmap */
      XCopyPlane( dpy, icon_bitmap, icon_pixmap, gctemp, 0,0,
                  icon_width,icon_height,0,0,1);
      XFreePixmap(dpy,icon_bitmap);
      XFreeGC(dpy,gctemp);

/*    Create an icon, and if required create a dynamic icon window */
      PFM_LOAD_RESOURCE(ursl[0],"useIconWindow","UseIconWindow",
                          XtRBool,sizeof(Bool),0,XtRString,"True");
      XtGetApplicationResources(toplevel_wdgt,&want_icon_window,ursl,1,NULL,0);
      SetIconPixmap( toplevel_wdgt, icon_pixmap, want_icon_window );
    }

/*  Open the UID files (the output of the UIL compiler) in the hierarchy */
    if (MrmOpenHierarchy(db_filename_num, /* Number of files.            */
                         db_filename_vec, /* Array of file names.        */
                         NULL,            /* Default OS extenstion.      */
                         &s_MrmHierarchy) /* Pointer to returned MrM ID  */
        != MrmSUCCESS) {
      s_error("can't open hierarchy");
    }

/*  Initialize various data structures */
    init_cpu_times();
    initialize_topproc();

/*  Register the items MrM needs to bind for us */
    MrmRegisterNames(reglist, reglist_num);

/*  Go get the main part of the application */
    if (MrmFetchWidget(s_MrmHierarchy, "PERF_METER_MAIN", toplevel_wdgt,
                       &main_window_wdgt, &dummy_class) != MrmSUCCESS)
        s_error("can't fetch main window");

/*  Get the sho proc/cont widget */
    if (MrmFetchWidget(s_MrmHierarchy, "CONTPROC_MAIN", main_window_wdgt,
                       &contproc_wdgt, &dummy_class) != MrmSUCCESS)
        s_error("can't fetch contproc_wdgt");

/*  Manage main window,realize everything. Interface appears on the display */
    XtManageChild(main_window_wdgt);

/*  Get the user specified resources from the default database */

/*  Main window */
    sprintf(string,"%d",main_xpos);
    PFM_LOAD_RESOURCE(ursl[0],"xpos","Xpos",
                        XtRPosition,sizeof(Position),0,XtRString,string);
    XtGetApplicationResources(toplevel_wdgt,&main_xpos,ursl,1,NULL,0);
    sprintf(string,"%d",main_ypos);
    PFM_LOAD_RESOURCE(ursl[0],"ypos","Ypos",
                        XtRPosition,sizeof(Position),0,XtRString,string);
    XtGetApplicationResources(toplevel_wdgt,&main_ypos,ursl,1,NULL,0);
    set_something(toplevel_wdgt,XmNx,main_xpos);
    set_something(toplevel_wdgt,XmNy,main_ypos);

    sprintf(string,"%d",main_update_secs);
    PFM_LOAD_RESOURCE(ursl[0],"UpdatePeriod","UpdatePeriod",
                      XtRInt,sizeof(int),0,XtRString,string);
    XtGetApplicationResources(toplevel_wdgt,&main_update_secs,ursl,1,NULL,0);
    main_update_secs = (main_update_secs > 1) ? main_update_secs : 1;
    main_update_int = main_update_secs*1000;

/*  ContProc window */
    sprintf(string,"%d",contproc_x);
    PFM_LOAD_RESOURCE(ursl[0],"contproc_x","ContProc_x",
                        XtRPosition,sizeof(Position),0,XtRString,string);
    XtGetApplicationResources(toplevel_wdgt,&contproc_x,ursl,1,NULL,0);
    sprintf(string,"%d",contproc_y);
    PFM_LOAD_RESOURCE(ursl[0],"contproc_y","ContProc_y",
                        XtRPosition,sizeof(Position),0,XtRString,string);
    XtGetApplicationResources(toplevel_wdgt,&contproc_y,ursl,1,NULL,0);
    set_something(contproc_wdgt,XmNx,contproc_x);
    set_something(contproc_wdgt,XmNy,contproc_y);

    sprintf(string,"%d",update_delay);
    PFM_LOAD_RESOURCE(ursl[0],"ContProcUpdatePeriod","ContProcUpdatePeriod",
                        XtRInt,sizeof(int),0,XtRString,string);
    XtGetApplicationResources(toplevel_wdgt,&update_delay,ursl,1,NULL,0);
    update_delay = (update_delay > 1) ? update_delay : 1;
    contproc_update_int = update_delay*1000;

    XtRealizeWidget(toplevel_wdgt);

/*  Get windows and set up GCs for the graphs */
    for (i = 1; i < MX_STATES; i++) {
      cpu_graph_gc[i-1] = MakeGC(dpy,XtWindow(widget_array[PFM_CPU_GRAPH]),
                        GXcopy,XWhitePixelOfScreen(screen),state_colour_pxl[i]);
    }
    cpu_graph_gc[6] = cpu_graph_gc[0];
    for (i = 0; i < MX_STATES; i++) {
      state_graph_gc[i] = MakeGC(dpy,XtWindow(widget_array[PFM_STATE_GRAPH]),
                        GXcopy,XWhitePixelOfScreen(screen),state_colour_pxl[i]);
    }
    good_colour_gc = MakeGC(dpy,root_win,GXcopy,
                            XWhitePixelOfScreen(screen),good_colour_pxl);
    XSetArcMode(dpy,good_colour_gc,ArcPieSlice);
    bad_colour_gc = MakeGC(dpy,root_win,GXcopy,
                           XWhitePixelOfScreen(screen),bad_colour_pxl);
    XSetArcMode(dpy,bad_colour_gc,ArcPieSlice);
    neutral_colour_gc = MakeGC(dpy,root_win,GXcopy,
                           XWhitePixelOfScreen(screen),neutral_colour_pxl);
    XSetArcMode(dpy,neutral_colour_gc,ArcPieSlice);

/*  Get the sho proc/cont widget */
    if (MrmFetchWidget(s_MrmHierarchy, "MAIN_POPUP", main_window_wdgt,
                       &main_popup_wdgt, &dummy_class) != MrmSUCCESS)
        s_error("can't fetch main_popup_wdgt");

/*  Hook up the main popup menu */
    XtAddEventHandler(main_window_wdgt, ButtonPressMask, 0,
                      doMainPopupMenu, main_popup_wdgt);

/*  Add timeout callback that will be called every main_update_secs seconds */
    XtAppAddTimeOut( app_context, main_update_int, update,
                     (XtPointer)widget_array[PFM_CPU_GRAPH]/* client data */);

/*  Enter the continous event loop ... */
    XtAppMainLoop(app_context);
}

/*----------------------------------------------------------------------------*/
/* Error reporter - All errors are fatal.                                     */
/*----------------------------------------------------------------------------*/
void s_error(char* problem_string)
{
/*  End of declarations ... */
    printf("%s\n", problem_string);
    exit(EXIT_FAILURE);
}

/*----------------------------------------------------------------------------*/
/* Simplified SET VALUE routine to use only when changing a single attribute. */
/* If we need to change more than one, all new values should be put into one  */
/* arglist and we should make one XtSetValues call (MUCH more efficient).     */
/*----------------------------------------------------------------------------*/
void set_something(Widget w, char* resource, XtArgVal value)
{
    Arg al[1];
/*  End of declarations ... */

    XtSetArg(al[0], resource, value);
    XtSetValues(w, al, 1);
}

/*----------------------------------------------------------------------------*/
/* Simplified SET VALUE routine to use only when changing a label widget's    */
/* label string.  Does a Latin1 String and frees the compound string after    */
/* setting label value                                                        */
/*----------------------------------------------------------------------------*/
void set_label(Widget w, char* value)
{
    Arg al[1];
    XmString latin_label;
/*  End of declarations ... */

    latin_label = XmStringCreateSimple(value);
    XtSetArg(al[0], XmNlabelString, latin_label);
    XtSetValues(w, al, 1);
    XmStringFree(latin_label);
}

/*  Set label widget using index in widget array */
void set_label_str (char *value, int w_idx)
{
    Arg al[1];
    XmString CStr;
/*  End of declarations ... */
    CStr = XmStringCreateSimple(value);
    XtSetArg(al[0], XmNlabelString, CStr);
    XtSetValues(widget_array[w_idx], al, 1);
    XmStringFree(CStr);
}

/*----------------------------------------------------------------------------*/
/* process_num() handles all of the repetitive stuff needed to track          */
/* what's changed from update to update.  We also update the widget           */
/* label if something has changed since the last time.                        */
/*----------------------------------------------------------------------------*/
void process_num(long *NewValue, long *Value, long *DeltaValue,
		 long *LastDeltaValue, int ValueIdx, int DeltaIdx,
		 char *ValueFmt, char *DeltaFmt)
{
    char label_buffer[15];
    int first_time;
/*  End of declarations ... */

    *DeltaValue = *NewValue - *Value;
    first_time = !(*Value);
    if (*DeltaValue) {
      *Value = *NewValue;
      sprintf (label_buffer, ValueFmt, *Value);
      set_label_str( label_buffer, ValueIdx );
    }
    if ( *DeltaValue != *LastDeltaValue ) {
      if (*DeltaValue == 0 || first_time ) {
        label_buffer[0] = ' ';
        label_buffer[1] = '\0';
      } else {
        sprintf (label_buffer, DeltaFmt, *DeltaValue);
      }
      set_label_str( label_buffer, DeltaIdx );

      *LastDeltaValue = *DeltaValue;
    }
}

/*----------------------------------------------------------------------------*/
/* Simplified GET VALUE routine to use only when retrieving a SINGLE attribute*/
/*----------------------------------------------------------------------------*/
void get_something(Widget w, char* resource, XtArgVal value)
{
    Arg al[1];
/*  End of declarations ... */

    XtSetArg(al[0], resource, value);
    XtGetValues(w, al, 1);
}

/*----------------------------------------------------------------------------*/
/* All widgets that are created call back to this procedure. We just log the  */
/* ID in the global array.                                                    */
/*----------------------------------------------------------------------------*/
void create_proc(Widget w, XtPointer client_data, XmAnyCallbackStruct *cb_s)
{
    int widget_num = *(int *)client_data;
/*  End of declarations ... */

    widget_array[widget_num] = w;

    switch ( widget_num ) {
      case PFM_CPU_GRAPH:
        break;
      case PFM_STATE_GRAPH:
        break;
      case PFM_DEVICE_0_IOCOUNT_NAME:
        update_disk_name_label(w, 0);
        break;
      case PFM_DEVICE_1_IOCOUNT_NAME:
        update_disk_name_label(w, 1);
        break;
      case PFM_DEVICE_2_IOCOUNT_NAME:
        update_disk_name_label(w, 2);
        break;
      case PFM_DEVICE_3_IOCOUNT_NAME:
        update_disk_name_label(w, 3);
        break;
        default:
      break;
    }
}

/*----------------------------------------------------------------------------*/
/* exposure_proc() handles exposure events for the cpu histogram and          */
/* state graph.  This is called whenever the window receives an               */
/* exposure event.   We need to redraw the graphics  ourselves.               */
/*----------------------------------------------------------------------------*/
void exposure_proc (Widget w, XtPointer client_data, XmAnyCallbackStruct *cb_s)
{
    int i;
    int ix;
    int index;
    Display *dpy;
    Window window;
/*  End of declarations ... */

    dpy = XtDisplay(w);
    window = XtWindow(w);

    index = *(int *)client_data;
    switch ( index ) {
      case PFM_CPU_GRAPH:
        for (ix = 0; ix < 100; ix++) {
          for (i = 0; i < MX_STATES; i++) {
            if (i == 0) {
              XClearArea(dpy,window,ix,0,1,100,0);
            } else {
              if (CPUPercent[i][ix]) {
                XFillRectangle(dpy,window,cpu_graph_gc[i],
                ix,100-CPUPercent[i][ix]-CPUbase[i][ix],1,CPUPercent[i][ix]);
              }
            }
          }
        }
        break;
      case PFM_STATE_GRAPH:
        DrawStateGraph( XtDisplay(w), XtWindow(w) );
        break;
    }
}

/*----------------------------------------------------------------------------*/
/* All push buttons in this application call back to this routine.  We        */
/* use the client_data to tell us what widget it is, then react accordingly.  */
/*----------------------------------------------------------------------------*/
void activate_proc(Widget w, XtPointer client_data, XmAnyCallbackStruct *cb_s)
{
    int widget_num = *(int *)client_data; /* Convert to widget number. */
/*  End of declarations ... */

    switch ( widget_num ) {
      case PFM_EXIT_BUTTON: exit(EXIT_SUCCESS);
      default: break;
    }
}

/*----------------------------------------------------------------------------*/
/* Not used, apparently                                                       */
/*----------------------------------------------------------------------------*/
void cancel_proc(Widget w, XtPointer client_data, XmAnyCallbackStruct *cb_s)
{
    int widget_num = *(int *)client_data; /* Convert to widget number. */
/*  End of declarations ... */

    switch ( widget_num ) {
      default: break;
    }
}

/*----------------------------------------------------------------------------*/
/* Update the performance meter main statistics - called as a timer callback  */
/*----------------------------------------------------------------------------*/
void update( XtPointer wdgt, XtIntervalId *id )
{
    register int i,j;
    static long NewNullTime,NewUserTime,NewSupeTime,NewExecTime,
                NewKernTime,NewInteTime,NewCompTime,
                NewDevice_0_IO, DeltaDevice_0_IO, LastDeltaDevice_0_IO,
                NewDevice_1_IO, DeltaDevice_1_IO, LastDeltaDevice_1_IO,
                NewDevice_2_IO, DeltaDevice_2_IO, LastDeltaDevice_2_IO,
                NewDevice_3_IO, DeltaDevice_3_IO, LastDeltaDevice_3_IO,
                LastDeltaPageFaults, LastDeltaPageIO,
                LastDeltaFreePages, LastDeltaModPages;

    long NewPageFaults,DeltaPageFaults,NewPageIO,DeltaPageIO,
         NewFreePages, DeltaFreePages,
         NewModPages, DeltaModPages, NewDispFlag, pri0;
    int ix;
    signed int olddelta, delta;
    long CPUtime[MX_STATES], rCPUtime[MX_STATES];
    double fCPUtime[MX_STATES];
    double dfactor, displace;
    char *label_topproc;
    static char label_buffer[32];
    XmString cs;
    Widget cpu_graph_wdgt;
    Display *dpy;
    Window window;
/*  End of declarations ... */

/*  Convert client data back to a display pointer */
    cpu_graph_wdgt = (Widget)wdgt;
    dpy = XtDisplay(cpu_graph_wdgt);
    window = XtWindow(cpu_graph_wdgt);

    get_cpu_times(&NewNullTime,&NewUserTime,&NewSupeTime,&NewExecTime,
                  &NewKernTime,&NewInteTime,&NewCompTime);
    topproc = get_topproc(&pri0);
    pri0 *= 10; /* Dunno why ... */
/*
    printf ("Nl %d U %d S %d E %d K %d I %d C %d 0 %d\n", NewNullTime,
		NewUserTime, NewSupeTime, NewExecTime, NewKernTime,
		NewInteTime, NewCompTime, pri0);
*/
    if ((pri0>0) && (total_time >= pri0))
    {
	NewCompTime += pri0;
	if (NewUserTime>=pri0)
	{   NewUserTime -= pri0;
	} else {
	    pri0 -= NewUserTime;
	    NewUserTime = 0;
	    if (NewSupeTime>=pri0)
	    {	NewSupeTime -= pri0;
	    } else {
		pri0 -= NewSupeTime;
		NewSupeTime = 0;
		if (NewExecTime>=pri0)
		{   NewExecTime -= pri0;
		} else {
		    pri0 -= NewExecTime;
		    NewExecTime = 0;
		    if (NewKernTime>=pri0)
		    {	NewKernTime -= pri0;
		    } else {
			pri0 -= NewKernTime;
			NewKernTime = 0;
			if (NewInteTime>=pri0)
			{   NewInteTime -= pri0;
			} else {
			    pri0 -= NewInteTime;
			    NewInteTime = 0;
			    NewNullTime -= pri0;
			}
		    }
		}
	    }
	};
    };

    NewPageFaults = CURRENT_FAULTS;
    NewPageIO     = CURRENT_PAGEIO;
    NewFreePages  = CURRENT_FREEPAGES;
    NewModPages   = CURRENT_MODPAGES;

    total_time = NewNullTime + NewUserTime + NewSupeTime + NewExecTime +
         NewKernTime + NewInteTime + NewCompTime;
    if (total_time > 0) {
      NullTime = 0.5 + (fCPUtime[0] = (NewNullTime * 100.0) / total_time);
      UserTime = 0.5 + (fCPUtime[1] = (NewUserTime * 100.0) / total_time);
      SupeTime = 0.5 + (fCPUtime[2] = (NewSupeTime * 100.0) / total_time);
      ExecTime = 0.5 + (fCPUtime[3] = (NewExecTime * 100.0) / total_time);
      KernTime = 0.5 + (fCPUtime[4] = (NewKernTime * 100.0) / total_time);
      InteTime = 0.5 + (fCPUtime[5] = (NewInteTime * 100.0) / total_time);
      CompTime = 0.5 + (fCPUtime[6] = (NewCompTime * 100.0) / total_time);
    }
    else {
      NullTime = fCPUtime[0] = NewNullTime;
      UserTime = fCPUtime[1] = NewUserTime;
      SupeTime = fCPUtime[2] = NewSupeTime;
      ExecTime = fCPUtime[3] = NewExecTime;
      KernTime = fCPUtime[4] = NewKernTime;
      InteTime = fCPUtime[5] = NewInteTime;
      CompTime = fCPUtime[6] = NewCompTime;
    }

    rCPUtime[0] = CPUtime[0] = NullTime;
    rCPUtime[1] = CPUtime[1] = UserTime;
    rCPUtime[2] = CPUtime[2] = SupeTime;
    rCPUtime[3] = CPUtime[3] = ExecTime;
    rCPUtime[4] = CPUtime[4] = KernTime;
    rCPUtime[5] = CPUtime[5] = InteTime;
    rCPUtime[6] = CPUtime[6] = CompTime;

    if (total_time > 0) {
	displace = 0.01;
	olddelta = 0;
	while (0 != (delta = rCPUtime[0] + rCPUtime[1] + rCPUtime[2] +
		rCPUtime[3] + rCPUtime[4] + rCPUtime[5] + rCPUtime[6] - 100))
	{
	    if (olddelta*delta < 0)
	    {
		displace *= 0.5;
/*
 consider 49.5%, 49.5%, 1%:
    gives 101% when rounded
    successive adjustments will give 99%,101,99,101,99,...
    until we reach the limit of precision, when 1 +/- displace will be 1,
    and we'll be stuck in an infinite loop.
 I've observed infinite loops, terminations after 28 iterations (including
 18 halvings of displace = approx 4e-8). 
 double precision makes this scenario less likely (since we'll be less likely
 to get identical numbers), but not impossible. What we need is a deadlock
 breaker: when displace gets too small we should break out of the loop and
 manually adjust (one of) the largest numbers
*/

		if (displace<0.00001)
		{
		    ix = 0;
		    for (i = 1; i < MX_STATES; i++)
			if (fCPUtime[ix]<fCPUtime[i]) ix = i;
		    rCPUtime[ix] -= delta;
		    break;
		};
	    };
	    dfactor = (delta>0) ? (1.0-displace) : (1.0+displace);
/*
	    printf ("Old %d Delta %d disp %g dfac %g\n", olddelta,
		delta, displace, dfactor);
*/
	    for (i = 0; i < MX_STATES; i++)
		rCPUtime[i] = 0.5 + (fCPUtime[i] *= dfactor);
	    olddelta = delta;
	}
    }

    for (ix = 0; ix < 100; ix++) {
      for (i = 0; i < MX_STATES; i++) {
        CPUbase[i][100] = 0;
        CPUPercent[i][100] = 100;
        for (j = 0; j < MX_STATES; j++) {
          if (j != i) CPUPercent[i][100] -= rCPUtime[j];
        }
        if (i == 0) {
          XClearArea(dpy,window,ix,0,1,100,0);
        } else {
          if (i > 1) CPUbase[i][100] = rCPUtime[i-1]+CPUbase[i-1][100];
          CPUPercent[i][ix] = CPUPercent[i][ix+1];
          CPUbase[i][ix] = CPUbase[i][ix+1];
          if (CPUPercent[i][ix]) {
            XFillRectangle(dpy,window,cpu_graph_gc[i],
            ix,100-CPUPercent[i][ix]-CPUbase[i][ix],1,CPUPercent[i][ix]);
          }
        }
      }
    }

/*  Get latest top process */
    label_topproc = format_topproc(topproc);
    set_label_str (label_topproc, PFM_CONTPROC_TOGGLE);

/*  Draw state graph on display window */
    DrawStateGraph( dpy, XtWindow(widget_array[PFM_STATE_GRAPH]) );
/*  Draw state graph on icon window too */
    DrawStateGraph( dpy, GetIconWindow() );

    if ((widget_array[PFM_DEVICE_0_IOCOUNT_TOTAL] != NULL) &&
	(widget_array[PFM_DEVICE_0_IOCOUNT_DELTA] != NULL)) {
      get_disk_opcnt(0, &NewDevice_0_IO);
      process_num (&NewDevice_0_IO, &Device_0_IO, &DeltaDevice_0_IO, &LastDeltaDevice_0_IO,
                   PFM_DEVICE_0_IOCOUNT_TOTAL, PFM_DEVICE_0_IOCOUNT_DELTA, " %10d", "  %4d");
    }


    if ((widget_array[PFM_DEVICE_1_IOCOUNT_TOTAL] != NULL) &&
        (widget_array[PFM_DEVICE_1_IOCOUNT_DELTA] != NULL)) {
      get_disk_opcnt(1, &NewDevice_1_IO);
      process_num (&NewDevice_1_IO, &Device_1_IO, &DeltaDevice_1_IO, &LastDeltaDevice_1_IO,
                   PFM_DEVICE_1_IOCOUNT_TOTAL, PFM_DEVICE_1_IOCOUNT_DELTA, " %10d", "  %4d");
    }

    if ((widget_array[PFM_DEVICE_2_IOCOUNT_TOTAL] != NULL) &&
        (widget_array[PFM_DEVICE_2_IOCOUNT_DELTA] != NULL)) {
      get_disk_opcnt(2, &NewDevice_2_IO);
      process_num (&NewDevice_2_IO, &Device_2_IO, &DeltaDevice_2_IO, &LastDeltaDevice_2_IO,
                   PFM_DEVICE_2_IOCOUNT_TOTAL, PFM_DEVICE_2_IOCOUNT_DELTA, " %10d", "  %4d");
    }

    if ((widget_array[PFM_DEVICE_3_IOCOUNT_TOTAL] != NULL) &&
	(widget_array[PFM_DEVICE_3_IOCOUNT_DELTA] != NULL)) {
      get_disk_opcnt(3, &NewDevice_3_IO);
      process_num (&NewDevice_3_IO, &Device_3_IO, &DeltaDevice_3_IO, &LastDeltaDevice_3_IO,
                   PFM_DEVICE_3_IOCOUNT_TOTAL, PFM_DEVICE_3_IOCOUNT_DELTA, " %10d", "  %4d");
    }

    process_num (&NewPageFaults, &PageFaults, &DeltaPageFaults, &LastDeltaPageFaults,
                 PFM_PAGEFAULTS_TOTAL, PFM_PAGEFAULTS_DELTA, " %10d", "  %4d");
    set_something(widget_array[PFM_PAGEFAULTS_DELTA],XmNforeground,
                  (DeltaPageFaults > 0) ? bad_colour_pxl : good_colour_pxl );

    process_num (&NewPageIO, &PageIO, &DeltaPageIO, &LastDeltaPageIO,
                 PFM_PAGEIO_TOTAL, PFM_PAGEIO_DELTA, " %10d", "  %4d");
    set_something(widget_array[PFM_PAGEIO_DELTA],XmNforeground,
                  (DeltaPageIO > 0) ? bad_colour_pxl : good_colour_pxl );

    process_num (&NewFreePages, &FreePages, &DeltaFreePages, &LastDeltaFreePages,
                 PFM_FREEPAGES_TOTAL, PFM_FREEPAGES_DELTA, " %10d", "  %4d");
    set_something(widget_array[PFM_FREEPAGES_DELTA],XmNforeground,
                  (DeltaFreePages < 0) ? bad_colour_pxl : good_colour_pxl );

    process_num (&NewModPages, &ModPages, &DeltaModPages, &LastDeltaModPages,
                 PFM_MODPAGES_TOTAL, PFM_MODPAGES_DELTA, " %10d", "  %4d");
    set_something(widget_array[PFM_MODPAGES_DELTA],XmNforeground,
                    (DeltaModPages > 0) ? bad_colour_pxl : good_colour_pxl );

    XtAppAddTimeOut(app_context, main_update_int, update, wdgt /*client data*/);
}

/*----------------------------------------------------------------------------*/
/* Draw the cpu state graph                                                   */
/*----------------------------------------------------------------------------*/
void DrawStateGraph( Display *dpy, Window window )
{
    static const int iLBarOffset[MX_STATES] = {
      /* NullTimeX */ 12,
      /* CompTimeX */  2,
      /* UserTimeX */ 22,
      /* SupeTimeX */ 32,
      /* ExecTimeX */ 42,
      /* KernTimeX */ 52,
      /* InteTimeX */ 62 }; /* Logical units for display and bar sizes */
    static const int iLBarWidth = 7, iLHeight = 100, iLWidth = 70;
    int iArcDiameter, iFreeArcAngle, iModArcAngle, i;
    long CPUtime[MX_STATES];
    XWindowAttributes xwa;
/*  End of declarations ... */

    if ( window ) {
      if ( XGetWindowAttributes( dpy,  window, &xwa ) ) {
/*      Only bother drawing anything if window is visible */
        if ( IsViewable == xwa.map_state ) {
          CPUtime[0] = NullTime;
          CPUtime[1] = CompTime;
          CPUtime[2] = UserTime;
          CPUtime[3] = SupeTime;
          CPUtime[4] = ExecTime;
          CPUtime[5] = KernTime;
          CPUtime[6] = InteTime;

/*        Draw bar for time in each state */
          for ( i = 0; i < MX_STATES; ++i) {
            DrawStateBar( dpy, window, state_graph_gc[i], xwa.width, xwa.height,
              CPUtime[i], iLBarOffset[i], iLBarWidth, iLWidth, iLHeight );
          }

/*        Draw arc showing memory use (save 64 factor for XFillArc call */
/*        to avoid integer overflow in xxxxArcAngle calculation on */
/*        large memory systems [lucky s*ds]  :-) */
          iArcDiameter = 5*xwa.height/16;
          if ( iArcDiameter % 2 ) iArcDiameter +=1;
          XFillArc(dpy,window,bad_colour_gc, xwa.width-iArcDiameter-2, 1,
                   iArcDiameter,iArcDiameter,0,360*64);
          iFreeArcAngle = FreePages*360/TotalPages;
          XFillArc(dpy,window,good_colour_gc, xwa.width-iArcDiameter-2, 1,
                   iArcDiameter,iArcDiameter,0,iFreeArcAngle*64);
          iModArcAngle = ModPages*360/TotalPages;
          XFillArc(dpy,window,neutral_colour_gc, xwa.width-iArcDiameter-2, 1,
               iArcDiameter,iArcDiameter,iFreeArcAngle*64,iModArcAngle*64);
        }
      }
    }
}

/*----------------------------------------------------------------------------*/
/* Clear existing and draw new bar on the cpu state graph                                          */
/*----------------------------------------------------------------------------*/
void DrawStateBar( Display *dpy, Window window, GC gc,
       int iWindowWidth, int iWindowHeight,                 /* Pixel size */
       int iLValue, int iLX, int iLBarW, int iLW, int iLH ) /* Logical sizes */
{
    int  iBarHeight, iBarWidth, iX, iY;
/*  End of declarations ... */

    iX = iWindowWidth*iLX/iLW;
    iY = iWindowHeight*(iLH-iLValue)/iLH;
    iBarHeight = iWindowHeight*iLValue/iLH;
    iBarWidth = iWindowWidth*iLBarW/iLW;
    XClearArea(dpy,window,iX,0,iLBarW,iWindowHeight,0);
    XFillRectangle(dpy,window,gc,iX,iY,iBarWidth,iBarHeight);
}

/*----------------------------------------------------------------------------*/
/* Create a Graphics Context with appropriate function and foreground         */
/*----------------------------------------------------------------------------*/
GC MakeGC(Display* disp, Window wind, long fun, Pixel background, Pixel foreground)
{
    XGCValues gcvalues;
/*  End of declarations ... */

    gcvalues.background = background;;
    gcvalues.foreground = foreground;;
    gcvalues.function = fun;
    return(XCreateGC(disp,wind,GCFunction|GCForeground|GCBackground,&gcvalues));
}

/*----------------------------------------------------------------------------*/
/* Get first set of CPU stats by setting up a kernel mode routine             */
/*----------------------------------------------------------------------------*/
void init_cpu_times()
{
/*  End of declarations ... */

    last_user = last_super = last_exec = last_kernel = last_interrupt =
    last_compat = last_spin = last_idle = 0;

#ifdef __ALPHA
    sys$cmkrnl(&get_initial_kernelly_stuff, NULL);
#else
    get_initial_kernelly_stuff();
#endif

    last_interrupt -= last_idle;
}

/*----------------------------------------------------------------------------*/
/* This is called at Kernel mode to actually look up the information          */
/*----------------------------------------------------------------------------*/
void get_initial_kernelly_stuff()
{
    register i, j, k;
    long    *PMS$GL_KERNEL;
/*  End of declarations ... */

    for (k = SMP$GL_ACTIVE_CPUS, j = 0; k != 0; j++, k = k >> 1) {
      if ((k & 01) == 0) continue;
#ifdef __ALPHA
      PMS$GL_KERNEL = (SMP$GL_CPU_DATA[j])->cpu$q_kernel;
      last_compat = 0;
#else
      PMS$GL_KERNEL = (SMP$GL_CPU_DATA[j])->cpu$l_kernel;
      last_compat += PMS$GL_KERNEL[C_TIME];
#endif
      last_kernel += PMS$GL_KERNEL[K_TIME];
      last_exec += PMS$GL_KERNEL[E_TIME];
      last_super += PMS$GL_KERNEL[S_TIME];
      last_user += PMS$GL_KERNEL[U_TIME];
      last_interrupt += PMS$GL_KERNEL[I_TIME];
      last_spin += PMS$GL_KERNEL[SPIN_TIME];
/*    here is where the null CPU time is collected.  */
      last_idle += PMS$GL_KERNEL[IDLE_TIME];
    }
}

/*----------------------------------------------------------------------------*/
/* Get next set of CPU stats by setting up a kernel mode routine              */
/*----------------------------------------------------------------------------*/
void get_cpu_times(long* null, long* user, long* super, long* exec,
                          long* kernel, long* interrupt, long* compat)
{
    long new_abstime, totaltime;
/*  End of declarations ... */

    new_user = new_super = new_exec = new_kernel = new_interrupt =
    new_compat = new_spin = new_idle = 0;

#ifdef __ALPHA
    sys$cmkrnl(&get_new_kernelly_stuff, NULL);
#else
    get_new_kernelly_stuff();
#endif

    new_interrupt -= new_idle;

    *user = new_user - last_user; last_user = new_user;
    *super = new_super - last_super; last_super = new_super;
    *exec = new_exec - last_exec; last_exec = new_exec;
    *kernel = new_kernel - last_kernel; last_kernel = new_kernel;
    *interrupt = new_interrupt - last_interrupt; last_interrupt = new_interrupt;
    *compat = new_compat - last_compat; last_compat = new_compat;

    *null = new_idle - last_idle; last_idle = new_idle;
}

/*----------------------------------------------------------------------------*/
/* This is called at Kernel mode to actually look up the information          */
/*----------------------------------------------------------------------------*/
void get_new_kernelly_stuff()
{
    int j, k;
    long *PMS$GL_KERNEL;
/*  End of declarations ... */

    for (k = SMP$GL_ACTIVE_CPUS, j = 0; k != 0; j++, k = k >> 1) {
      if ((k & 01) == 0) continue;
#ifdef __ALPHA
        PMS$GL_KERNEL = (SMP$GL_CPU_DATA[j])->cpu$q_kernel;
        new_compat = 0;
#else
        PMS$GL_KERNEL = (SMP$GL_CPU_DATA[j])->cpu$l_kernel;
        new_compat += PMS$GL_KERNEL[C_TIME];
#endif
        new_kernel += PMS$GL_KERNEL[K_TIME];
        new_exec += PMS$GL_KERNEL[E_TIME];
        new_super += PMS$GL_KERNEL[S_TIME];
        new_user += PMS$GL_KERNEL[U_TIME];
        new_interrupt += PMS$GL_KERNEL[I_TIME];
        new_spin += PMS$GL_KERNEL[SPIN_TIME];
        new_idle += PMS$GL_KERNEL[IDLE_TIME];
    }
}

/*----------------------------------------------------------------------------*/
/* Get disk IO operations                                                     */
/*----------------------------------------------------------------------------*/
void get_disk_opcnt(register long devnum, register long* count)
{
    static struct {
      unsigned short  blen;
      unsigned short  icode;
      long*           buff;
      unsigned short* rlen;
    } dvi_itmlst[] = { {sizeof(int), DVI$_OPCNT, NULL, NULL},
                       {0, 0, NULL, NULL} };
    static struct {
      unsigned short  istat;
      unsigned short  bytes;
      long            fill;
    } iosb;
    static char devname[64];
    $DESCRIPTOR(dev,devname);
    long status;
/*  End of declarations ... */

    sprintf(devname, "PFM$DEVICE_%d_NAME", devnum);
    dvi_itmlst[0].buff = count;
    *count = -1;
    dev.dsc$w_length = strlen(devname);
    status = sys$getdviw(0,0,&dev,dvi_itmlst,&iosb,0,0,0);
    if (status == SS$_IVDEVNAM) {
      *count = 0;
      if (devnum == 0) {
        $DESCRIPTOR(sysdev, "SYS$SYSDEVICE:");
        sys$getdviw(0,0,&sysdev,dvi_itmlst,&iosb,0,0,0);
      }
    }
}

/*----------------------------------------------------------------------------*/
/* Set title for monitored disk IO                                            */
/*----------------------------------------------------------------------------*/
void update_disk_name_label(Widget w, long devnum)
{
    char  devname[64];
    char *title;
/*  End of declarations ... */

    sprintf(devname, "PFM$DEVICE_%d_TITLE", devnum);
    title = getenv(devname);
    if (NULL == title) {
      sprintf(devname, "PFM$DEVICE_%d_NAME", devnum);
      title = getenv(devname);
      if ((NULL == title) && (0 == devnum)) title = "SYS$SYSDEVICE:";
    }
    if (title != NULL) set_label(w,title);
}

/*----------------------------------------------------------------------------*/
/* This callback is invoked when user clicks on the `contproc' toggle         */
/*----------------------------------------------------------------------------*/
void toggle_contproc (Widget w, XtPointer client_data, XmToggleButtonCallbackStruct *toggle)
{
/*  End of declarations ... */
    contproc_off();                  /* Turn off any existing display */
    if (toggle->set) contproc_on (); /* Turn display on if needed */
}

/*----------------------------------------------------------------------------*/
/* Procecures to turn 'show proc/cont' display off, unmanaging the dialog box */
/*----------------------------------------------------------------------------*/
void contproc_off(void)
{
/*  End of declarations ... */

    if ( contproc_IntervalId != 0 ) {
      XtRemoveTimeOut(contproc_IntervalId);
      contproc_IntervalId = 0;
    }
    if ( contproc_wdgt != 0 ) {
      /* Save the window position so we can put it back exactly */
      /* where the use last moved it to, otherwise, it pops up  */
      /* all over the place. */
      if ( XtIsManaged(contproc_wdgt) ) {
        get_something( XtParent(contproc_wdgt), XmNx, (XtArgVal)&contproc_x );
        get_something( XtParent(contproc_wdgt), XmNy, (XtArgVal)&contproc_y );
        XtUnmanageChild( contproc_wdgt );
      }
    }
}

/*----------------------------------------------------------------------------*/
/* contrproc_on()  initializes the 'show proc/cont' display.  That is,        */
/* it looks up the static information for this process, zeros our counters,   */
/* realizes the window, and makes the initial call to the contproc_update     */
/* procedure.                                                                 */
/*----------------------------------------------------------------------------*/
void contproc_on(void)
{
    char text[100];
    int userlen, prcstrt, status;
    unsigned int timequad[2];

    static int pid;
    static char username[12], prcnam[15];
    static VMS_ItemList JPI_static_itemlist[4] = {
        sizeof(pid),      JPI$_PID,      &pid, 0,
        sizeof(username), JPI$_USERNAME, &username, 0,
        sizeof(prcnam),   JPI$_PRCNAM,   &prcnam, 0,
        0, 0, 0, 0};
/*  End of declarations ... */

/*  Zero out `previous' values so process_num doesn't give silly deltas */
    DirIO = BufIO = Ppgcnt = Gpgcnt = Pageflts = Cputim = WSSize = Pgfluse = 0;

    sys$gettim( &timequad[0] );		/* To calc % cpu used */
    timelast = timequad[0];

    /* Remember the PID of the top user - we monitor this same process */
    /* as long the this window is active */
    topPID = topproc_getPID (topproc);

    status = sys$getjpiw (0, &topPID, 0, &JPI_static_itemlist, 0,0,0);
    if (status != SS$_NORMAL) {
      contproc_jpi_status (status);
      return;
    }

    /* Trim trailing spaces from username */
    for (userlen=0; userlen<=sizeof(username) && username[userlen]!=' '; userlen++);

    /* Trim leading spaces from process name */
    for (prcstrt=0; prcstrt<=sizeof(prcnam) && prcnam[prcstrt]==' '; prcstrt++);

    sprintf (text, "PID: %08X  User: %.*s  Process: %.*s\0",
	pid, userlen, username,  (sizeof(prcnam)-prcstrt), (prcnam+prcstrt));
    set_label_str (text, PFM_HEADER1);

    /* Put the window back where it was last time */
    XtManageChild (contproc_wdgt);
    set_something (XtParent(contproc_wdgt), XmNx, contproc_x);
    set_something (XtParent(contproc_wdgt), XmNy, contproc_y);
    XtRealizeWidget (contproc_wdgt);

    DirIO = LastDeltaDirIO = BufIO = LastDeltaBufIO = Ppgcnt =
        LastDeltaPpgcnt = Gpgcnt = LastDeltaGpgcnt = 
        Pageflts = LastDeltaPageflts = Cputim = LastDeltaCputim = 
	WSSize = LastDeltaWSSize = Pgfluse = LastDeltaPgfluse = 0;

    update_contproc(NULL, NULL);		/* get things rolling */
}

/*----------------------------------------------------------------------------*/
/* Timer callback to update stats for cont. monitored `top CPU' process       */
/*----------------------------------------------------------------------------*/
void update_contproc( XtPointer data, XtIntervalId *id )
{
    int status, tmp;
    char CputimStr[12], CpupctStr[6];
    static unsigned int timequad[2], timediff;

    static int imaglen, pagfilcnt, pgflquota;
    static char imagname[80];
    static VMS_ItemList JPI_contproc_itemlist[10] = {
        sizeof(NewDirIO),    JPI$_DIRIO,    &NewDirIO, 0,
        sizeof(NewBufIO),    JPI$_BUFIO,    &NewBufIO, 0,
        sizeof(NewCputim),   JPI$_CPUTIM,   &NewCputim, 0,
        sizeof(NewPpgcnt),   JPI$_PPGCNT,   &NewPpgcnt, 0,
        sizeof(NewGpgcnt),   JPI$_GPGCNT,   &NewGpgcnt, 0,
	sizeof(pagfilcnt),   JPI$_PAGFILCNT,&pagfilcnt, 0,
	sizeof(pgflquota),   JPI$_PGFLQUOTA,&pgflquota, 0,
        sizeof(NewPageflts), JPI$_PAGEFLTS, &NewPageflts, 0,
        sizeof(imagname),    JPI$_IMAGNAME, &imagname, &imaglen,
        0, 0, 0, 0};
/*  End of declarations ... */

    /* We need the change in system time to calc the % cpu used */
    sys$gettim (&timequad[0]);
    timediff = timequad[0] - timelast;
    timelast = timequad[0];

    /* Now get the stats for the top process */
    status = sys$getjpiw (0, &topPID, 0, &JPI_contproc_itemlist, 0,0,0);
    if (status != SS$_NORMAL)
        {
        contproc_jpi_status (status);
        return;
        }

    /* It makes more sense to me to see page file use */
    /* instead of page file quota remaining. */
    NewPgfluse = pgflquota - pagfilcnt;

    /* We show process and group page counts, but it's also useful */
    /* to see the totaled working set size */
    NewWSSize = NewPpgcnt + NewGpgcnt;


    process_num (&NewPpgcnt, &Ppgcnt, &DeltaPpgcnt, &LastDeltaPpgcnt,   
		 PFM_PPGCNT_TOTAL, PFM_PPGCNT_DELTA,  "%8d", "%8d");
    process_num (&NewGpgcnt, &Gpgcnt, &DeltaGpgcnt, &LastDeltaGpgcnt,   
		 PFM_GPGCNT_TOTAL, PFM_GPGCNT_DELTA,  "%8d", "%8d");
    process_num (&NewWSSize, &WSSize, &DeltaWSSize, &LastDeltaWSSize,   
		 PFM_WSSIZE_TOTAL, PFM_WSSIZE_DELTA,  "%8d", "%8d");
    process_num (&NewPageflts, &Pageflts, &DeltaPageflts, &LastDeltaPageflts, 
		 PFM_PGFAULTS_TOTAL, PFM_PGFAULTS_DELTA,"%8d", "%8d");
    process_num (&NewPgfluse, &Pgfluse, &DeltaPgfluse, &LastDeltaPgfluse, 
		 PFM_PGFLUSE_TOTAL, PFM_PGFLUSE_DELTA,"%8d", "%8d");
    process_num (&NewDirIO, &DirIO, &DeltaDirIO, &LastDeltaDirIO,    
		 PFM_DIRIO_TOTAL, PFM_DIRIO_DELTA,   "%8d", "%8d");
    process_num (&NewBufIO, &BufIO, &DeltaBufIO, &LastDeltaBufIO,
    		 PFM_BUFIO_TOTAL, PFM_BUFIO_DELTA, "%8d", "%8d");

    format_cputime (NewCputim, CputimStr);
    set_label_str (CputimStr, PFM_CPUTIME_TOTAL);

    if ( Cputim ) {
      DeltaCputim = NewCputim - Cputim;
    } else {
      DeltaCputim = 0;
    }
    Cputim = NewCputim;
    LastDeltaCputim = DeltaCputim;
    tmp = timediff / 100000;
    sprintf (CpupctStr, " %3d%%\0", (DeltaCputim*100+50) / (tmp ? tmp : 1) );
    set_label_str (CpupctStr, PFM_CPUTIME_DELTA);

    imagname[imaglen] = '\0';
    set_label_str (imagname, PFM_HEADER2);
             
    contproc_IntervalId =
      XtAppAddTimeOut(app_context, contproc_update_int, update_contproc, 0);
}

/*----------------------------------------------------------------------------*/
/* Find state of monitored process, adjust monitoring accordingly             */
/*----------------------------------------------------------------------------*/
void contproc_jpi_status (int status)
{
    static char text[35];
/*  End of declarations ... */

    switch (status) {
        case SS$_SUSPENDED:
            set_label_str ("(Suspended)", PFM_HEADER2);
            contproc_IntervalId = XtAppAddTimeOut(app_context,
              2*contproc_update_int, update_contproc, 0);
            break;
        case SS$_NONEXPR:
            set_label_str ("Nonexistant process", PFM_HEADER1);
            set_label_str ("(display update stopped)", PFM_HEADER2);
            contproc_IntervalId = 0;
            break;
        default:
            sprintf (text, "Error in $GETJPI, stat=%d", status);
            set_label_str (text, PFM_HEADER1);
            set_label_str ("(display update stopped)", PFM_HEADER2);
            contproc_IntervalId = 0;
            break;
    }
}

/*----------------------------------------------------------------------------*/
/* doMainPopupMenu() is the callback whenever a button is clicked in          */
/* one of the graph displays.   All we do here is display a popup menu        */
/* which contains the option to exit.                                         */
/*----------------------------------------------------------------------------*/
void doMainPopupMenu(Widget wdgt,XtPointer client_data,
                     XEvent *event, Boolean *propogate)
{
    Widget popup;
/*  End of declarations ... */

/*  This makes our popup menu visible */
    popup = (Widget)client_data;
    XmMenuPosition (popup, (XButtonPressedEvent *)event);
    XtManageChild(popup);
    XtRealizeWidget(popup);
}

/*----------------------------------------------------------------------------*/
/* format_cputime() takes an integer (t) which represents a time value        */
/* in 10 millisecond ticks, and returns a string (*s) which contains          */
/* the time formatted as hh:mm:ss.hh                                          */
/*----------------------------------------------------------------------------*/
void format_cputime (int t, char *s)
{
    int hour, min, sec, hund, tmp;
/*  End of declarations ... */

    hour = t / 360000;
    tmp = t - hour*360000;
    min = tmp / 6000;
    tmp = tmp - min*6000;
    sec = tmp / 100;
    hund = tmp - sec*100;

    sprintf (s, "%02d:%02d:%02d.%02d\0", hour, min, sec, hund);
}

void SetIconPixmap( Widget shell, Pixmap pxmap, Bool want_icon_window  )
{
    Window icon_window;
    Display *dpy = XtDisplay(shell);
/*  End of declarations ... */

/*  Set icon pixmap as usual */
    XtVaSetValues(shell, XmNiconPixmap, pxmap, NULL);
/*  If we can make an icon window thsi will override the pixmap */
    if ( want_icon_window  &&  (icon_window = MakeIconWindow(shell,pxmap)) ) {
/*    Set windows background pixmap to be the image we want to display */
      XSetWindowBackgroundPixmap( dpy, icon_window, pxmap );
/*    Cause redisplay of window so that update shows up */
      XClearWindow( dpy, icon_window );
    }
}

/*----------------------------------------------------------------------------*/
/* GetIconWindow return window created by a previous call to MakeIconWindow   */
/*----------------------------------------------------------------------------*/
static Window icon_window;
Window GetIconWindow( void )
{
    return( icon_window );
}

/*----------------------------------------------------------------------------*/
/* Makes icon window for top level shell taking size from pixmap              */
/*----------------------------------------------------------------------------*/
Window MakeIconWindow( Widget shell, Pixmap pxmap )
{
    Window root;
    int x, y;
    unsigned int width, height, border_width, depth;
    Display *dpy;
/*  End of declarations ... */

    if ( !icon_window ) {
/*    Get the current window associated with the shell */
      XtVaGetValues(shell, XmNiconWindow, &icon_window, NULL);

      if ( !icon_window ) {
        dpy = XtDisplay(shell);
/*      If no window associated with the shell create one. Make it as big as the*/
/*      pixmap we're going to use.icon window only needs to be a simple window  */
        if ( !XGetGeometry(dpy, pxmap, &root, &x, &y,
                      &width, &height, &border_width, &depth)  ||
             !( icon_window = 
                XCreateSimpleWindow(dpy, root, 0, 0, width, height,
                                    0, CopyFromParent, CopyFromParent)) ) {
          return( icon_window );
        }
/*      Now that the window is created, set it */
        XtVaSetValues( shell, XmNiconWindow, icon_window, NULL);
      }
    }
    return( icon_window );
}
