/*+ VAXTYPES
  Module (VAXTYPES) of FERMI$LIB:LCLCDEF.TLB and file VAXTYPES.H.
 
 This module defines some data types, constants and macros for use
 with VAX C under VAX/VMS.  Note that using these definitions
 renders the program "non-portable".  The "VAXTYPES_DEFINED" symbol
 is defined by this module and can be used to eliminate additional
 calls in other #include modules by:
 
        #ifndef   VAXTYPES_DEFINED
        #include  vaxtypes
        #endif
 
2 Constants
 Defined constants for general VAX C usage:
 
  TRUE       conditional true value (1)
  FALSE      conditional false value (0)
  EOF        End-of-file flag (for standard I/O)
  NULL       Undefined pointer value (0)
 
  MAX_...    Maximum value for several data types:
                  int,ulong,short,uword,char,ubyte
  MIN_...    Minimum value for several data types which
             may take on negative values:
                  int,short,char
 
2 Types
 Additional types (or structures) for use with VAX C
 
  byte / ubyte    signed / unsigned 8-bit byte
  word / uword    signed / unsigned 16-bit word
  long / ulong    signed / unsigned 32-bit longword
  quad / uquad    signed / unsigned 64-bit quadword
  octa / uocta    signed / unsigned 128-bit octaword
 
  address    general address (pointer to byte or char)
  struct queue structure defining a queue header or the queue links
             (fields flink and blink).
  VAX_QUEUE  same as struct queue, but as a typedef.
  vmstime    VMS-returned system time type (long array of [2]).
 
 For simplicity, [u]quad, [u]octa and vmstime are defined as
 arrays of long's (or unsigned long's).  See also $bit, $[u]field
 under the "Macros" subtopic.
 
  vms_iosb    structure defining I/O status block (fields: status,
              count and device_data).
  VMS_IOSB    type definition made from structure vms_iosb.
  item_list_2 structure defining the short format item element from an
              item list (fields: buflen, code, buffer);
  ITEM_LIST_2 type definition made from structure item_list_2.
  item_list_3 structure defining the normal format item element from an
              item list (fields: buflen, code, buffer, retlen);
  ITEM_LIST_3 type definition made from structure item_list_3.
2 Macros
 Macros defined for use with VAX C under VMS
 
 $VMS_SUCCESS(code)     TRUE if the status code has a severity code of
                        success or informational (success indication).
 $VMS_FAILURE(code)     TRUE if the status code has a severity code of
                        warning, error or severe (error indication).
 SIGNAL_FAILURE(code)   if code has an error indication severity, then
                        call LIB$SIGNAL to signal the error.
 ABORT_ON_FAILURE(code) if code has an error indication severity, then
                        call LIB$STOP to signal the error and abort.
 RETURN_ON_FAILURE(code) if code has an error indication severity, then
                        exit the routine returning the status code (use
                        only in routines which return VMS status).
 
  $bit(name)            defines a single bit in a struct (unsigned)
  $field(name,size)     defines a bit field of "size" bits in a struct
  $ufield(name,size)    same as $field but for an unsigned field
  $align                align to next byte field in struct
  $unused(size)         unused field of "size" bits
-*/
/*
  V1.0  08-Feb-83  FJN  Created
  V2.0  09-Feb-83  FJN  Corrected and expanded
  V3.0  22-Feb-83  FJN  Changed names to be similar to VAX PASCAL V2
  V3.1  17-Apr-83  FJN  Added $VMS... macros
  V3.2  22-Jul-83  FJN  Dropped $ from byte/ubyte, etc. types
  V3.3  19-Sep-83  FJN  Added macros $unused, SIGNAL_FAILURE, and
                        ABORT_ON_FAILURE
  V3.4  01-Aug-85  FJN  Added RETURN_ON_FAILURE
  V3.5  19-Jan-87  FJN  Converted to lowercase lib$signal and lib$stop
  V4.0  19-Apr-89  FJN  Added invocation of VAXC$MEM_OPT module, changed
                        to VAXTYPES_DEFINED and added some new macros and
                        definitions
 V4.1   11-Jul-89  FJN  Revert address back to char *
*/
 
#ifndef VAXTYPES_DEFINED
#define VAXTYPES_DEFINED 1      /* To invoke this procedure only once! */
 
#ifndef VAXC$MEM_OPT
#include  "vaxc$mem_opt.h"
#endif
 
typedef char    *address;               /* General address */
 
typedef char    byte;                   /* 8-bit signed byte */
typedef short   word;                   /* 16-bit signed word */
/* typedef long long; */                /* 32-bit signed longword (already
                                           defined by VAX C */
typedef long    quad[2];                /* 64-bit signed quadword */
typedef long    octa[4];                /* 128-bit signed octaword */
 
typedef unsigned char   ubyte;          /* 8-bit unsigned byte */
typedef unsigned short  uword;          /* 16-bit unsigned word */
typedef unsigned long   ulong;          /* 32-bit unsigned longword */
typedef unsigned long   uquad[2];       /* 64-bit unsigned quadword */
typedef unsigned long   uocta[4];       /* 128-bit unsigned octaword */
 
typedef long int vmstime[2];            /* Quadword with VMS time */
 
typedef struct queue {                  /* Queue header/links */
    unsigned int *flink;                /* Forward link pointer */
    unsigned int *blink;                /* Backward link pointer */
    } VAX_QUEUE;
 
 
/*
  Define shorthands for struct bit fields:
*/
#define $bit(name)  unsigned name : 1           /* Single bit */
#define $field(name,size)  int name : size      /* Field of "size" bits */
#define $ufield(name,size) unsigned name : size /* Field of "size" bits */
#define $align  unsigned : 0                    /* Re-align on byte */
#define $unused(size)  unsigned : size          /* Unused "size" bit field */
 
/*
 Some structures used by VMS:
*/
typedef struct vms_iosb {               /* I/O status block */
    unsigned short status;              /* Completion status */
    unsigned short count;               /* Usually bytes transferred count */
    unsigned long device_data;          /* Device dependent data */
    } VMS_IOSB;
 
typedef struct item_list_2 {            /* Short form of item list element */
    unsigned short buflen;              /* Buffer length */
    unsigned short code;                /* Item code */
    void *buffer;                       /* Pointer to buffer */
    } ITEM_LIST_2;
 
typedef struct item_list_3 {            /* Normal form of item list element */
    unsigned short buflen;              /* Buffer length */
    unsigned short code;                /* Item code */
    void *buffer;                       /* Pointer to buffer */
    unsigned long *retlen;              /* Pointer to return length (long) */
    } ITEM_LIST_3;
 
#endif
 
/* Some (re-)definitions from STDIO module (used when STDIO is not called) */
#ifndef EOF
#define TRUE    1
#define FALSE   0
#define EOF     (-1)
#define NULL    0
#endif
 
/* Define constants giving minimum and maximum values for various types */
#ifndef MAX_INT
#define MAX_INT         2147483647      /* Maximum value in signed longword */
#define MAX_ULONG       4294967295      /* Maximum value in unsigned long */
#define MAX_SHORT       32767           /* Maximum value in signed word */
#define MAX_UWORD       65535           /* Maximum value in unsigned word */
#define MAX_CHAR        127             /* Maximum value in signed byte */
#define MAX_UBYTE       255             /* Maximum value in unsigned byte */
#define MIN_INT         -2147483648     /* Minimum value in signed longword */
#define MIN_SHORT       -32768          /* Minimum value in signed word */
#define MIN_CHAR        -128            /* Minimum value in signed byte */
#endif
 
/*
 Define macros used to provide quick test of VMS completion codes
 and macros to signal or abort (signal and stop) on failure codes.
*/
#ifndef $VMS_SUCCESS
#define $VMS_SUCCESS(code)  ((code) & 1)          /* for success */
#define $VMS_FAILURE(code)  (!$VMS_SUCCESS(code)) /* for failure */
                                        /* Signal failure completion code */
#define SIGNAL_FAILURE(code) if ($VMS_FAILURE(code)) lib$signal(code)
                                        /* Abort on failure completion code */
#define ABORT_ON_FAILURE(code) if ($VMS_FAILURE(code)) lib$stop(code)
                                        /* Return failure completion code */
#define RETURN_ON_FAILURE(code) if ($VMS_FAILURE(code)) return(code)
#define LIB$SIGNAL      lib$signal
#define LIB$STOP        lib$stop
#endif
