/*  C_MY_CREATE_STATS_SUB.C
 *
 *  Subroutine to create shiny new stats tables
 *  in the Mega Zillionare MySQL database.
 */
/*
 *      ANSI headers
 */
#include <stdlib>
#include <stdio>
#include <string>
#include <time>

/*
 *      VMS headers
 */

/*
 *  Our headers
 */
#include <mysql.h>

#pragma member_alignment save
#pragma nomember_alignment
#pragma dictionary "cdd_records.zillionare_stats_record"
#pragma member_alignment restore


/*;;;;;
 *  Program constants and global data
 *;;;;;
 */
#define ELM_COUNT 52

/*;;;;;
 *      Function prototypes
 *;;;;;
 */
void calc_final_stats( struct zillionare_stats_record *d_stats,
                       struct zillionare_stats_record *m_stats,
                       int l_draw_no);

void initialize_stat_arrays( struct zillionare_stats_record *d_stats,
                             struct zillionare_stats_record *m_stats);

void update_dstats( struct zillionare_stats_record *m_stats, 
                    int l_no, int l_draw_no);

void update_mstats( struct zillionare_stats_record *m_stats, 
                    int l_no, int l_draw_no);

/*;;;;;
 *      Functions and subroutines
 *;;;;;
 */
void c_my_create_stats_sub( MYSQL *conn)
{
    int                     l_x, l_sub, l_draw_no,
                            l_no_1, l_no_2, l_no_3, l_no_4,
                            l_no_5, l_mega_no;
    char                    command_str[1024];


    MYSQL_STMT              *stmt;
    MYSQL_RES               *prep_meta_result;
    MYSQL_BIND              bind[6];
    my_bool                 is_null[6];
    unsigned long           fld_length[6];


    const char              *SELECT_STR = 
               "select no_1, no_2, no_3, no_4, no_5, mega_no\
               from drawing_data order by draw_dt;";

    //
    //  Arrays to hold stats records
    //
    struct zillionare_stats_record  d_stats[ ELM_COUNT];
    struct zillionare_stats_record  m_stats[ ELM_COUNT];




    initialize_stat_arrays( d_stats, m_stats);

    //
    //  Nuke and recreate tables
    //
    l_x = mysql_query( conn, "DROP TABLE IF EXISTS DRAW_STATS;");
    printf( "Result of drop %d\n", mysql_errno( conn));
    l_x = mysql_commit( conn);

    l_x = mysql_query( conn, "DROP TABLE IF EXISTS MEGA_STATS;");
    printf( "Result of drop %d\n", mysql_errno( conn));
    l_x = mysql_commit( conn);

    l_x = mysql_query( conn, "create table draw_stats\
                                (elm_no          int     not null,\
                                 hit_count       int     not null,\
                                 last_draw_no    int     not null,\
                                 since_last      int     not null,\
                                 curr_seq        int     not null,\
                                 longest_seq     int     not null,\
                                 pct_hits        double  not null,\
                                 max_btwn        int     not null,\
                                 ave_btwn        double  not null,\
                                 primary key k0 (elm_no));");

    printf( "Result of creating draw_stats %d\n", mysql_errno( conn));


    l_x = mysql_query( conn, "create table mega_stats\
                                (elm_no          int     not null,\
                                 hit_count       int     not null,\
                                 last_draw_no    int     not null,\
                                 since_last      int     not null,\
                                 curr_seq        int     not null,\
                                 longest_seq     int     not null,\
                                 pct_hits        double  not null,\
                                 max_btwn        int     not null,\
                                 ave_btwn        double  not null,\
                                 primary key k0 (elm_no));");


    printf( "Result of creating mega_stats %d\n", mysql_errno( conn));


    //
    //  Prepare an SQL statement
    //
    stmt = mysql_stmt_init( conn);
    if ( !stmt)
    {
        printf( "Unable to init sql statementn");
        return;
    }  /* end test for stmt init */

    if ( mysql_stmt_prepare(stmt, SELECT_STR, strlen( SELECT_STR)))
    {
        printf( "failed to prepare select statement\n");
        printf( "%s\n", mysql_stmt_error(stmt));
        return;
    }  /* end test for prepare success */

    prep_meta_result = mysql_stmt_result_metadata(stmt);
    if ( !prep_meta_result)
    {
        printf( "No meta information found\n");
        printf( "%s\n", mysql_stmt_error( stmt));
        return;
    }  /* end test for meta result */


    //
    //  Execute the select
    //
    if ( mysql_stmt_execute( stmt))
    {
        printf( "Error executing statement\n");
        printf( "%s\n", mysql_stmt_error( stmt));
        return; 
    }  /* end test for successfull execute */

    memset( bind, 0, sizeof( bind));

    //
    //  Set up the bind array, one element for each column
    //  We have a not NULL constraint on each column
    //  so technically we could have skipped the is_null
    //  elements, but best to give the server some place
    //  to put the result.
    //
    bind[0].buffer_type     = MYSQL_TYPE_LONG;
    bind[0].buffer          = (char *)&l_no_1;
    bind[0].is_null         = &is_null[0];
    bind[0].length          = &fld_length[0];

    bind[1].buffer_type     = MYSQL_TYPE_LONG;
    bind[1].buffer          = (char *)&l_no_2;
    bind[1].is_null         = &is_null[1];
    bind[1].length          = &fld_length[1];

    bind[2].buffer_type     = MYSQL_TYPE_LONG;
    bind[2].buffer          = (char *)&l_no_3;
    bind[2].is_null         = &is_null[2];
    bind[2].length          = &fld_length[2];

    bind[3].buffer_type     = MYSQL_TYPE_LONG;
    bind[3].buffer          = (char *)&l_no_4;
    bind[3].is_null         = &is_null[3];
    bind[3].length          = &fld_length[3];

    bind[4].buffer_type     = MYSQL_TYPE_LONG;
    bind[4].buffer          = (char *)&l_no_5;
    bind[4].is_null         = &is_null[4];
    bind[4].length          = &fld_length[4];

    bind[5].buffer_type     = MYSQL_TYPE_LONG;
    bind[5].buffer          = (char *)&l_mega_no;
    bind[5].is_null         = &is_null[5];
    bind[5].length          = &fld_length[5];

    //
    //  bind the result buffers
    //
    if (mysql_stmt_bind_result( stmt, bind))
    {
        printf( "Unable to bind results\n");
        printf( "%s\n", mysql_stmt_error(stmt));
        return;
    }  /* end test for successful bind */

    //
    //  Store the results on the client
    //
    if (mysql_stmt_store_result( stmt))
    {
        printf( "Error storing results\n");
        printf( "%s\n", mysql_stmt_error( stmt));
        return;
    }  /* end test for successful store */


    //
    //  Read through until end of input.
    //  Update each array element as found.
    //
    l_draw_no = 0;

    while ( !mysql_stmt_fetch(stmt))
    {
            l_draw_no++;
            //
            //  if we wanted to be iron clad here we
            //  would only call update if the column 
            //  wasn't null.  Since we have not null
            //  constraint on the column, I feel safe
            //  doing this.
            //
            update_dstats( d_stats, l_no_1, l_draw_no);
            update_dstats( d_stats, l_no_2, l_draw_no);
            update_dstats( d_stats, l_no_3, l_draw_no);
            update_dstats( d_stats, l_no_4, l_draw_no);
            update_dstats( d_stats, l_no_5, l_draw_no);

            update_mstats( m_stats, l_mega_no, l_draw_no);


            if ( (l_draw_no % 100) == 0)
                printf( "Processed %d records\n", l_draw_no);

    }  /* end while stmt_fetch */


    printf( "Processed %d records\n", l_draw_no);     // show final count

    mysql_free_result(prep_meta_result);
    mysql_stmt_close(stmt);

    calc_final_stats( d_stats, m_stats, l_draw_no);

    //
    //  Write the new records
    //
    for( l_sub=0; l_sub < ELM_COUNT; l_sub++)
    {

        sprintf( command_str, "INSERT INTO DRAW_STATS SET elm_no=%d,\
                               hit_count=%d, last_draw_no=%d,\
                               since_last=%d, curr_seq=%d,\
                               longest_seq=%d, pct_hits=%f,\
                               max_btwn=%d, ave_btwn=%f;",
                  d_stats[ l_sub].elm_no,
                  d_stats[ l_sub].hit_count,
                  d_stats[ l_sub].last_draw_no,
                  d_stats[ l_sub].since_last,
                  d_stats[ l_sub].curr_seq,
                  d_stats[ l_sub].longest_seq,
                  d_stats[ l_sub].pct_hits,
                  d_stats[ l_sub].max_btwn,
                  d_stats[ l_sub].ave_btwn);


        l_x = mysql_query( conn, command_str);

        if (l_x != 0)
        {
            printf( "Error writing drawing_stat\n");
            printf( "%s\n", mysql_error( conn));
        }

        sprintf( command_str, "INSERT INTO MEGA_STATS SET elm_no=%d,\
                               hit_count=%d, last_draw_no=%d,\
                               since_last=%d, curr_seq=%d,\
                               longest_seq=%d, pct_hits=%f,\
                               max_btwn=%d, ave_btwn=%f;",
                  m_stats[ l_sub].elm_no,
                  m_stats[ l_sub].hit_count,
                  m_stats[ l_sub].last_draw_no,
                  m_stats[ l_sub].since_last,
                  m_stats[ l_sub].curr_seq,
                  m_stats[ l_sub].longest_seq,
                  m_stats[ l_sub].pct_hits,
                  m_stats[ l_sub].max_btwn,
                  m_stats[ l_sub].ave_btwn);


        l_x = mysql_query( conn, command_str);

        if (l_x != 0)
        {
            printf( "Error writing mega_stat\n");
            printf( "%s\n", mysql_error( conn));
        }

    }  /* end for l_sub loop */

    l_x = mysql_commit( conn);

}  /* end c_my_create_stats_sub */

/*;;;;;
 *  Subroutine to initialize the arrays which will be written to disk.
 *;;;;;
 */
void initialize_stat_arrays( struct zillionare_stats_record *d_stats,
                             struct zillionare_stats_record *m_stats)
{
    int     l_x;

    for (l_x=0; l_x < ELM_COUNT; l_x++)
    {
        memset( &d_stats[ l_x], 0, sizeof( struct zillionare_stats_record));

        d_stats[ l_x].elm_no        = l_x + 1;
        d_stats[ l_x].pct_hits      = 0.0;
        d_stats[ l_x].ave_btwn      = 0.0;


        memset( &m_stats[ l_x], 0, sizeof( struct zillionare_stats_record));

        m_stats[ l_x].elm_no        = l_x + 1;
        m_stats[ l_x].pct_hits      = 0.0;
        m_stats[ l_x].ave_btwn      = 0.0;

    }  /* end for l_x loop */

}  /* end initialize_stat_arrays subroutine */

/*;;;;;
 *  Subroutine to update a drawing stats record
 *;;;;;
 */
void update_dstats( struct zillionare_stats_record *d_stats, 
                    int l_no, int l_draw_no)
{
    int     l_since, l_sub;


    l_sub   = l_no - 1;
    l_since = l_draw_no - d_stats[ l_sub].last_draw_no;

    //
    //  Sequence or random hit
    //
    if ( l_since == 1)
    {
        d_stats[ l_sub].curr_seq++;
        if ( d_stats[ l_sub].curr_seq > d_stats[ l_sub].longest_seq)
            d_stats[ l_sub].longest_seq = d_stats[ l_sub].curr_seq;
    }
    else
    {
        d_stats[ l_sub].curr_seq = 0;
        if ( l_since > d_stats[ l_sub].max_btwn)
            d_stats[ l_sub].max_btwn = l_since;

    }  /* end test for sequence or random hit */

    d_stats[ l_sub].hit_count++;
    d_stats[ l_sub].last_draw_no = l_draw_no;
    d_stats[ l_sub].since_last = l_since;

}  /* end update_dstats subroutine */

/*;;;;;
 *  Subroutine to update a drawing stats record
 *;;;;;
 */
void update_mstats( struct zillionare_stats_record *m_stats, 
                    int l_no, int l_draw_no)
{
    int     l_since, l_sub;


    l_sub   = l_no - 1;
    l_since = l_draw_no - m_stats[ l_sub].last_draw_no;

    //
    //  Sequence or random hit
    //
    if ( l_since == 1)
    {
        m_stats[ l_sub].curr_seq++;
        if ( m_stats[ l_sub].curr_seq > m_stats[ l_sub].longest_seq)
            m_stats[ l_sub].longest_seq = m_stats[ l_sub].curr_seq;
    }
    else
    {
        m_stats[ l_sub].curr_seq = 0;
        if ( l_since > m_stats[ l_sub].max_btwn)
            m_stats[ l_sub].max_btwn = l_since;

    }  /* end test for sequence or random hit */

    m_stats[ l_sub].hit_count++;
    m_stats[ l_sub].last_draw_no = l_draw_no;
    m_stats[ l_sub].since_last = l_since;

}  /* end update_mstats subroutine */

/*;;;;;
 *  Subroutine to calculate final drawing stats information
 *;;;;;
 */
void calc_final_stats( struct zillionare_stats_record *d_stats,
                       struct zillionare_stats_record *m_stats,
                       int l_draw_no)
{
    int     l_x, l_missed;
    double  d_1, d_2;

    for( l_x=0; l_x < ELM_COUNT; l_x++)
    {
        d_1 = d_stats[ l_x].hit_count;
        d_2 = l_draw_no;

        l_missed = l_draw_no - d_stats[ l_x].hit_count;

        d_stats[ l_x].pct_hits = (double) d_stats[ l_x].hit_count / (double) l_draw_no;

        d_stats[ l_x].ave_btwn = (double) l_missed / 
                                 (double) d_stats[ l_x].hit_count;

        d_stats[ l_x].since_last = l_draw_no - d_stats[ l_x].last_draw_no;


        d_1 = m_stats[ l_x].hit_count;
        d_2 = l_draw_no;

        l_missed = l_draw_no - m_stats[ l_x].hit_count;

        m_stats[ l_x].pct_hits = (double) m_stats[ l_x].hit_count / (double) l_draw_no;

        m_stats[ l_x].ave_btwn = (double) l_missed / 
                                 (double) m_stats[ l_x].hit_count;

        m_stats[ l_x].since_last = l_draw_no - m_stats[ l_x].last_draw_no;


    }  /* end for l_x loop */


}  /* end calc_final_stats subroutine */
