/****************************************************************************/
/*                                                                          */
/* Copyright Digital Equipment Corporation 1996.       All rights reserved. */
/*                                                                          */
/* Restricted Rights: Use, duplication, or disclosure by the U.S. Government*/
/* is subject to restrictions as set forth in subparagraph (c) (1) (ii) of  */
/* DFARS 252.227-7013, or in FAR 52.227-19, or in FAR 52.227-14 Alt. III, as*/
/* applicable.                                                              */
/*                                                                          */
/* This software is proprietary to and embodies the confidential technology */
/* of Digital Equipment Corporation.  Possession, use, or copying of this   */
/* software and media is authorized only pursuant to a valid written license*/
/* from Digital or an authorized sublicensor.                               */
/*                                                                          */
/* spi_example_init.c                                                       */
/****************************************************************************/

#ifdef VMS
#ifndef SUCCESS
#define SUCCESS 1
#endif

#ifndef FAILURE
#define FAILURE 0
#endif
#endif

#ifdef unix
#ifndef SUCCESS
#define SUCCESS 0
#endif

#ifndef FAILURE
#define FAILURE 1
#endif
#endif

#include <stdio.h>
#include <stddef.h>
#include <stdlib.h>
#include <string.h>

#include <osak_api.h>
#include <osak_api_codes.h>
#include <osak_api_message.h>

#define TRUE 1
#define FALSE 0
#define OSAK_EXAMPLE_WS_SIZE 1024
#define OSAK_EXAMPLE_BUFFER_SIZE 2048
#define OSAK_EXAMPLE_TIMEOUT 60
#define OSAK_TTNAME_SIZE 7

void open_initiator (osak_port *port, struct osak_parameter_block *pb);
unsigned long int connect_req (osak_port port, 
                               struct osak_parameter_block *pb);
unsigned long int release_req (osak_port port, 
                               struct osak_parameter_block *pb);
unsigned long int give_buffer (osak_port port);
void reuse_buffers (struct osak_buffer **ptr);
void wait_for_event (osak_port port, struct osak_parameter_block *pb, 
                     int queued);
void *alloc_memory (unsigned long int size);
unsigned long int free_memory (unsigned long int size, void *ptr);

extern unsigned long int spi_open_initiator ();
extern unsigned long int spi_give_buffers ();
extern unsigned long int spi_connect_req ();
extern unsigned long int spi_get_event ();
extern unsigned long int spi_collect_pb ();
extern unsigned long int spi_select ();
extern unsigned long int spi_close_port ();
extern unsigned long int spi_release_req ();

unsigned char remote_nsap[] =
    { 0x49, 0x00, 0x04, 0xAA, 0x00, 0x04, 0x00, 0x33, 0x10, 0x21 };

unsigned char user_information[] =
    { 'I', 'n', 'i', 't', 'i', 'a', 't', 'o', 'r', ' ', 't', 'o', ' ',
      'R', 'e', 's', 'p', 'o', 'n', 'd', 'e', 'r'
    };

/*--------------------------------------------------------------------------*/
/* Global variables                                                         */
/*--------------------------------------------------------------------------*/
    /* Local AEI */
struct osak_aei local_address;
 
    /* Remote AEI */
struct osak_aei remote_address;

    /* Transport template */
struct osak_transport_templates transport_template;

    /* Protocol versions */
struct osak_protocol_versions protocol_versions;

   /* Structure to hold functional units. Functional units include things */
   /* like what kind of data we're using, whether the data transfer is    */
   /* duplex or half duplex, etc.                                         */
struct osak_fus fus;

    /* Data to send on the connect req */
struct osak_buffer send_buffer;

    /* Queue of buffers for spi_give_buffers.  These are used in the    */
    /* routines give_buffer and reuse_buffers.                          */
struct osak_buffer *free_buffers = NULL;
struct osak_buffer *free_buffers_end = NULL;
   
/****************************************************************************/
/* FUNCTION: main                                                           */
/****************************************************************************/
main (int argc, char *argv[])
{
   
   unsigned long int status;                /* To hold return values from  */
                                            /* routines                    */
   struct osak_parameter_block *in_pb;      /* Parameter block to use in   */
                                            /* calls to spi_get_event      */
   struct osak_parameter_block *out_pb;     /* Parameter block to use in   */  
                                            /* outbound calls              */
   osak_port port;                          /* To hold the port handle     */
                                            /* returned from               */
                                            /* spi_open_initiator          */
   struct osak_buffer *buffer_ptr;          /* To hold list of buffers     */
                                            /* returned from               */
                                            /* spi_close_port              */
   struct osak_parameter_block *pb_ptr;     /* To hold list of parameter   */
                                            /* blocks returned from        */
                                            /* spi_close_port              */
   int need_to_collect_pb;                  /* Flag for call to            */
                                            /* wait_for_event. TRUE if the */
                                            /* call to spi_select should   */
                                            /* wait for OSAK_C_WRITEEVENT, */
                                            /* to guarantee the completion */

   /*----------------------------------------------------------------------*/
   /* Allocate a parameter block.  The same parameter block is used on all */
   /* outbound calls to OSAK.  Any second failure to allocate an out_pb    */
   /* (or in_pb) may warrant some memory clean-up.                         */
   /*----------------------------------------------------------------------*/
   out_pb = (struct osak_parameter_block *)
       malloc (sizeof(struct osak_parameter_block) + OSAK_EXAMPLE_WS_SIZE);
   if (out_pb == NULL)
   {
      printf("Failed to allocate outbound parameter block.\n");
      exit(FAILURE);
   }
   
   /*----------------------------------------------------------------------*/
   /* Allocate a parameter block.  The parameter block is used on all      */
   /* inbound calls to OSAK. Any second failure to allocate an in_pb       */
   /* (or out_pb) may warrant some memory clean-up.                        */
   /*----------------------------------------------------------------------*/
   in_pb = (struct osak_parameter_block *)
      malloc (sizeof(struct osak_parameter_block) + OSAK_EXAMPLE_WS_SIZE);
   if (in_pb == NULL)
   {
      printf("Failed to allocate inbound parameter block.\n");
      exit(FAILURE);
   }

   /*-----------------------------------------------------------------------*/
   /* Open Initiator                                                        */
   /*-----------------------------------------------------------------------*/
   printf ("Opening an initiator port.\n");
   open_initiator (&port, out_pb);

   /*-----------------------------------------------------------------------*/
   /* Connect Request                                                       */
   /*-----------------------------------------------------------------------*/
   printf("Sending S-CONNECT request.\n");
   status = connect_req(port, out_pb);
   if (status == OSAK_S_QUEUED)
      need_to_collect_pb = TRUE;
   else
      need_to_collect_pb = FALSE;


   /*-----------------------------------------------------------------------*/
   /* Wait for accept cnf and collect connect_req pb                        */
   /*-----------------------------------------------------------------------*/
   printf("Waiting for S-CONNECT-ACCEPT confirm.\n");
   if (need_to_collect_pb)
      printf("Also waiting for completion of S-CONNECT request.\n");
   
   /* Since wait_for_event contains more that one call to spi routines,     */
   /* the status's of the individual calls are checked inside               */
   /* wait_for_event.                                                       */ 

   wait_for_event (port, in_pb, need_to_collect_pb);
   /* reuse_buffers does not call any spi routines. It is not necessary     */
   /* to check the status of this call.                                     */
   reuse_buffers (&in_pb->tsdu_ptr);

   if (in_pb->event_type != OSAK_C_ACCEPT_CNF)
   {
      printf("Expected S-CONNECT-ACCEPT confirm. Received some other event.\n");
      exit(FAILURE);
   }
   printf("Received S-CONNECT-ACCEPT confirm.\n");
   /* reuse_buffers does not call any spi routines. It is not necessary     */
   /* to check the status of this call.                                     */
   reuse_buffers (&in_pb->tsdu_ptr);
   
   /*----------------------------------------------------------------------*/
   /* Release the association                                              */
   /*----------------------------------------------------------------------*/
   printf("Sending S-RELEASE request.\n");
   status = release_req (port, out_pb);
   if (status == OSAK_S_QUEUED)
      need_to_collect_pb = TRUE;
   else
      need_to_collect_pb = FALSE;


   /*----------------------------------------------------------------------*/
   /* Wait for release-cnf and collect release_req pb                      */
   /*----------------------------------------------------------------------*/
   printf("Waiting for S-RELEASE confirm.\n");
   if (need_to_collect_pb)
      printf("Also waiting for completion of S-RELEASE request.\n");
   
   /* Since wait_for_event contains more that one call to spi routines,     */
   /* the status's of the individual calls are checked inside               */
   /* wait_for_event.                                                       */ 

   wait_for_event (port, in_pb, need_to_collect_pb);
   /* reuse_buffers does not call any spi routines. It is not necessary     */
   /* to check the status of this call.                                     */
   reuse_buffers (&in_pb->tsdu_ptr);

   if (in_pb->event_type != OSAK_C_RELEASE_CNF)
   {
      printf("Expected S-RELEASE confirm. Received some other event.\n");
      exit(FAILURE);
   }
   printf("Received S-RELEASE confirm.\n");
   /* reuse_buffers does not call any spi routines. It is not necessary     */
   /* to check the status of this call.                                     */
   reuse_buffers (&in_pb->tsdu_ptr);

   /*----------------------------------------------------------------------*/
   /* Close the port                                                       */
   /*----------------------------------------------------------------------*/
   printf("Closing initiator port.\n");
   status = spi_close_port (port, &buffer_ptr, &pb_ptr, 
                            OSAK_C_NON_DESTRUCTIVE);
   if (status != OSAK_S_NORMAL)
   {
      printf("Call to spi_close_port failed.\n");
      exit(FAILURE);
   }
   /* reuse_buffers does not call any spi routines. It is not necessary     */
   /* to check the status of this call.                                     */
   reuse_buffers (&buffer_ptr);
}
/************************************************************************/
/* FUNCTION: open_initiator                                             */
/*                                                                      */
/* This routine sets up the parameter block for a call to               */
/* spi_open_initiator and makes the call.                                     
/************************************************************************/
void open_initiator (osak_port *port, struct osak_parameter_block *pb)
{
   unsigned long int status;
   
   /* initialize parameter block */
   memset ((void *)pb, 0,
          sizeof(struct osak_parameter_block) + OSAK_EXAMPLE_WS_SIZE);
   pb->pb_length = sizeof(struct osak_parameter_block);
   pb->ws_length = OSAK_EXAMPLE_WS_SIZE;
   pb->api_version = OSAK_C_API_VERSION_3;
   pb->alloc_rtn = (osak_rtn)alloc_memory;
   pb->dealloc_rtn = (osak_rtn)free_memory;
   
   status = spi_open_initiator (port, pb);
   if (status != OSAK_S_NORMAL)
   {
      printf("Call to spi_open_initiator failed.\n");
      exit(FAILURE);
   }

   return;
}
   
/************************************************************************/
/* FUNCTION:  connect_req                                               */
/*                                                                      */
/* This routine sets up the parameter block for a call to               */
/* spi_connect_req and makes the call.                                  */
/************************************************************************/
unsigned long int connect_req (osak_port port, struct osak_parameter_block *pb)
{
   unsigned long int status;
   
   /* Set up local address */
   memset ((void *)&local_address, 0, sizeof(struct osak_aei));
   local_address.paddress.psel.size = sizeof(unsigned long int) ;
   local_address.paddress.psel.pointer = (unsigned char *)"INIT-PSEL" ;
   local_address.paddress.ssel.size = sizeof(unsigned long int);
   local_address.paddress.ssel.pointer = (unsigned char *)"INIT-SSEL";
   local_address.paddress.tsel.size = sizeof(unsigned long int);
   local_address.paddress.tsel.pointer = (unsigned char *)"INIT-TSEL";
   local_address.paddress.nsap.type = OSAK_C_CLNS;
   
   /* Set up peer address (the responder's address) */
   memset((void *)&remote_address, 0, sizeof(struct osak_aei));
   remote_address.paddress.psel.size = sizeof(unsigned long int);
   remote_address.paddress.psel.pointer = (unsigned char *)"RESP-PSEL";
   remote_address.paddress.ssel.size = sizeof(unsigned long int);
   remote_address.paddress.ssel.pointer = (unsigned char *)"RESP-SSEL";
   remote_address.paddress.tsel.size = sizeof(unsigned long int);
   remote_address.paddress.tsel.pointer = (unsigned char *)"RESP-TSEL";
   remote_address.paddress.nsap.id.size = sizeof(remote_nsap);
   remote_address.paddress.nsap.id.pointer = remote_nsap;
   remote_address.paddress.nsap.type = OSAK_C_CLNS;

   /* Set up transport template */
   transport_template.next = NULL;
   transport_template.name.size = OSAK_TTNAME_SIZE;
   transport_template.name.pointer = (unsigned char *)"Default";

   /* Set up protocol versions */
   /* Select session version 2 */
   protocol_versions.sversion.version1 = FALSE;
   protocol_versions.sversion.version2 = TRUE;

   /* Set up functional unit */
   /* Zero out all functional units before setting those required */
   memset ((void *)&fus, 0, sizeof(struct osak_fus));
   
   /* Request either duplex or half duplex.  In this example we are */
   /* actually expecting the responder to accept with duplex.       */
   fus.duplex = TRUE;
   fus.half_duplex = TRUE;

   /* Set up the buffer containing the data to send */
   send_buffer.next = NULL;
   send_buffer.buffer_ptr = &user_information[0];
   send_buffer.buffer_length = sizeof(user_information);
   send_buffer.data_ptr = &user_information[0];
   send_buffer.data_length = sizeof(user_information); 
  
   /* initialize parameter block */
   memset ((void *)pb, 0,
          sizeof(struct osak_parameter_block) + OSAK_EXAMPLE_WS_SIZE);
   pb->pb_length = sizeof(struct osak_parameter_block);
   pb->ws_length = OSAK_EXAMPLE_WS_SIZE;
   pb->calling_aei = &local_address;
   pb->called_aei = &remote_address;
   pb->transport_template = &transport_template;
   pb->functional_units = &fus;
   pb->protocol_versions = &protocol_versions;
   pb->user_data = &send_buffer;

   /* All other fields using default values */
   
   status = spi_connect_req (port, pb);
   if ((status != OSAK_S_NORMAL) && (status != OSAK_S_QUEUED))
   {
      printf("Call to spi_connect_req failed.\n");
      printf("%d",status);
      exit(FAILURE);
   }

   return status;
}
   

/************************************************************************/
/* FUNCTION:  release_req                                               */
/*                                                                      */
/* This routine sets up the parameter block for a call to               */
/* spi_release_req and makes the call.                                  */
/************************************************************************/
unsigned long int release_req (osak_port port, struct osak_parameter_block *pb)
{
   unsigned long int status;
 
   /* initialize parameter block */
   memset ((void *)pb, 0,
          sizeof(struct osak_parameter_block) + OSAK_EXAMPLE_WS_SIZE);
   pb->pb_length = sizeof(struct osak_parameter_block);
   pb->ws_length = OSAK_EXAMPLE_WS_SIZE;
   pb->release_reason = OSAK_C_RLRQ_NORMAL;

   status = spi_release_req (port, pb);
   if ((status != OSAK_S_NORMAL) && (status != OSAK_S_QUEUED))
   {
      printf("Call to spi_release_req failed.\n");
      exit(FAILURE);
   }

   return status;
}
/***************************************************************************/
/* FUNCTION: wait_for_event                                                */
/*                                                                         */
/* This routine waits for an inbound event to occur.  If there is also a   */
/* queued parameter block from a previous call to OSAK then it also waits  */
/* for that parameter block to be returned by OSAK. spi_select is used to  */
/* wait for the inbound event and outbound completion.   spi_get_event is  */
/* used to receive the inbound event and spi_collect_pb is used to get the */
/* parameter block returned by OSAK when the outbound event has completed. */
/*                                                                         */
/* The example of spi_example_resp.c does this differently.  It has two    */
/* routines to do the same job as this one routine:                        */
/* wait_for_outbound_completion and wait_for_inbound.  This routine shows  */
/* how spi_select can be used to combine those two routines.               */
/***************************************************************************/
void wait_for_event (osak_port port, struct osak_parameter_block *pb, 
                     int queued)
{
   struct osak_parameter_block *ret_pb;
   osak_handle_count handle_count;
   osak_handle handle;
   unsigned long int status;
   int read_event = TRUE;
   osak_time select_time = OSAK_EXAMPLE_TIMEOUT;
   
   /* The following declaration is used for the status returned from       */
   /* give_buffer.                                                         */
   unsigned long int buf_status;

   /* Give a buffer to OSAK to get inbound event */
   buf_status = give_buffer (port);
   if (buf_status != OSAK_S_NORMAL)
   {
      printf("Call to spi_give_buffers failed.\n");
      exit(FAILURE);
   }

   /* Loop until not waiting for any more events */
   do 
   {
      /* Set up parameters to call spi_select() */
      handle_count = 1;
      handle.id = (unsigned long int)port;
      handle.request_mask = 0;
      if (read_event)
         handle.request_mask |= OSAK_C_READEVENT;
      if (queued)
         handle.request_mask |= OSAK_C_WRITEEVENT;
      handle.returned_mask = 0;

      status = spi_select (handle_count, &handle, &select_time);
      if (status != OSAK_S_NORMAL)
      {
         printf("Call to spi_select failed.\n");
         exit(FAILURE);
      }
      
      /* See if the queued parameter block has been returned */
      if (queued && (handle.returned_mask & OSAK_C_WRITEEVENT))
      {
         ret_pb = NULL;
         status = spi_collect_pb (port, &ret_pb);
         
         if ((status != OSAK_S_NORMAL) && (status != OSAK_S_NOEVENT))
         {
            printf("Call to spi_collect_pb failed.\n");
            exit(FAILURE);
         }
          
         if (status == OSAK_S_NORMAL && ret_pb != NULL)
         {
            queued = FALSE;

            /* Look at the status block in the PB returned to see if an */
            /* error occurred */
            if (ret_pb->status_block.osak_status_1 != OSAK_S_NORMAL)
            {
               printf("Error in status block of PB returned from collect_pb.\n");
               printf("%d",ret_pb->status_block.osak_status_1);
               exit(FAILURE);
            }
         }   
      }

      /* See if there is an inbound event.  If so call spi_get_event() */
      if (read_event && (handle.returned_mask & OSAK_C_READEVENT))
      {
         do
         {
            /* Initialize parameter block */
            memset ((void *)pb, 0, 
                   sizeof(struct osak_parameter_block) + 
                   OSAK_EXAMPLE_WS_SIZE );
            pb->pb_length = sizeof(struct osak_parameter_block);
            pb->ws_length = OSAK_EXAMPLE_WS_SIZE;
            pb->completion_rtn = NULL;
            pb->completion_param = 0;

            status = spi_get_event (port, pb);
                
            /* If OSAK needs more buffer to decode the event then give */
            /* more buffers.                                           */
            if (status == OSAK_S_NOBUFFERS)
            {
               buf_status = give_buffer (port);
               if (buf_status != OSAK_S_NORMAL)
               {
                  printf("Call to spi_give_buffers failed.\n");
                  exit(FAILURE);
               }
            }
         } while (status == OSAK_S_NOBUFFERS);

         if ((status != OSAK_S_NORMAL) && (status != OSAK_S_NOEVENT))
         {
            printf("Call to spi_get_event failed.\n");
            exit(FAILURE);
         }
         if (status == OSAK_S_NORMAL)
         {
            read_event = FALSE;
         }
      }
   } while (read_event || queued);
}
  


/*************************************************************************/
/* FUNCTION: give_buffer                                                 */
/*                                                                       */
/* This routine is called to pass a buffer to OSAK for OSAK to use to    */
/* receive inbound events.                                               */
/*                                                                       */
/* A list of unused buffers is maintained.  One buffer from this list is */
/* passed to OSAK using spi_give_buffers.  If the list is empty a new    */
/* buffer is allocated.                                                  */
/*************************************************************************/
unsigned long int give_buffer (osak_port port)
{
   unsigned long int status;
   struct osak_buffer *give_buf;

   /* Give a buffer to OSAK */
   if (free_buffers == NULL)
   {
      give_buf = (struct osak_buffer *)malloc (sizeof( struct osak_buffer));
      if (give_buf == NULL )
      {
         printf("Failed to allocate an osak buffer.\n");
         exit(FAILURE);
      }
      give_buf->next = NULL;
      give_buf->buffer_length = OSAK_EXAMPLE_BUFFER_SIZE;
      give_buf->buffer_ptr = (unsigned char *)malloc (OSAK_EXAMPLE_BUFFER_SIZE);
      if (give_buf->buffer_ptr == NULL)
      {
         printf("Failed to allocate buffer.\n");
         exit(FAILURE);
      }
   }
   else 
   {
      give_buf = free_buffers;
      free_buffers = free_buffers -> next;
      give_buf -> next = NULL;
      free_buffers = NULL;
   }
   
   status = spi_give_buffers (port, give_buf);
   return status;
}

/**************************************************************************/
/* FUNCTION: reuse_buffers                                                */
/*                                                                        */ 
/* This routine is called to place buffers returned by OSAK onto the list */
/* of unused buffers.                                                     */
/**************************************************************************/
void reuse_buffers (struct osak_buffer **buf_ptr)
{
   struct osak_buffer *buf, *last_buf;

   buf = *buf_ptr;
   if (buf == NULL)
      return;
   
   last_buf = buf; 
   while (last_buf->next != NULL)
      last_buf = last_buf->next;

   if (free_buffers == NULL)
   {
      free_buffers = buf;
   }
   else
   {
      free_buffers_end->next = buf;
   }
   free_buffers_end = last_buf; 
   *buf_ptr = NULL;

}
/*****************************************************************************/
/* FUNCTION:  alloc_memory                                                   */
/*                                                                           */
/* Memory allocation routine expected by OSAK.                               */
/*****************************************************************************/
void *alloc_memory (unsigned long int size)
{
   return (void *)malloc(size);
}
/*****************************************************************************/
/* FUNCTION:  free_memory                                                    */
/*                                                                           */
/* Memory deallocation routine expected by OSAK                              */
/*****************************************************************************/
unsigned long int free_memory (unsigned long int size, void *ptr)
{
   free(ptr);
   return(0);
}
