/*  C_SQLM_CREATE_STATS_SUB.C
 *
 *  Subroutine to create shiny new stats files
 *
 */
/*
 *      ANSI headers
 */
#include <stdlib>
#include <stdio>
#include <string>
#include <time>


/*
 *      VMS headers
 */
#include <starlet>
#include <lib$routines>
#include <descrip>
#include <libdtdef>
#include <ints.h>
#include <sql_literals>

#define LENGTH(descriptor)      descriptor.dsc$w_length

/*
 *  Our headers
 */
struct z_struct
{
    int8    elm_no;
    int     hit_count;
    int     last_draw_no;
    int     since_last;
    int     curr_seq;
    int     longest_seq;
    double  pct_hits;
    int     max_btwn;
    double  ave_btwn;
};

/*;;;;;
 *  Program constants and global data
 *;;;;;
 */
#define ELM_COUNT 52

/*;;;;;
 *      Function prototypes
 *;;;;;
 */
void calc_final_stats( struct z_struct *d_stats,
                       struct z_struct *m_stats,
                       int l_draw_no);

void close_all_draw( int *sqlcode);

void commit_mega( int *sqlcode);

void delete_all_draw_stats( int *sqlcode);

void delete_all_mega_stats( int *sqlcode);

void fetch_all_draw( int *sqlcode, int64 *draw_dt, int8 *no_1, int8 *no_2, int8 *no_3,
                     int8 *no_4, int8 *no_5, int8 *mega_no);

void initialize_stat_arrays( struct z_struct *d_stats, struct z_struct *m_stats);

void insert_draw_stats( int *sqlcode, int8 *elm_no, int *hit_count, int *last_draw_no,
                        int *since_last, int *curr_seq, int *longest_seq, 
                        double *pct_hits, int *max_btwn, double *ave_btwn);

void insert_mega_stats( int *sqlcode, int8 *elm_no, int *hit_count, int *last_draw_no,
                        int *since_last, int *curr_seq, int *longest_seq, 
                        double *pct_hits, int *max_btwn, double *ave_btwn);

void open_all_draw( int *sqlcode);

void update_dstats( struct z_struct *m_stats, int l_no, int l_draw_no);

void update_mstats( struct z_struct *m_stats, int l_no, int l_draw_no);

/*;;;;;
 *      Functions and subroutines
 *;;;;;
 */
void c_sqlm_create_stats_sub( )
{
    int                     l_x, l_sub, l_draw_no;
    char                    drawing_file_str[255], command_str[255];
    FILE                    *rpt_file;

    int                     sqlcode;
    int64                   draw_dt;
    int8                    no_1, no_2, no_3, no_4, no_5, mega_no;

    //
    //  Arrays to hold stats records
    //
    struct z_struct  d_stats[ ELM_COUNT];
    struct z_struct  m_stats[ ELM_COUNT];

    $DESCRIPTOR( command_str_desc, command_str);



    initialize_stat_arrays( d_stats, m_stats);

    //
    //  empty out any prior stats table rows
    //
    delete_all_mega_stats( &sqlcode);
    commit_mega( &sqlcode);
    delete_all_draw_stats( &sqlcode);
    commit_mega( &sqlcode);


    //
    //  Read through until end of input.
    //  Update each array element as found.
    //
    open_all_draw( &sqlcode);

    l_draw_no = 0;

    while ( sqlcode == SQLCODE_SUCCESS)
    {
        fetch_all_draw( &sqlcode, &draw_dt, &no_1, &no_2, &no_3, &no_4, &no_5, &mega_no);

        if ( sqlcode == SQLCODE_SUCCESS)
        {
            l_draw_no++;
            update_dstats( d_stats, no_1, l_draw_no);
            update_dstats( d_stats, no_2, l_draw_no);
            update_dstats( d_stats, no_3, l_draw_no);
            update_dstats( d_stats, no_4, l_draw_no);
            update_dstats( d_stats, no_5, l_draw_no);

            update_mstats( m_stats, mega_no, l_draw_no);


            if ( (l_draw_no % 100) == 0)
                printf( "Processed %d records\n", l_draw_no);

        }  /* end test for successful read */

        
    }  /* end while l_x */


    printf( "Processed %d records\n", l_draw_no);     // show final count

    close_all_draw( &sqlcode);
    commit_mega( &sqlcode);

    calc_final_stats( d_stats, m_stats, l_draw_no);

    //
    //  Write the new records
    //
    for( l_sub=0; l_sub < ELM_COUNT; l_sub++)
    {
        insert_draw_stats( &sqlcode, &d_stats[ l_sub].elm_no,
                            &d_stats[ l_sub].hit_count,
                            &d_stats[ l_sub].last_draw_no,
                            &d_stats[ l_sub].since_last,
                            &d_stats[ l_sub].curr_seq,
                            &d_stats[ l_sub].longest_seq,
                            &d_stats[ l_sub].pct_hits,
                            &d_stats[ l_sub].max_btwn,
                            &d_stats[ l_sub].ave_btwn);

        if ( sqlcode != SQLCODE_SUCCESS)
        {
            printf( "Error %d writing drawing stat\n", sqlcode);
        }



        insert_mega_stats( &sqlcode, &m_stats[ l_sub].elm_no,
                            &m_stats[ l_sub].hit_count,
                            &m_stats[ l_sub].last_draw_no,
                            &m_stats[ l_sub].since_last,
                            &m_stats[ l_sub].curr_seq,
                            &m_stats[ l_sub].longest_seq,
                            &m_stats[ l_sub].pct_hits,
                            &m_stats[ l_sub].max_btwn,
                            &m_stats[ l_sub].ave_btwn);

        if ( sqlcode != SQLCODE_SUCCESS)
        {
            printf( "Error %d writing drawing stat\n", sqlcode);
        }


    }  /* end for l_sub loop */

    commit_mega( &sqlcode);

}  /* end c_sqlm_create_stats_sub */

/*;;;;;
 *  Subroutine to initialize the arrays which will be written to disk.
 *;;;;;
 */
void initialize_stat_arrays( struct z_struct *d_stats, struct z_struct *m_stats)
{
    int     l_x;

    for (l_x=0; l_x < ELM_COUNT; l_x++)
    {
        memset( &d_stats[ l_x], 0, sizeof( struct z_struct));

        d_stats[ l_x].elm_no        = l_x + 1;
        d_stats[ l_x].pct_hits      = 0.0;
        d_stats[ l_x].ave_btwn      = 0.0;


        memset( &m_stats[ l_x], 0, sizeof( struct z_struct));

        m_stats[ l_x].elm_no        = l_x + 1;
        m_stats[ l_x].pct_hits      = 0.0;
        m_stats[ l_x].ave_btwn      = 0.0;

    }  /* end for l_x loop */

}  /* end initialize_stat_arrays subroutine */

/*;;;;;
 *  Subroutine to update a drawing stats record
 *;;;;;
 */
void update_dstats( struct z_struct *d_stats, int l_no, int l_draw_no)
{
    int     l_since, l_sub;


    l_sub   = l_no - 1;
    l_since = l_draw_no - d_stats[ l_sub].last_draw_no;

    //
    //  Sequence or random hit
    //
    if ( l_since == 1)
    {
        d_stats[ l_sub].curr_seq++;
        if ( d_stats[ l_sub].curr_seq > d_stats[ l_sub].longest_seq)
            d_stats[ l_sub].longest_seq = d_stats[ l_sub].curr_seq;
    }
    else
    {
        d_stats[ l_sub].curr_seq = 0;
        if ( l_since > d_stats[ l_sub].max_btwn)
            d_stats[ l_sub].max_btwn = l_since;

    }  /* end test for sequence or random hit */

    d_stats[ l_sub].hit_count++;
    d_stats[ l_sub].last_draw_no = l_draw_no;
    d_stats[ l_sub].since_last = l_since;

}  /* end update_dstats subroutine */

/*;;;;;
 *  Subroutine to update a drawing stats record
 *;;;;;
 */
void update_mstats( struct z_struct *m_stats, int l_no, int l_draw_no)
{
    int     l_since, l_sub;


    l_sub   = l_no - 1;
    l_since = l_draw_no - m_stats[ l_sub].last_draw_no;

    //
    //  Sequence or random hit
    //
    if ( l_since == 1)
    {
        m_stats[ l_sub].curr_seq++;
        if ( m_stats[ l_sub].curr_seq > m_stats[ l_sub].longest_seq)
            m_stats[ l_sub].longest_seq = m_stats[ l_sub].curr_seq;
    }
    else
    {
        m_stats[ l_sub].curr_seq = 0;
        if ( l_since > m_stats[ l_sub].max_btwn)
            m_stats[ l_sub].max_btwn = l_since;

    }  /* end test for sequence or random hit */

    m_stats[ l_sub].hit_count++;
    m_stats[ l_sub].last_draw_no = l_draw_no;
    m_stats[ l_sub].since_last = l_since;

}  /* end update_mstats subroutine */

/*;;;;;
 *  Subroutine to calculate final drawing stats information
 *;;;;;
 */
void calc_final_stats( struct z_struct *d_stats, struct z_struct *m_stats, int l_draw_no)
{
    int     l_x, l_missed;
    double  d_1, d_2;

    for( l_x=0; l_x < ELM_COUNT; l_x++)
    {
        d_1 = d_stats[ l_x].hit_count;
        d_2 = l_draw_no;

        l_missed = l_draw_no - d_stats[ l_x].hit_count;

        d_stats[ l_x].pct_hits = (double) d_stats[ l_x].hit_count / (double) l_draw_no;

        d_stats[ l_x].ave_btwn = (double) l_missed / 
                                 (double) d_stats[ l_x].hit_count;

        d_stats[ l_x].since_last = l_draw_no - d_stats[ l_x].last_draw_no;


        d_1 = m_stats[ l_x].hit_count;
        d_2 = l_draw_no;

        l_missed = l_draw_no - m_stats[ l_x].hit_count;

        m_stats[ l_x].pct_hits = (double) m_stats[ l_x].hit_count / (double) l_draw_no;

        m_stats[ l_x].ave_btwn = (double) l_missed / 
                                 (double) m_stats[ l_x].hit_count;

        m_stats[ l_x].since_last = l_draw_no - m_stats[ l_x].last_draw_no;


    }  /* end for l_x loop */


}  /* end calc_final_stats subroutine */
