/*
 * Server Application extension to let an embedded perl application commandeer
 * the standard input/output stream and environment variables seen by an
 * executing perl script.  File operations on stdin/stdout result in callbacks
 * to routines in the embedded application.
 *
 * This file is used as input to the xsubpp.pl (extension subroutine 
 * pre-preocessor) perl program, which produces a C source file for compilation
 * and linking with the embedded application. The embedded application must
 * modify the initialization routine passed to perl_parse() to register this
 * extension by including the following statement:
 *
 *      newXS("ServerApp::bootstrap", boot_ServerApp, __FILE__);
 *
 * The application must then activate the extension by forcing a load
 * of the ServerApp.pm module.
 *
 * Author:	David Jones
 * Date:	16-JUL-2000
 * Revised:	19-JUL-2000		Cleanup work.
 * Revised:	29-JAN-2002		Fixes readstdin function
 */
#include "EXTERN.h"
#include "perl.h"
#include "XSUB.h"

#include "serverapp.h"			/* verify prototypes */
/*
 * Define package definition string that define export list and bootstrap
 * for module.
 */
char *serverapp_module_def = 
  "package ServerApp;\n@EXPORT_OK = qw(fetch_hash);bootstrap ServerApp;\n1;\n";
/*
 * The following structure saves user callback routines for intercepting I/O to 
 * standard input and output and getenv calls.  
 */
static struct {
   FILE *out, *in;
   io_callback alt_write, alt_read;
   env_callback alt_getenv;
   CV *cv;
   int bootstrap_run;
} intercept = {
    (FILE *) 0, (FILE *) 0,
    (io_callback) 0, (io_callback) 0,
    (env_callback) 0,
    (CV *) 0,
     0
};
/*
 * Tie_io_handle and tie_env modify perl globals so that operations
 * on them are handled by our callback functions.
 */
static GV *tie_io_handle ( char *stream_name, char *class_name, void *data )
{
    GV *handle;
    SV *new_obj;
    /*
     * Lookup the IO handle by name.
     */
    handle = gv_fetchpv ( stream_name, TRUE, SVt_PVIO );
    if ( !handle ) {
	fprintf(stderr,"Error fetching IO handle for '%s'\n", stream_name );
	return handle;
    }
    /*
     * Make data into a reference variable blessed by indicated package.
     */
    new_obj = sv_newmortal();
    new_obj = sv_setref_pv ( new_obj, class_name, data );
    /*
     * Replace the vtable entry with our own object.
     */
    sv_unmagic ( (SV *) handle, 'q' );
    sv_magic ( (SV *) handle, new_obj, 'q', Nullch, 0);

    return handle;
}

HV *tie_env ( char *hash_name, char *class_name, void *data )
{
    HV *handle;
    SV *new_obj;
    /*
     * Lookup the IO handle by name.
     */
    handle = get_hv ( hash_name, FALSE );
    if ( !handle ) {
	fprintf(stderr,"Error fetching handle for hash '%s'\n", hash_name );
	return handle;
    }
    /*
     * Make data into a reference variable blessed by indicated package.
     */
    new_obj = sv_newmortal();
    new_obj = sv_setref_pv ( new_obj, class_name, data );
    /*
     * Replace the vtable entry with our own object.
     */
    sv_unmagic ( (SV *) handle, 'P' );		/* mark as tied hash */
    sv_magic ( (SV *) handle, new_obj, 'P', Nullch, 0);

    return handle;
}
/*
 * Initialization routine called when module in boostrapped.
 */
void serverapp_initialize ( CV *cv )
{
#ifdef BOOT_NOTIFY
    printf("ServerApp Extension booting has_run: %d...\n", 
	intercept.bootstrap_run);
#endif
    /*
     * Save the cv for use by the hijacking routines.
     */
    if ( intercept.cv ) {
	if ( intercept.cv != cv ) printf ( "  Extension cv changed\n");
    }
    if ( intercept.bootstrap_run ) {
	/* require_pv ( "ServerApp" ); */
	/* return; */
    }
    if ( cv ) intercept.cv = cv;
    if ( !tie_io_handle ( "STDOUT", "ServerApp", "Trouble-now" ) )
	fprintf(stderr,"Error tieing STDOUT to embedded application\n" );
    if ( !tie_io_handle ( "STDIN", "ServerApp", "Trouble-then" ) )
	fprintf(stderr,"Error tieing STDIN to embedded application\n" );
    if ( !tie_env ( "ENV", "ServerApp", "Trouble-always" ) )
	fprintf(stderr,"Error tieing ENV hash to embedded application\n" );

    intercept.bootstrap_run = 1;
}
/*
 * The following routine is called by the embedded application to
 * hijack STDIN, STDOUT, and @env array so they are handled via the
 * supplied callbacks instead of the perl core.
 */
int serverapp_hijack_std_streams ( 
	io_callback alt_write,
	io_callback alt_read,
	env_callback alt_getenv )
{
    static int hijacks = 0;
    intercept.out = stdout;
    intercept.in = stdin;
    intercept.alt_write = alt_write;
    intercept.alt_read = alt_read;
    intercept.alt_getenv = alt_getenv;
    /*
     * Reset the overrides.
     */
    if ( intercept.bootstrap_run && (hijacks < 0) ) {
	CV *cur_cv;
	cur_cv = get_cv ( "ServerApp::FETCH", 0 );
	if ( !cur_cv ) {
	    printf("Restart bootstrap failed to get CV\n");
	    cur_cv = intercept.cv;
	}
	serverapp_initialize ( cur_cv );
    }
    hijacks++;
    return 1;
}

/*
 * End of standard C section.  Lines following the MODULE = 
 * are process by xsubpp into the 'glue' code for the module.
 */
MODULE = ServerApp		PACKAGE = ServerApp

PROTOTYPES: Enable

BOOT:
#
#   Place additional code for bootstrap routine.  Note that boot section
#   is terminated by first blank line.
#
    serverapp_initialize ( cv );

#
##########################################################################
# Overrides for standard I/O functions.
#
# Override routine for STDOUT write function passes data to callback instead
# of writing to file.
#
int 
print( SV *fmt, ...)
#   Set override for print and syswrite methods
    ALIAS:
    ServerApp::PRINT = 1
    ServerApp::WRITE = 2

    PREINIT:
    int i;
    STRLEN len;

    CODE:

    RETVAL = 0;
    for ( i = 1; i < items; i++ ) {
	char *buffer;
	SV *arg;
	arg = ST(i);
	if ( SvROK(arg) && (SvTYPE(arg) == SVt_PV) ) {
	    /* Dereference the pointer */
	    arg = (SV*) SvRV(arg);
	}
	buffer = SvPV(arg,len);
	if ( intercept.alt_write ) {
	    len = intercept.alt_write ( buffer, len );
	    RETVAL += len;
	} else {
	    len = fwrite(buffer, sizeof(char), len, stdout );
	    RETVAL += len;
	}
    }

    OUTPUT:
    RETVAL

#
# Override routine for STDIN read function.
#
int 
readstdin ( SV *f, SV *buf, int bufsize, int offset=0 )
    ALIAS:
    ServerApp::READ = 1
    ServerApp::READLINE = 2
    ServerApp::GETC = 3

    PREINIT:
    int i, alt_len;
    STRLEN orig_len, len, alloc_size, trailing;
    char *buffer, *tbuffer, *rptr;
    SV *arg;

    CODE:
    /*
     * Locate actual buffer object if this is a reference, then get
     * pointer to the actual data and it's length.
     */
    /* printf ( "/readin(%d)/ f is %x, items=%d", ix, f, items ); */
    if ( SvROK(buf) && (SvTYPE(buf) == SVt_PV) ) {
	buf = (SV *) SvRV(buf);
    }
    buffer = SvPV(buf,orig_len);
    /*
     * Allocate a buffer of the target size and set read position based
     * on offset.  trailing is number of bytes at end of original buffer
     * to append to end of read buffer.
     */
    alloc_size = bufsize + offset;
    trailing = 0;
    if ( alloc_size < orig_len ) {
	trailing = orig_len - alloc_size;
	alloc_size = orig_len;
    }
    tbuffer = malloc ( (alloc_size > 0) ? alloc_size : 1 );
    if ( !tbuffer ) {
    }
    rptr = &tbuffer[offset];
    /*
     * Read data into buffer.
     */
    if ( intercept.alt_read ) {
	/* Convert signed result */
	alt_len = intercept.alt_read ( rptr, bufsize );
	len = (alt_len > 0) ? alt_len : 0;
    } else {
	/* No intercept set, fallback to standard input stream */
	len = fread ( rptr, sizeof(char), bufsize, stdin );
    }
    /*
     * Replace old string with new buffer.
     */
    if ( offset > 0 ) memcpy ( tbuffer, buffer, offset );
    if ( trailing > 0 ) memcpy (&rptr[len], &buffer[offset+bufsize], trailing);
    if ( len > 0 ) {
	/* Save a copy, swap pointers */
	sv_usepvn ( buf, tbuffer, len+offset+trailing );
	RETVAL = len;
    } else {
	free ( tbuffer );
	sv_setpvn ( buf, "", 0 );	/* make a null string */
	RETVAL = 0;
    }

    RETVAL = len;

    OUTPUT:
    RETVAL

#
#  Override for BINMODE and CLOSE operations.
#
void 
binmode ( SV *hndle )
    ALIAS:
    ServerApp::BINMODE = 1
    ServerApp::CLOSE = 2

    CODE:
    /* printf("binmode called, ix=%d, item = %d, sv: %x\n", ix, items,hndle); */
    if ( ix == 1 ) {
	/* Nothing really to do. */
    } else if ( ix == 2 ) {	/* CLOSE */
	/* Can't really close stream, maybe we should flush */
    }

##########################################################################
#
# Following section is for overidding ENV hash variable.
#

SV *
fetch_hash ( SV *hz, SV *key )
    ALIAS:
    ServerApp::FETCH = 1

    PREINIT:
    int i;
    char *keystr, *valstr;
    STRLEN keylen;

    CODE:
    /*
     * Extract string pointer from key argument.
     */
    keystr = SvPV(key,keylen);
    /*
     * Lookup environment variable using intercept routine if set,
     * fallback to standard perl runtime if alternate lookup fails.
     */
    if ( intercept.alt_getenv ) {
	valstr = intercept.alt_getenv ( keystr, 0 );
    } else valstr = (char *) 0;
    if ( !valstr ) valstr = Perl_my_getenv ( keystr, 0 );
    /*
     * Convert result into a scalar to be returned;
     */
    RETVAL = (valstr) ? (newSVpv(valstr, strlen(valstr))) : Nullsv;

    OUTPUT:
    RETVAL
