/*
 * Prototypes for file cache support routines.
 */
#include "data_cache.h"
#ifndef VMS
#include <time.h>
#endif
/*
 * Every entry saved in cache has as it's last record a tfc_info structure
 * that stores the header information of the corresponding file ('header' is
 * last record so that size info may be accumulated.
 */
struct tfc_finfo {			/* File information */
#ifdef VMS
    long cache_expiration[2];		/* VMS time of cache expriation */
    long cache_retention[2];		/* per-entry retention time VMS delta*/
#else
    time_t cache_expiration;		/* Time entry considered stale */
    time_t cache_retention;
#endif
    int hits;				/* Number of successful accesses */
    int size;				/* data-bytes, -1 if put overflowed */
    unsigned uic;			/* file owner */
    unsigned int cdate;			/* file's create date (UTC) */
    unsigned int mdate;			/* files last modification date */
    unsigned int edate;			/* file expiration date */
};
struct tfc_context {
    int mode;				/* context status: 1-read, 2-write */
    int data_size;			/* Number of put data bytes (mode 2) */
    struct tcache_context ctx;		/* data_cache context */
    int hdr_size;			/* size of header record */
    void *hdr_rec;			/* Address of last record in cache */
    struct tfc_finfo hdr;		/* Working copy of header record */
};
typedef struct tfc_context *tfc_ctxptr;
/*
 * The init routine must be called in single-threaded mode.  Refresh_time
 * is seconds item can stay in cache before modification check (may be zero).
 * total_size is size in bytes (should be > 200,000) of cache.
 *
 * Note that these routines do not set up exit handlers to rundown contexts
 * on thread exit, so caller of tfc_check_cache must call tfc_rundown_context
 * before context argument goes out of scope.
 */
int tfc_init ( int refresh_time, int total_size, int item_size_limit );
int tfc_set_tf_options ( int flags, int *enabled );
/*
 * Lookup cache item named by type and ident.  Open is done via tf_open()
 * (file_access.h).  If type is 0 (binary), open mode is "rb"; if type is 1
 * (text), mode is  "rd".  Char buffer recieves error message from tf_open.
 */
int tfc_cached_open ( int type, char *ident, void **fptr, char *buffer,
	tfc_ctxptr tfc );
   /* return values: 0 - cache disabled or other failure, ignore.
    *		     1 - Valid item found in cache, fptr set to NULL.
    *		     2 - New item created or existing time stale, fptr opened
    *			 for read access on ident.
    */
/*
 * Context rundown will automatically write the tfc as the last record
 * if mode=2 and deaccess entry.  If abort_flag true, entry is deleted.
 */
int tfc_rundown_context ( struct tfc_context *tfc, int abort_flag );
/*
 * tfc_get can only be called if tfc_cached_open returned 1 (valid).
 * tfc_put can only be called if tfc_cached_open return 2 (created), if
 * the data limit is exceeded during put, the entry is cleared and replaced
 * with a -1 sized entry (overflow) and then deaccessed. (mode changed to 0).
 */
int tfc_get ( tfc_ctxptr tfc, void **next_rec, int *length );
    /* return values: -1 - tfc context invalid.
     *		       0 - no more records in cache (excluding header record).
     *		       1 - Normal, successful completion.
     */

int tfc_put ( tfc_ctxptr tfc, void *rec, int length );
/*
 * Expire_all function reset saved expiration time in all entries so they will
 * be re-verified on next access.
 */
int tfc_expire_all();
int tfc_update_expiration ( tfc_ctxptr tfc, int retention );
int tfc_verify_expiration ( tfc_ctxptr tfc, unsigned int edate, int reset_if );
/*
 * Format statistics counters as text (for display by management interface)
 */
void tfc_display_counters ( char *buffer, int bufsize, int *length, int zero );
int tfc_display_entries ( tcache_ctxptr ctx, char *buffer, int bufsize,
	int *length, int *count );
/*
 * Following routines maintain bitmap for recording which TCP ports have
 * caching inhibited.   Bit numbers range is 0 to 65535 (16 bits).
 * Initially all bits are clear.
 * Return value for test: 0-port number not marked, 1-port number marked.
 */
int tfc_mark_port_number ( int port_num );
int tfc_test_port_number ( int port_num );
