/*
 * Copyright (c) 1992 The Regents of the University of California.
 * All rights reserved.
 * 
 * This new dither is modified from the original 2x2.c software. It is
 * intended to allow the user to see the images as they are in the MPEG
 * stream, without any dithering to distort them. -PGE 7/26/94
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without written agreement is
 * hereby granted, provided that the above copyright notice and the following
 * two paragraphs appear in all copies of this software.
 * 
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
 * CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

#include "video.h"
#include "dither.h"
#include "proto.h"

/* Array containing actual pixel values for each possible 1x1 dither color. */

static unsigned char *dith_a;

/* Arrays mapping lum, cr, and cb values to portions of dither pattern code. 
   The addtion of one value from each array yields a valid dither pattern
   code.
*/

static int lval_a [256];
static int rval_a [256];
static int bval_a [256];


/*
 *--------------------------------------------------------------
 *
 *  Init1x1Dither--
 *
 *	Initializes structures used for 1x1 dithering.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

void
Init1x1Dither()
{  
  int numcodes;
  int l_dith, cr_dith, cb_dith;
  int i;

  numcodes =  LUM_RANGE * CR_RANGE * CB_RANGE;

  dith_a = (unsigned char *) malloc(numcodes);

  for (i=0; i<numcodes; i++) {
    l_dith = i % LUM_RANGE;
    cr_dith = (i/LUM_RANGE) % CR_RANGE;
    cb_dith = (i/(CR_RANGE*LUM_RANGE)) % CB_RANGE;

    dith_a [i] = (l_dith * CR_RANGE * CB_RANGE) +
                 (cr_dith * CB_RANGE) +
                 cb_dith;
  }

  for (i=0; i<256; i++) {
    lval_a[i] = (i * LUM_RANGE)/256;
    rval_a[i] = (i * CR_RANGE)/256;
    bval_a[i] = (i * CB_RANGE)/256;

    bval_a[i] *= CR_RANGE * LUM_RANGE;
    rval_a[i] *= LUM_RANGE;
  }

}


/*
 *--------------------------------------------------------------
 *
 *  PostInit1x1Dither--
 *
 *	Remaps color numbers in dither patterns to actual pixel
 *      values allocated by the X server.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *--------------------------------------------------------------
 */

void
PostInit1x1Dither() 
{
  int i;

  for (i=0; i < (LUM_RANGE * CR_RANGE * CB_RANGE); i++) {
    dith_a [i] = pixel [dith_a [i]];
  }
}


/*
 *--------------------------------------------------------------
 *
 * Onex1DitherImage --
 *
 *	Dithers lum, cr, and cb channels togethor using predefined
 *      and computed 1x1 dither patterns. Each possible combination of
 *      lum, cr, and cb values combines to point to a particular dither
 *      color which is used to represent the pixel. This assumes that 
 *      the display plane is the same size as the lumianance plane. 
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *--------------------------------------------------------------
 */

void 
Onex1DitherImage(lum, cr, cb, out, h, w)
    unsigned char *lum;
    unsigned char *cr;
    unsigned char *cb;
    unsigned char *out;
    int w, h;
{
  int i, j;
  unsigned char *o1, *o2;
  unsigned char *l1, *l2, *base;
  unsigned char B, R;
  int b_val, r_val, l_val;

  o1 = out;
  o2 = out+w;
  l1 = lum;
  l2 = lum+w;

  for (i=0; i<h; i+=2) {
    for(j=0; j<w; j+= 4) {
 
      B = cb[0];
      b_val = bval_a[B];
      R = cr[0];
      r_val = rval_a[R];
      base = dith_a + b_val + r_val;

      l_val = lval_a[l1[0]];
      o1[0] = *(base + l_val);

      l_val = lval_a[l1[1]];
      o1[1] = *(base + l_val);

      l_val = lval_a[l2[0]];
      o2[0] = *(base + l_val);

      l_val = lval_a[l2[1]];
      o2[1] = *(base + l_val);

      B = cb[1];
      b_val = bval_a[B];
      R = cr[1];
      r_val = rval_a[R];
      base = dith_a + b_val + r_val;

      l_val = lval_a[l1[2]];
      o1[2] = *(base + l_val);

      l_val = lval_a[l1[3]];
      o1[3] = *(base + l_val);

      l_val = lval_a[l2[2]];
      o2[2] = *(base + l_val);

      l_val = lval_a[l2[3]];
      o2[3] = *(base + l_val);

      o1 += 4;
      o2 += 4;
      l1 += 4;
      l2 += 4;
      cb += 2;
      cr += 2;
    }    

    l1 += w;
    l2 += w;
    o1 += w;
    o2 += w;
  }
}
