/* --------------------------------------------------------------------------
 * GRAB.C - a VMS search utility
 *
 * Copyright (C) 2002 Graham Burley
 *
 * This software is covered by the GNU General Public License, see the
 * file LICENSE.TXT for details.
 *
 * Revision History:
 *	0.22	Graham Burley		05-Feb-2003
 *		Fix very silly ACCVIO in highlight due to incorrect
 *		size in s_vs struct (s/be MAX_BUFLEN not MAX_STRLEN)
 *
 *	0.23	Graham Burley		26-Feb-2003
 *		Add support to save output records in symbols (/SYMBOL)
 *
 */
#define VERSION "0.23"
#ifdef VAXC
#module GRAB VERSION
#else
#pragma module GRAB VERSION
#endif

#include <descrip.h>
#ifndef VAXC
#include <cli$routines.h>
#endif
#include <climsgdef.h>
#include <lib$routines.h>
#include <libdef.h>
#include <libclidef.h>
#include <lnmdef.h>
#include <otsdef.h>
#include <ots$routines.h>
#include <rms.h>
#include <rmsdef.h>
#include <psldef.h>
#include <ssdef.h>
#include <str$routines.h>
#include <strdef.h>
#include <starlet.h>

#define TRUE 1
#define FALSE 0

#define Q_ABSENT 0
#define Q_PRESENT 1
#define Q_MORE 2

#define M_OR 1
#define M_AND 2

#define S_NONE 1
#define S_READ 2
#define S_WRITE 3

#define MAX_BUFLEN 32767
#define MAX_STR 16		/* maximum search strings */
#define MAX_CUT 2
#define MAX_STRLEN 255
#define MAX_SYMLEN 1024
#define MAX_HIL 32		/* maximum highlights per record */

/* CLD */

globalref int GRAB_CLD;

/* Messages */

globalref int MSG__BADHEADER;
globalref int MSG__BADTRAILER;
globalref int MSG__FAILED;
globalref int MSG__FIND;
globalref int MSG__GETRFA;
globalref int MSG__LIMITED;
globalref int MSG__LOG;
globalref int MSG__INVCONTEXT;
globalref int MSG__INVALLOC;
globalref int MSG__INVEXTEN;
globalref int MSG__INVLIMIT;
globalref int MSG__NEWCONTEXT;
globalref int MSG__NOFILES;
globalref int MSG__NOMATCHES;
globalref int MSG__NORECORDS;
globalref int MSG__NORMAL;
globalref int MSG__OPEN;
globalref int MSG__RECLIMIT;
globalref int MSG__RMSRET;
globalref int MSG__RTLRET;
globalref int MSG__SECLIMIT;
globalref int MSG__STATISTICS;
globalref int MSG__SYMTABFULL;
globalref int MSG__SYSRET;
globalref int MSG__TOOMANYCUT;
globalref int MSG__TOOMANYSTR;
globalref int MSG__TRUNCATED;
globalref int MSG__USECONTEXT;
globalref int MSG__VERSION;


struct d_str {
	unsigned short len;
	char *str;
};

struct s_dsc {
	struct dsc$descriptor dsc;
	char buff[MAX_STRLEN];
};

struct s_vs {
	unsigned short curlen;
	char buff[MAX_BUFLEN];
};

struct quals {
	int		alloc;
	struct s_dsc	alloc_string;
	unsigned long	alloc_value;
	int		context;
	struct	s_dsc	context_string;
	int		cut;
	int		cut_max;
	struct	s_dsc	cut_string[MAX_CUT];
	int		exact;
	int		exten;
	struct s_dsc	exten_string;
	unsigned short	exten_value;
	int		filter;
	int		header;
	struct	s_dsc	header_fao;
	struct	s_dsc	header_string;
	int		highlight;
	struct s_dsc	high_string;
	struct dsc$descriptor high_style;
	int		limit;
	struct	s_dsc	limit_string;
	unsigned long	limit_value;
	int		limit_reached;
	int		log;
	int		match;
	int		match_type;
	struct	s_dsc	match_string;
	int		new_context;
	int		output;
	struct	s_dsc	output_string;
	struct	FAB	output_fab;
	struct	RAB	output_rab;
	int		redirect_output;
	int		save;
	struct	s_dsc	save_string;
	int		save_files;
	int		save_status;
	int		save_summary;
	int		sep;
	struct	s_dsc	sep_string;
	int		share;
	int		share_type;
	struct	s_dsc	share_string;
	int		statistics;
	int		string_max;
	struct	s_dsc	string[MAX_STR];
	int		symbol;
	struct	s_dsc	symbol_string;
	unsigned long	symbol_count;	/* shouldn't be here */
	int		trailer;
	struct	s_dsc	trailer_fao;
	struct	s_dsc	trailer_string;
	int		wild;
};

struct stats {
	unsigned long	files;
	unsigned long	records;
	unsigned long	bytes;
	unsigned long	output;
	unsigned long	sections;
	unsigned long	match[MAX_STR];
};

struct file {
	int num;
	struct dsc$descriptor name;
	struct FAB fab;
	struct RAB rab;
	struct NAM nam;
	struct dsc$descriptor rec;
	char rec_buff[MAX_BUFLEN];
	unsigned short saved_rfa[3];
	unsigned short last_rfa[3];
	struct stats stat;
};

struct context {
	unsigned short length;
	char dvi[16];
	unsigned short did[3];
	unsigned short fid[3];
	unsigned short rfa[3];
	long tim[2];
	unsigned long check;
	unsigned long trail;
};

/* Highlights */

#define HL_BOLD    0
#define HL_REVERSE 1
#define HL_RED     2
#define HL_GREEN   3
#define HL_BLUE    4
struct d_str highlight_style[5] = { 4 , "\x1b[1m"
				  , 4 , "\x1b[7m"
				  , 5 , "\x1b[31m"
				  , 5 , "\x1b[32m"
				  , 5 , "\x1b[34m" };

/* functions */

int  get_command     ();
void get_params      (struct quals *);
void eval_params     (struct quals *, struct stats *);
void find_file       (struct quals *, struct stats *);
int  open_file       (struct file *, struct quals*);
void close_file      (struct file *, struct quals *, struct stats *);
void save_context    (struct file *, struct quals *);
int  load_context    (struct file *, struct quals *);
void search_file     (struct file *, struct quals *);
void rewind_file     (struct file *, struct quals *);
int  get_qualifier   (int, char *, struct s_dsc *);
int  redirect_output (struct quals *);
void put_output      (struct quals *, struct dsc$descriptor *);
void close_output    (struct quals *);
void save_summary    (struct quals *, struct stats *, struct file *);
void zero_stats      (struct stats *);
void add_stats       (struct stats *, struct stats *);
void show_stats      (struct quals *, struct stats *);
void make_header     (struct file *, struct quals *);
void show_version    ();
void save_status     (struct quals *, int);
void convert_tu_int  (struct dsc$descriptor *, void *, int size, int m);
void highlight       (struct quals *, struct dsc$descriptor *);
void put_symbol      (struct quals *, struct dsc$descriptor *);
void put_sym_count   (struct quals *);


/* --------------------------------------------------------------------------
 * main
 *
 */
int main()
{
	int s;
	struct quals q;
	struct stats x;

	s = get_command();
	if (s != CLI$_NORMAL) {
		return(s);
	}

	lib$init_timer();

	zero_stats(&x);
	get_params(&q);
	eval_params(&q, &x);
	q.redirect_output = redirect_output(&q);

	find_file(&q, &x);

	if (q.save_summary) {
		save_summary(&q, &x, 0);
	}
	if (q.statistics) {
		show_stats(&q, &x);
	}

	s = (int) &MSG__NORMAL | 0x10000000;
	if (x.files == 0) {
		s = (int) &MSG__NOFILES;
	} else {
		if (x.records == 0) {
			s = (int) &MSG__NORECORDS;
		} else {
			if (x.output == 0) {
				s = (int) &MSG__NOMATCHES;
			} else {
				if ((q.limit) && (q.limit_reached)) {
					s = (int) &MSG__LIMITED | 0x10000000;
				}
			}
		}
	}
	lib$signal(s);

	if (q.save_status) {
		save_status(&q, s);
	}

	return(s | 0x10000000);
}
/* --------------------------------------------------------------------------
 * show_version
 *
 */
void show_version ()
{
	$DESCRIPTOR(v, VERSION);

	lib$signal(&MSG__VERSION, 1, &v);
	sys$exit(SS$_NORMAL);
}
/* --------------------------------------------------------------------------
 * init_s_dsc
 *
 */
void init_s_dsc(struct s_dsc *d)
{
	d->dsc.dsc$b_class = DSC$K_CLASS_S;
	d->dsc.dsc$w_length = MAX_STRLEN;
	d->dsc.dsc$a_pointer = &d->buff[0];
}
/* --------------------------------------------------------------------------
 * save_status
 *
 */
void save_status(struct quals *q, int m)
{
	int s;
	struct dsc$descriptor m_text;
	char m_buff[MAX_STRLEN];
	$DESCRIPTOR(d, "GRAB_STATUS");

	m_text.dsc$b_class = DSC$K_CLASS_S;
	m_text.dsc$w_length = MAX_STRLEN;
	m_text.dsc$a_pointer = &m_buff[0];

	s = sys$getmsg(m, &m_text.dsc$w_length, &m_text, 2, 0);
	if (s != SS$_NORMAL) {
		lib$stop(&MSG__SYSRET, 0, s);
	}

	s = lib$set_symbol(&d, &m_text, 0);
	if (s != SS$_NORMAL) {
		lib$stop(&MSG__RTLRET, 0, s);
	}
}
/* --------------------------------------------------------------------------
 * find_file
 *
 */
void find_file(struct quals *q, struct stats *x)
{
	int i = 0;
	int r;
	int s;
	struct s_dsc fs;
	unsigned long ctx = 0;
	struct file f;
	struct dsc$descriptor rel_fs;
	$DESCRIPTOR(wild,"*%");
	$DESCRIPTOR(def_fs,"SYS$DISK:[].LIS");

	f.num = 0;
	f.name.dsc$b_class = DSC$K_CLASS_D;
	f.name.dsc$w_length = 0;
	f.name.dsc$a_pointer = 0;

	rel_fs.dsc$b_class = DSC$K_CLASS_D;
	rel_fs.dsc$w_length = 0;
	rel_fs.dsc$a_pointer = 0;
	str$copy_dx(&rel_fs, &def_fs);

	while (r = get_qualifier(8, "FILESPEC", &fs)) {
		if (i == 0) {
			while (q->header) {
				if (r == Q_MORE) {
					break;
				}
				if (str$find_first_in_set(&fs, &wild) != 0) {
					break;
				}
				q->header = FALSE;
			}
		}
		i++;
		while ((s = lib$find_file(&fs, &f.name, &ctx, 0, &rel_fs, 0, 0))
				!= RMS$_NMF) {
			switch (s) {
			  case RMS$_NORMAL :
				f.num++;
				if (open_file(&f, q)) {
					make_header(&f, q);
					search_file(&f, q);
					close_file(&f, q, x);
					if (q->save_files) {
						save_summary(q, &f.stat, &f);
					}
				}
				break;
			  default :
				lib$signal(&MSG__FIND, 1, &fs.dsc, s);
			}
			str$copy_dx(&rel_fs, &fs);
		}
		s = lib$find_file_end(&ctx);
		if (s != SS$_NORMAL) {
			lib$stop(&MSG__RTLRET, 0, s);
		}
	}
}

/* --------------------------------------------------------------------------
 * make_header
 *
 */
void make_header(struct file *f, struct quals *q)
{
	int s;

	if (q->header) {
		q->header_string.dsc.dsc$b_class = DSC$K_CLASS_S;
		q->header_string.dsc.dsc$w_length = MAX_STRLEN;
		q->header_string.dsc.dsc$a_pointer = &q->header_string.buff[0];
		s = sys$fao(&q->header_fao.dsc
			, &q->header_string.dsc.dsc$w_length
			, &q->header_string.dsc
			, &f->name
			, &q->header_string.dsc.dsc$w_length);
		if (s != SS$_NORMAL) {
			lib$signal(&MSG__BADHEADER, 0, s);
			q->header = FALSE;
		}
	}

	if (q->trailer) {
		q->trailer_string.dsc.dsc$b_class = DSC$K_CLASS_S;
		q->trailer_string.dsc.dsc$w_length = MAX_STRLEN;
		q->trailer_string.dsc.dsc$a_pointer = &q->trailer_string.buff[0];
		s = sys$fao(&q->trailer_fao.dsc
			, &q->trailer_string.dsc.dsc$w_length
			, &q->trailer_string.dsc
			, &f->name
			, &q->trailer_string.dsc.dsc$w_length);
		if (s != SS$_NORMAL) {
			lib$signal(&MSG__BADTRAILER, 0, s);
			q->trailer = FALSE;
		}
	}
}

/* --------------------------------------------------------------------------
 * get_command
 *
 */
int get_command()
{
	int s;
	struct dsc$descriptor d;
	$DESCRIPTOR(v, "GRAB ");

	d.dsc$b_class = DSC$K_CLASS_D;
	d.dsc$w_length = 0;
	d.dsc$a_pointer = 0;

	s = lib$get_foreign(&d, 0, 0, 0);
	if (s != SS$_NORMAL) {
		lib$stop(&MSG__RTLRET, 0, s);
	}

	s = str$prefix(&d, &v);

	s = cli$dcl_parse(&d, &GRAB_CLD, lib$get_input, lib$get_input, 0);

	return(s);
}

/* --------------------------------------------------------------------------
 * get_params
 *	get values for all qualifiers
 *
 */
void get_params(struct quals *q)
{
	int r;

	if ( get_qualifier( 7, "VERSION", 0) ) {
		show_version();
	}

	q->alloc       = get_qualifier(10, "ALLOCATION"   , &q->alloc_string);
	q->context     = get_qualifier( 7, "CONTEXT"      , &q->context_string);
	q->exact       = get_qualifier( 5, "EXACT"        , 0);
	q->exten       = get_qualifier( 9, "EXTENSION"    , &q->exten_string);
	q->filter      = get_qualifier( 6, "FILTER"       , 0);
	q->header      = get_qualifier( 6, "HEADER"       , &q->header_fao);
	q->highlight   = get_qualifier( 9, "HIGHLIGHT"    , &q->high_string);
	q->log         = get_qualifier( 3, "LOG"          , 0);
	q->limit       = get_qualifier( 5, "LIMIT"        , &q->limit_string);
	q->match       = get_qualifier( 5, "MATCH"        , &q->match_string);
	q->new_context = get_qualifier(11, "NEW_CONTEXT"  , 0);
	q->output      = get_qualifier( 6, "OUTPUT"       , &q->output_string);
	q->sep         = get_qualifier( 9, "SEPARATOR"    , &q->sep_string);
	q->share       = get_qualifier( 5, "SHARE"        , &q->share_string);
	q->statistics  = get_qualifier(10, "STATISTICS"   , 0);
	q->symbol      = get_qualifier( 6, "SYMBOL"       , &q->symbol_string);
	q->trailer     = get_qualifier( 7, "TRAILER"      , &q->trailer_fao);
	q->wild        = get_qualifier( 8, "WILDCARD"     , 0);

	q->string_max = 0;
	while ( r = get_qualifier( 7, "STRINGS", &q->string[q->string_max]) ) {
		q->string_max++;
		if (q->string_max >= MAX_STR) {
			if (r == Q_MORE) {
				lib$signal(&MSG__TOOMANYSTR);
			}
			break;
		}
	}

	q->cut_max = 0;
	while ( r = get_qualifier( 3, "CUT", &q->cut_string[q->cut_max]) ) {
		q->cut_max++;
		if (q->cut_max >= MAX_CUT) {
			if (r == Q_MORE) {
				lib$signal(&MSG__TOOMANYCUT);
			}
			break;
		}
	}

	while ( get_qualifier( 4, "SAVE", &q->save_string) ) {
		switch (q->save_string.buff[1]) {
		  case 'I' :
			q->save_files = TRUE;
			break;
		  case 'T' :
			q->save_status = TRUE;
			break;
		  case 'U' :
			q->save_summary = TRUE;
			break;
		}
	}
}

/* --------------------------------------------------------------------------
 * eval_params
 *	interpret qualifiers
 *
 */
void eval_params(struct quals *q, struct stats *x)
{
	int s;
	int i;

	if (q->save_status) {
		save_status(q, (int) &MSG__FAILED);
	}
	if (q->save_files) {
		q->save_summary = TRUE;
	}
	if (q->save_summary) {
		save_summary(q, x, 0);
	}

	if (q->match) {
		switch (q->match_string.buff[0]) {
		  case 'A' :
			q->match_type = M_AND;
			break;
		  default :
		  case 'O' :
			q->match_type = M_OR;
		}
	}

	if (q->share) {
		switch (q->share_string.buff[0]) {
		  case 'N' :
			q->share_type = S_NONE;
			break;
		  case 'R' :
			q->share_type = S_READ;
			break;
		  default :
		  case 'W' :
			q->share_type = S_WRITE;
		}
	}

	if (q->highlight) {
		switch (q->high_string.buff[2]) {
		  case 'D' :
			i = HL_RED;
			break;
		  case 'E' :
			i = HL_GREEN;
			break;
		  case 'U' :
			i = HL_BLUE;
			break;
		  case 'V' :
			i = HL_REVERSE;
			break;
		  default :
		  case 'L' :
			i = HL_BOLD;
		}
		q->high_style.dsc$b_class = DSC$K_CLASS_S;
		q->high_style.dsc$w_length = highlight_style[i].len;
		q->high_style.dsc$a_pointer = &highlight_style[i].str[0];

	    /* disable highlighting if /NOOUT or /OUT=blah */
		if (!q->output) {
			q->highlight = FALSE;
		}
		if (q->output_string.dsc.dsc$w_length != 0) {
			q->highlight = FALSE;
		}
	}

	if (q->cut_max > 0) {
		q->cut = TRUE;
		if (q->cut_max == 1) {
			q->cut_string[1].dsc = q->cut_string[0].dsc;
		}
	}

	if ( !q->exact ) {
		for (i=0; i<q->string_max; i++) {
			str$upcase(&q->string[i].dsc, &q->string[i].dsc);
		}
		if (q->cut) {
			for (i=0; i<MAX_CUT; i++) {
				str$upcase(&q->cut_string[i].dsc
						, &q->cut_string[i].dsc);
			}
		}
	}

	if (q->limit) {
		convert_tu_int( &q->limit_string.dsc, &q->limit_value, 4
			, (int) &MSG__INVLIMIT );
	}

	if (q->alloc) {
		convert_tu_int( &q->alloc_string.dsc, &q->alloc_value, 4
			, (int) &MSG__INVALLOC );
	}

	if (q->exten) {
		convert_tu_int( &q->exten_string.dsc, &q->exten_value, 2
			, (int) &MSG__INVEXTEN );
	}

	if (q->symbol) {
		q->output = TRUE;	/* prefer /SYMBOL over /NOOUTPUT */
		q->highlight = FALSE;	/* formatting just adds confusion ? */
		q->filter = FALSE;;
		q->header = FALSE;
		q->trailer = FALSE;
		q->sep = FALSE;
		q->symbol_count = 0;	/* we might never set this otherwise */
		put_sym_count(q);
	}

}

/* --------------------------------------------------------------------------
 * convert_tu_int
 *
 */
void convert_tu_int(struct dsc$descriptor *d, void *v, int size, int m)
{
	int s;
	int ok = FALSE;
	unsigned short *p2;
	unsigned long *p4;

	s = ots$cvt_tu_l( d, v, size, 0);
	switch (s) {
	  case SS$_NORMAL :
		ok = TRUE;
		break;
	  case OTS$_INPCONERR :
		break;
	  default :
		lib$stop(&MSG__RTLRET, 0, s);
	}

	switch (size) {
	  case 2 :
		p2 = v;
		if (*p2 == 0) {
			ok = FALSE;
		}
		break;
	  case 4 :
		p4 = v;
		if (*p4 == 0) {
			ok = FALSE;
		}
		break;
	}

	if (!ok)  {
		lib$signal(m, 1, d);
	}
}
/* --------------------------------------------------------------------------
 * get_qualifier
 *	is qualifier <label> present?
 *	if so and value address is defined we get the qualifier value
 *
 *	NB: if expect a value and no value return false even if present
 */
int get_qualifier(int len, char *l, struct s_dsc *v)
{
	int r = Q_ABSENT;
	int s;
	struct dsc$descriptor_s d;

	d.dsc$b_class = DSC$K_CLASS_S;
	d.dsc$a_pointer = l;
	d.dsc$w_length = len;

	s = cli$present(&d);
	switch(s) {
	  case CLI$_ABSENT :
	  case CLI$_NEGATED :
		break;
	  case CLI$_PRESENT :
	  case CLI$_DEFAULTED :
		r = Q_PRESENT;
		break;
	  default :
		lib$stop(&MSG__RTLRET, 0, s);
	}

	if (v != 0) {
		v->dsc.dsc$b_class = DSC$K_CLASS_S;
		if (r) {
			v->dsc.dsc$w_length = MAX_STRLEN;
			v->dsc.dsc$a_pointer = &v->buff[0];
			s = cli$get_value(&d, &v->dsc, &v->dsc.dsc$w_length);
			switch (s) {
			  case CLI$_ABSENT :
			  case CLI$_NEGATED :
				v->dsc.dsc$w_length = 0;
				r = Q_ABSENT;
				break;
			  case CLI$_COMMA :
				r = Q_MORE;
				break;
			  case SS$_NORMAL :
				r = Q_PRESENT;
				break;
			  default :
				lib$stop(&MSG__RTLRET, 0, s);
			}
		} else {
			v->dsc.dsc$w_length = 0;
		}
	}

	return(r);
}

/* --------------------------------------------------------------------------
 * redirect_output
 *
 */
int redirect_output(struct quals *q)
{
	int s;

	if ( (!q->output) || (q->output_string.dsc.dsc$w_length == 0) ) {
		return(FALSE);
	}

	q->output_fab = cc$rms_fab;
	q->output_fab.fab$l_fna = q->output_string.dsc.dsc$a_pointer;
	q->output_fab.fab$b_fns = q->output_string.dsc.dsc$w_length;
	q->output_fab.fab$b_fac = FAB$M_PUT;
	q->output_fab.fab$b_shr = FAB$M_NIL;
	q->output_fab.fab$b_org = FAB$C_SEQ;
	q->output_fab.fab$b_rfm = FAB$C_VAR;
	q->output_fab.fab$b_rat = FAB$M_CR;

	if (q->alloc) {
		q->output_fab.fab$l_alq = q->alloc_value;
	}
	if (q->exten) {
		q->output_fab.fab$w_deq = q->exten_value;
	}

	s = sys$create(&q->output_fab);
	if (s != RMS$_NORMAL) {
		lib$stop(&MSG__RMSRET, 0, s);
	}

	q->output_rab = cc$rms_rab;
	q->output_rab.rab$l_fab = &q->output_fab;
	q->output_rab.rab$b_rac = RAB$C_SEQ;
	q->output_rab.rab$l_rop = RAB$M_WBH;

	s = sys$connect(&q->output_rab);
	if (s != RMS$_NORMAL) {
		lib$stop(&MSG__RMSRET, 0, s);
	}

	return(TRUE);
}

/* --------------------------------------------------------------------------
 * put_output
 *
 */
void put_output(struct quals *q, struct dsc$descriptor *d)
{
	int s;
	$DESCRIPTOR(d_fao,"!AF");

	if (q->output) {
		if (q->filter) {
			s = sys$fao(&d_fao, &d->dsc$w_length, d
					, d->dsc$w_length, d->dsc$a_pointer);
				if (s != SS$_NORMAL) {
					lib$stop(&MSG__SYSRET, 0, s);
				}
		}
		if (q->redirect_output) {
			q->output_rab.rab$w_rsz = d->dsc$w_length;
			q->output_rab.rab$l_rbf = d->dsc$a_pointer;
			s = sys$put(&q->output_rab);
			if (s != RMS$_NORMAL) {
				lib$stop(&MSG__RMSRET, 0, s);
			}
		} else {
			if (q->symbol) {
				put_symbol(q, d);
			} else {
				s = lib$put_output(d);
				if (s != SS$_NORMAL) {
					lib$stop(&MSG__RTLRET, 0, s);
				}
			}
		}
	}
}

/* --------------------------------------------------------------------------
 * put_symbol
 *	save output in a symbol
 *
 */
void put_symbol(struct quals *q, struct dsc$descriptor *d)
{
	int s;
	struct s_dsc n;
	struct dsc$descriptor v;
	$DESCRIPTOR(n_fao, "!AS_!UL");

	q->symbol_count++;

	init_s_dsc(&n);
	s = sys$fao(&n_fao, &n.dsc.dsc$w_length, &n.dsc
		, &q->symbol_string.dsc, q->symbol_count);
	if (s != SS$_NORMAL) {
		lib$stop(&MSG__SYSRET, 0, s);
	}

	v = *d;
	if (v.dsc$w_length > MAX_SYMLEN) {
		v.dsc$w_length = MAX_SYMLEN;
		lib$signal(&MSG__TRUNCATED, 1, q->symbol_count);
	}

	s = lib$set_symbol(&n, &v, 0);
	switch (s) {
	  case SS$_NORMAL :
		break;
	  case LIB$_INSCLIMEM :
		lib$signal(&MSG__SYMTABFULL, 1, q->symbol_count, s);
	  default :
		lib$stop(&MSG__RTLRET, 0, s);
	}

	put_sym_count(q);
}

/* --------------------------------------------------------------------------
 * put_sym_count
 *	save output symbol count
 *
 */
void put_sym_count(struct quals *q)
{
	int s;
	struct s_dsc v;
	$DESCRIPTOR(v_fao, "!UL");

	init_s_dsc(&v);
	s = sys$fao(&v_fao, &v.dsc.dsc$w_length, &v.dsc, q->symbol_count);
	if (s != SS$_NORMAL) {
		lib$stop(&MSG__SYSRET, 0, s);
	}

	s = lib$set_symbol(&q->symbol_string.dsc, &v, 0);
	if (s != SS$_NORMAL) {
		lib$stop(&MSG__RTLRET, 0, s);
	}

}

/* --------------------------------------------------------------------------
 * close_output
 *
 */
void close_output(struct quals *q)
{
	int s;

	if (q->redirect_output) {
		s = sys$disconnect(&q->output_rab);
		if (s != RMS$_NORMAL) {
			lib$stop(&MSG__RMSRET, 0, s);
		}
		s = sys$close(&q->output_fab);
		if (s != RMS$_NORMAL) {
			lib$stop(&MSG__RMSRET, 0, s);
		}
	}
}

/* --------------------------------------------------------------------------
 * save_context
 *	save dvi, did, fid and rfa in context symbol
 *
 */
void save_context(struct file *f, struct quals *q)
{
	int i;
	int s;
	struct context c;
	struct dsc$descriptor v;

	v.dsc$b_class = DSC$K_CLASS_S;
	v.dsc$w_length = sizeof(c);
	v.dsc$a_pointer = (char *)&c;

	c.length = sizeof(c);
	c.check = 0;

	for (i=0; i<16; i++) {
		c.dvi[i] = f->nam.nam$t_dvi[i];
		c.check += c.dvi[i];
	}
	for (i=0; i<3; i++) {
		c.did[i] = f->nam.nam$w_did[i];
		c.check += c.did[i];
		c.fid[i] = f->nam.nam$w_fid[i];
		c.check += c.fid[i];
		c.rfa[i] = f->last_rfa[i];
		c.check += c.rfa[i];
	}

	s = sys$gettim(&c.tim[0]);
	if (s != SS$_NORMAL) {
		lib$stop(&MSG__SYSRET, 0, s);
	}
	c.check += c.tim[0];
	c.check += c.tim[1];

	/* set so that lib$set_symbol does not truncate */
	c.trail = 0x42415247;

	s = lib$set_symbol(&q->context_string, &v, 0);
	if (s != SS$_NORMAL) {
		lib$stop(&MSG__RTLRET, 0, s);
	}
}

/* --------------------------------------------------------------------------
 * load_context
 *	load, check, and restore context
 *	returns TRUE if loaded, FALSE if new
 *
 */
int load_context(struct file *f, struct quals *q)
{
	int i;
	int s;
	struct context c;
	struct dsc$descriptor v;
	unsigned short retlen;
	long symtyp;

	if (q->new_context) {
		return(FALSE);
	}

	v.dsc$b_class = DSC$K_CLASS_S;
	v.dsc$w_length = sizeof(c);
	v.dsc$a_pointer = (char *)&c;

	s = lib$get_symbol(&q->context_string, &v, &retlen, &symtyp);
	switch (s)
	{
	  case SS$_NORMAL :
		break;
	  case LIB$_NOSUCHSYM :
		return(FALSE);
	  default :
		lib$stop(&MSG__RTLRET, 0, s);
	}

	if (symtyp != LIB$K_CLI_LOCAL_SYM) {
		return(FALSE);
	}

    /* allow zero length symbol to indicate new */

	if (retlen == 0) {
		return(FALSE);
	}

    /* structure & file consistent? */

	if (retlen != sizeof(c)) {
		lib$signal(&MSG__INVCONTEXT, 1, &q->context_string);
	}
	if (retlen != c.length) {
		lib$signal(&MSG__INVCONTEXT, 1, &q->context_string);
	}
	if (c.trail != 0x42415247) {
		lib$signal(&MSG__INVCONTEXT, 1, &q->context_string);
	}
	for (i=0; i<16; i++) {
		if (c.dvi[i] != f->nam.nam$t_dvi[i]) {
			return(FALSE);
		}
		c.check -= c.dvi[i];
	}
	for (i=0; i<3; i++) {
		if (c.did[i] != f->nam.nam$w_did[i]) {
			return(FALSE);
		}
		if (c.fid[i] != f->nam.nam$w_fid[i]) {
			return(FALSE);
		}
		c.check -= c.did[i];
		c.check -= c.fid[i];
		c.check -= c.rfa[i];
	}
	c.check -= c.tim[0];
	c.check -= c.tim[1];
	if (c.check != 0) {
		lib$signal(&MSG__INVCONTEXT, 1, &q->context_string);
	}

    /* restore RFA context */

	if ( !((c.rfa[0] == 0) && (c.rfa[1] == 0) && (c.rfa[2] == 0)) ) {
		f->rab.rab$w_rfa[0] = c.rfa[0];
		f->rab.rab$w_rfa[1] = c.rfa[1];
		f->rab.rab$w_rfa[2] = c.rfa[2];
		f->rab.rab$b_rac = RAB$C_RFA;
		s = sys$get(&f->rab);
		f->rec.dsc$w_length = f->rab.rab$w_rsz;
		if ((s != RMS$_NORMAL) && (s != RMS$_OK_RRL)) {
			lib$stop(&MSG__GETRFA, 0, s);
		}
		f->rab.rab$b_rac = RAB$C_SEQ;
		f->last_rfa[0] = c.rfa[0];
		f->last_rfa[1] = c.rfa[1];
		f->last_rfa[2] = c.rfa[2];
	}

	return(TRUE);
}

/* --------------------------------------------------------------------------
 * show_stats
 *
 */
void show_stats(struct quals *q, struct stats *a)
{
	lib$signal(&MSG__STATISTICS
		, 4, a->files, a->records, a->bytes, a->output);
	lib$show_timer();
}

/* --------------------------------------------------------------------------
 * save_summary
 *	save summary statistics in a symbol, this is ugly
 *	this needs to check that symbol length doesn't overflow
 *
 */
void save_summary(struct quals *q, struct stats *a, struct file *f)
{
	int i;
	int s;
	struct s_dsc n;
	char value[MAX_SYMLEN];
	struct dsc$descriptor_s v;
	unsigned short offset = 0;
	unsigned short length = 0;
	$DESCRIPTOR(d_summary, "GRAB_SUMMARY");
	$DESCRIPTOR(d_file, "GRAB_FILE_!SL");
	$DESCRIPTOR(d_f_sum, "!UL");
	$DESCRIPTOR(d_f_fil, "!AS");
	$DESCRIPTOR(d_fix, ",!UL,!UL,!UL,!UL,!UL");
	$DESCRIPTOR(d_var, ",!UL");

    /* symbol name */

	init_s_dsc(&n);
	if (f == 0) {
		s = sys$fao(&d_summary, &n.dsc.dsc$w_length, &n.dsc);
	} else {
		s = sys$fao(&d_file, &n.dsc.dsc$w_length, &n.dsc, f->num);
	}
	if (s != SS$_NORMAL) {
		lib$stop(&MSG__SYSRET, 0, s);
	}


    /* value */

	v.dsc$b_class = DSC$K_CLASS_S;
	v.dsc$w_length = MAX_SYMLEN;
	v.dsc$a_pointer = &value[0];

	if (f == 0) {
		s = sys$fao(&d_f_sum, &length, &v, a->files);
	} else {
		s = sys$fao(&d_f_fil, &length, &v, &f->name);
	}
	if (s != SS$_NORMAL) {
		lib$stop(&MSG__SYSRET, 0, s);
	}
	offset += length;
	v.dsc$w_length -= length;

    /* fixed portion */

	v.dsc$a_pointer = &value[offset];
	s = sys$fao(&d_fix, &length, &v
		, a->records
		, a->bytes
		, a->output
		, a->sections
		, q->string_max);
	if (s != SS$_NORMAL) {
		lib$stop(&MSG__SYSRET, 0, s);
	}
	offset += length;
	v.dsc$w_length -= length;

    /* variable portton */

	for (i=0; i<q->string_max; i++) {
		v.dsc$a_pointer = &value[offset];
		v.dsc$w_length -= length;
		s = sys$fao(&d_var, &length, &v, a->match[i]);
		if (s != SS$_NORMAL) {
			lib$stop(&MSG__SYSRET, 0, s);
		}
		offset += length;
	}

	v.dsc$a_pointer = &value[0];
	v.dsc$w_length = offset;

	s = lib$set_symbol(&n, &v, 0);
	if (s != SS$_NORMAL) {
		lib$stop(&MSG__RTLRET, 0, s);
	}
}

/* --------------------------------------------------------------------------
 * zero_stats
 *
 */
void zero_stats(struct stats *a)
{
	int i;

	a->files    = 0;
	a->records  = 0;
	a->bytes    = 0;
	a->output   = 0;
	a->sections = 0;

	for (i=0; i<MAX_STR; i++) {
		a->match[i] = 0;
	}
}

/* --------------------------------------------------------------------------
 * add_stats
 *
 */
void add_stats(struct stats *a, struct stats *b)
{
	int i;

	a->files    += b->files;
	a->records  += b->records;
	a->bytes    += b->bytes;
	a->output   += b->output;
	a->sections += b->sections;

	for (i=0; i<MAX_STR; i++) {
		a->match[i] += b->match[i];
	}
}

/* --------------------------------------------------------------------------
 * open_file
 *
 */
int open_file(struct file *f, struct quals *q)
{
	int s;

	zero_stats(&f->stat);

    /* FAB - readonly, allow everything */

	f->fab = cc$rms_fab;
	f->fab.fab$b_fac = FAB$M_GET;
	f->fab.fab$l_fna = f->name.dsc$a_pointer;
	f->fab.fab$b_fns = f->name.dsc$w_length;

    /* NAM - required for context */

	f->nam = cc$rms_nam;
	f->fab.fab$l_nam = &f->nam;

    /* FAB$B_SHR and UPI
     * should really only be for sequential access
     * UPI is only enabled on S_WRITE as this reduces the locking activity
     * on SEQ files, it's not needed for S_READ or S_NONE. I believe that
     * UPI will be disabled for IDX or REL
     */

	switch (q->share_type)
	{
	  case S_NONE :
		f->fab.fab$b_shr = FAB$M_NIL;
		break;
	  case S_READ :
		f->fab.fab$b_shr = FAB$M_SHRGET;
		break;
	  default :
	  case S_WRITE :
		f->fab.fab$b_shr = FAB$M_UPI | FAB$M_SHRGET | FAB$M_SHRPUT
					| FAB$M_SHRUPD 	| FAB$M_SHRDEL;
	}

	s = sys$open(&f->fab);

	switch (s)
	{
	  case RMS$_NORMAL :
		break;
	  case RMS$_FNF :
		/* LIB$FIND_FILE found it, go figure */
	  case RMS$_FLK :
	  case RMS$_PRV :
		lib$signal(&MSG__OPEN, 1, &f->name, s);
		return(FALSE);
		break;
	  default :
		lib$stop(&MSG__RMSRET, 0, s);
	}

    /* RAB - do not lock, read regardless */

	f->rab = cc$rms_rab;
	f->rab.rab$l_fab = &f->fab;
	f->rab.rab$b_rac = RAB$C_SEQ;
	f->rab.rab$l_rop = RAB$M_NLK | RAB$M_RRL;

	switch (f->fab.fab$b_org)
	{
	  case FAB$C_SEQ :
		f->rab.rab$l_rop |= RAB$M_RAH;
		break;
	  default :
		break;
	}

	s = sys$connect(&f->rab);
	if (s != RMS$_NORMAL) {
		lib$stop(&MSG__RMSRET, 0, s);
	}

    /* record buffer */

	f->rab.rab$l_ubf = &f->rec_buff[0];
	f->rab.rab$w_usz = MAX_BUFLEN;

	f->rec.dsc$b_class = DSC$K_CLASS_S;
	f->rec.dsc$a_pointer = &f->rec_buff[0];

	f->last_rfa[0] = 0;
	f->last_rfa[1] = 0;
	f->last_rfa[2] = 0;

	if (q->context) {
		if ( load_context(f, q) ) {
			lib$signal(&MSG__USECONTEXT);
		} else {
			lib$signal(&MSG__NEWCONTEXT);
		}
	}

	return(TRUE);
}

/* --------------------------------------------------------------------------
 * close_file
 *
 */
void close_file(struct file *f, struct quals *q, struct stats *x)
{
	int s;

	s = sys$disconnect(&f->rab);
	if (s != RMS$_NORMAL) {
		lib$stop(&MSG__RMSRET, 0, s);
	}

	s = sys$close(&f->fab);
	if (s != RMS$_NORMAL) {
		lib$stop(&MSG__RMSRET, 0, s);
	}

	add_stats(x, &f->stat);
	if (q->log) {
		lib$signal(&MSG__LOG
			, 3, &f->name, f->stat.records, f->stat.output);
	}

	if (q->context) {
		save_context(f, q);
	}
}

/* --------------------------------------------------------------------------
 * rewind_file
 *
 */
void rewind_file(struct file *f, struct quals *q)
{
	int s;
	unsigned short rec_rfa[3] = {0,0,0};

	rec_rfa[0] = f->rab.rab$w_rfa[0];
	rec_rfa[1] = f->rab.rab$w_rfa[1];
	rec_rfa[2] = f->rab.rab$w_rfa[2];

	f->rab.rab$w_rfa[0] = f->saved_rfa[0];
	f->rab.rab$w_rfa[1] = f->saved_rfa[1];
	f->rab.rab$w_rfa[2] = f->saved_rfa[2];
	f->rab.rab$b_rac = RAB$C_RFA;
	s = sys$get(&f->rab);
	f->rec.dsc$w_length = f->rab.rab$w_rsz;
	if ((s != RMS$_NORMAL) && (s != RMS$_OK_RRL)) {
		lib$stop(&MSG__RMSRET, 0, s);
	}
	f->rab.rab$b_rac = RAB$C_SEQ;

	while (1) {
		if ((f->rab.rab$w_rfa[0] == rec_rfa[0])
		  && (f->rab.rab$w_rfa[1] == rec_rfa[1])
		  && (f->rab.rab$w_rfa[2] == rec_rfa[2])) {
			break;
		}
		if (q->highlight) {
			highlight(q, &f->rec);
		} else {
			put_output(q, &f->rec);
		}
		f->stat.output++;
		s = sys$get(&f->rab);
		f->rec.dsc$w_length = f->rab.rab$w_rsz;
		if ((s != RMS$_NORMAL) && (s != RMS$_OK_RRL)) {
			lib$stop(&MSG__RMSRET, 0, s);
		}
	}
}

/* --------------------------------------------------------------------------
 * search_file
 *
 *
 *
 *
 */
void search_file(struct file *f, struct quals *q)
{
	int i;
	int s;
	int rec_s;
	char cmp_buff[MAX_BUFLEN];
	struct dsc$descriptor d_cmp;
	int in_sec = FALSE;
	int new_sec = FALSE;
	int end_sec = FALSE;
	int flush_sec = FALSE;
	int match = FALSE;
	int str_match[MAX_STR];
	int cut_match = FALSE;
	int display_on = FALSE;
	int pos, start;

	d_cmp.dsc$b_class = DSC$K_CLASS_S;
	d_cmp.dsc$a_pointer = &cmp_buff[0];

	for (i=0; i<MAX_STR; i++) {
		str_match[i] = FALSE;
	}
	if (q->limit) {
		q->limit_reached = FALSE;
	}

	while( (rec_s = sys$get(&f->rab)) != RMS$_EOF ) {
		if ((rec_s != RMS$_NORMAL) && (rec_s != RMS$_OK_RRL)) {
			lib$stop(&MSG__RMSRET, 0, rec_s);
		}

		f->rec.dsc$w_length = f->rab.rab$w_rsz;
		f->stat.records++;
		f->stat.bytes+= f->rab.rab$w_rsz;

	    /* compare descriptor points at rms buffer if exact */

		if (q->exact) {
			d_cmp = f->rec;
		} else {
			d_cmp.dsc$w_length = f->rab.rab$w_rsz;
			str$upcase(&d_cmp, &f->rec);
		}

	    /* check for cut */

		if (q->cut) {
			if (end_sec) {
				end_sec = FALSE;
				in_sec = FALSE;
				flush_sec = FALSE;
				display_on = FALSE;
			}
			new_sec = FALSE;
			cut_match = FALSE;
			if (q->wild) {
				if (str$match_wild(&d_cmp
						,&q->cut_string[in_sec].dsc)
							== STR$_MATCH) {
					cut_match = TRUE;
				}
			} else {
				if (str$position(&d_cmp
						, &q->cut_string[in_sec].dsc)
							 != 0) {
					cut_match = TRUE;
				}
			}
			if (cut_match) {
				if (q->cut_max == 1) {
					in_sec = TRUE;
					new_sec = TRUE;
					display_on = FALSE;
				} else {
					if (in_sec) {
						end_sec = TRUE;
					} else {
						in_sec = TRUE;
						new_sec = TRUE;
					}
				}
				if (new_sec) {
					if (q->limit) {
						if (f->stat.sections >=
						    q->limit_value) {
							q->limit_reached = TRUE;
							break;
						}
					}
					flush_sec = TRUE;
					f->saved_rfa[0] = f->rab.rab$w_rfa[0];
					f->saved_rfa[1] = f->rab.rab$w_rfa[1];
					f->saved_rfa[2] = f->rab.rab$w_rfa[2];
				}
			}
		}

	    /* check for strings */

	      /* reset match indeces every record or section boundary */

		if ((!q->cut) || (new_sec)) {
			for (i=0; i<MAX_STR; i++) {
				str_match[i] = FALSE;
			}
		}

		for (i=0; i<q->string_max; i++) {
			if (q->string[i].dsc.dsc$w_length == 0) {
				str_match[i] = TRUE;
				f->stat.match[i]++;
			} else {
				if (q->wild) {
					if (str$match_wild(&d_cmp,
						&q->string[i].dsc)
							== STR$_MATCH) {
						str_match[i] = TRUE;
						f->stat.match[i]++;
					}
				} else {
					start = 0;
					while (1) {
						pos = str$position(&d_cmp
							, &q->string[i].dsc
							, &start);
						if (pos == 0) {
							break;
						}
						str_match[i] = TRUE;
						f->stat.match[i]++;
						start = pos +
						  q->string[i].dsc.dsc$w_length;
					}
				}
			}
		}

	    /* evalute match */
	     /* Q: should we reaaly allow strings in cut records? */
	      /* unecessary if already matching section (display_on) */

		switch (q->match_type) {
		  case M_OR :
			match = FALSE;
			for (i=0; i<q->string_max; i++) {
				if (str_match[i]) {
					match = TRUE;
					break;
				}
			}
			break;
		  case M_AND :
			match = TRUE;
			for (i=0; i<q->string_max; i++) {
				if (!str_match[i]) {
					match = FALSE;
					break;
				}
			}
			break;
		}

	    /* display? */

		if (q->cut) {
			if (in_sec && match) {
				display_on = TRUE;
			}
		} else {
			if (match) {
				display_on = TRUE;
			} else {
				display_on = FALSE;
			}
		}

		if (display_on) {
			if ((q->header) && (f->stat.output == 0)) {
				put_output(q, &q->header_string.dsc);
			}
			if (q->cut) {
				if (flush_sec) {
					f->stat.sections++;
					if (q->sep) {
						put_output(q
							, &q->sep_string.dsc);
					}
					rewind_file(f, q);
					flush_sec = FALSE;
				}
			}
			if (q->highlight) {
				highlight(q, &f->rec);
			} else {
				put_output(q, &f->rec);
			}
			f->stat.output++;
		}

		if (q->context) {
			f->last_rfa[0] = f->rab.rab$w_rfa[0];
			f->last_rfa[1] = f->rab.rab$w_rfa[1];
			f->last_rfa[2] = f->rab.rab$w_rfa[2];
		}
		if ((q->limit) && !(q->cut)) {
			if (f->stat.output >= q->limit_value) {
				q->limit_reached = TRUE;

				break;
			}
		}
	}

	if ((q->trailer) && (f->stat.output > 0)) {
		put_output(q, &q->trailer_string.dsc);
	}

     /* if limit reached see if we're at the end to return the right status */

	if ( (q->limit) && (q->limit_reached) ) {
		rec_s = sys$get(&f->rab);
		switch (rec_s) {
		  case RMS$_NORMAL :
		  case RMS$_OK_RRL :
			if (q->cut) {
				lib$signal(&MSG__SECLIMIT, 1, q->limit_value);
			} else {
				lib$signal(&MSG__RECLIMIT, 1, q->limit_value);
			}
			break;
		  case RMS$_EOF :
			q->limit_reached = FALSE;
			break;
		  default :
			lib$stop(&MSG__RMSRET, 0, rec_s);
		}
	}

	f->stat.files++;

}
/* --------------------------------------------------------------------------
 * highlight
 *
 *  because of section processing this is called at output, and involves
 *  a re-search ... seems inefficient
 *
 *  - no highlighting of cut strings
 *  - highlighting for wildcards is the whole record
 *
 */
void highlight(struct quals *q, struct dsc$descriptor *d_out)
{
	int i;
	char cmp_buff[MAX_BUFLEN];
	struct dsc$descriptor d_cmp;
	struct dsc$descriptor d_cop;
	struct dsc$descriptor d_new;
	struct s_vs new_buff;
	int pos, start;
	int h_count = 0;
	int h_pos[MAX_HIL];
	int h_len[MAX_HIL];
	int i_count, min, new_min, len, rem;
	int i_pos[MAX_HIL];
	int i_len[MAX_HIL];
	$DESCRIPTOR(d_off, "\x01b[0m");
	struct dsc$descriptor d_on;

	d_on = q->high_style;

	if (q->exact) {
		d_cmp = *d_out;
	} else {
		d_cmp.dsc$w_length = d_out->dsc$w_length;
		d_cmp.dsc$b_class = DSC$K_CLASS_S;
		d_cmp.dsc$a_pointer = &cmp_buff[0];
		str$upcase(&d_cmp, d_out);
	}

	for (i=0; i<q->string_max; i++) {
		if (q->string[i].dsc.dsc$w_length == 0) {
			/* NULL match - do nothing ? */
		} else {
			if (q->wild) {
				if (str$match_wild(&d_cmp,
					&q->string[i].dsc)
						== STR$_MATCH) {
					h_pos[0] = 0;
					h_len[0] = d_out->dsc$w_length;
					h_count = 1;
					break;
				}
			} else {
				start = 0;
				while (1) {
					pos = str$position(&d_cmp
						, &q->string[i].dsc
						, &start);
					if (pos == 0) {
						break;
					}
					start = pos +
					  q->string[i].dsc.dsc$w_length;
					h_pos[h_count] = pos - 1;
					h_len[h_count] =
						q->string[i].dsc.dsc$w_length;
					h_count++;
					/* too many, highlight whole record */
					if (h_count > MAX_HIL) {
						h_pos[0] = 0;
						h_len[0] = d_out->dsc$w_length;
						h_count = 1;
						break;
					}
				}
			}
		}
	}

	if (h_count == 0) {
		put_output(q, d_out);
		return;
	}

    /* sort positions into order & remove duplicates selecting longest */

	new_min = 0x7fffffff;
	rem = h_count;
	i_count = 0;
	while(rem > 0) {
		min = new_min;
		new_min = 0x7fffffff;
		len = 0;
		for (i=0; i<h_count; i++) {
			if (h_pos[i] == min) {
				h_pos[i] = 0x7fffffff;
				rem--;
				if (h_len[i] > len) {
					len = h_len[i];
				}
			} else {
				if (h_pos[i] < new_min) {
					new_min = h_pos[i];
				}
			}
		}
		if (len > 0) {
			i_pos[i_count] = min;
			i_len[i_count] = len;
			i_count++;
		}
	}

    /* consolidate overlapping areas */

	h_count = -1;
	pos = -1;
	for (i=0; i<i_count; i++) {
		if (i_pos[i] > pos) {
			h_count++;
			h_pos[h_count] = i_pos[i];
			h_len[h_count] = i_len[i];
		} else {
			h_len[h_count] += (i_pos[i] + i_len[i] - pos);
		}
		pos = i_pos[i] + i_len[i];
	}
	h_count++;

    /* now make the output */

	new_buff.curlen = 0;
	d_new.dsc$b_class = DSC$K_CLASS_VS;
	d_new.dsc$a_pointer = (char *) &new_buff.curlen;
	d_new.dsc$w_length = MAX_BUFLEN;

      /* check for overflow, str$append will truncate automatically */
      /* if it doesn't fit try and highlight whole record, or give up */

	len = d_out->dsc$w_length +
		( h_count * ( d_on.dsc$w_length + d_off.dsc$w_length ) );
	if (len > d_new.dsc$w_length) {
		len = d_out->dsc$w_length +
			d_on.dsc$w_length + d_off.dsc$w_length;
		if (len > d_new.dsc$w_length) {
			put_output(q, d_out);
			return;
		} else {
			h_pos[0] = 0;
			h_len[0] = d_out->dsc$w_length;
			h_count = 1;
		}
	}

	d_cop = *d_out;
	pos = 0;
	for (i=0; i<h_count; i++) {
		len = h_pos[i] - pos;
		d_cop.dsc$w_length = len;
		str$append(&d_new, &d_cop);
		str$append(&d_new, &d_on);
		pos += len;
		d_cop.dsc$a_pointer += len;
		d_cop.dsc$w_length = h_len[i];
		str$append(&d_new, &d_cop);
		str$append(&d_new, &d_off);
		pos += h_len[i];
		d_cop.dsc$a_pointer += h_len[i];
	}
	d_cop.dsc$w_length = d_out->dsc$w_length - pos;
	str$append(&d_new, &d_cop);

    /* lib$put_output doesn't seem to like VS so ... */

	d_cop.dsc$w_length = new_buff.curlen;
	d_cop.dsc$a_pointer = &new_buff.buff[0];

	put_output(q, &d_cop);
}
