/*
 * This routine is written to alleviate a shortcoming in RMS $SEARCH - RMS
 * will only return filenames in decreasing version number order.  This is
 * often the wrong thing, especially if a routine wants to see files in
 * chronological order.
 *
 * So what we do is $SEARCH, saving the FAB and NAM fields which are altered
 * by $SEARCH, and the resultant string, in a stack in memory.  When any part
 * of the file specification being returned by $SEARCH, other than the ver-
 * sion, is different, we pop the stack, returning properly-filled-in FABs
 * and NAMs to the caller.  When the stack is empty, we resume the $SEARCH.
 *
 * lib$search is called exactly as RMS $SEARCH - with a pointer to a FAB
 * which has been $PARSEd
 *
 *	Terry R. Friedrichsen
 *	Sunquest Information Systems
 *	terry@uplift.sunquest.com
 */

#include <stdio.h>
#include <stdlib.h>

#include <ssdef.h>
#include <rms.h>

#include <starlet.h>
#include <lib$routines.h>

#define	NIL   0				/* pointer to nowhere */

#define until(x)   while (! (x))

#define MAXFILESPEC   252		/* maximum length of a file spec */

/* save the FAB and NAM pieces we need */
struct file_node {
  struct file_node *prevfile;	/* pointer to previous spec */
  long retsts;				/* $SEARCH return status */
  long fabsts, fabstv;			/* FAB$L_STS and FAB$L_STV */
  long namfnb, namwcc;			/* NAM$L_FNB and NAM$L_WCC */
  short namfid[3];			/* NAM$W_FID */
  unsigned char namrsl;			/* NAM$B_RSL */
  unsigned char namrsa[MAXFILESPEC];	/* hold resultant string */
} *file_stack = (struct file_node *)NIL;

#define NODESIZE   sizeof(struct file_node)

/* next file after a string of matching filenames */
static struct file_node *next = (struct file_node *)NIL;

static long popstack(struct FAB *);
static void save_fab_nam(struct file_node *, struct FAB *);
static void restore_fab_nam(struct FAB *, struct file_node *);
static int semicmp(unsigned char *, unsigned char *);



/* the user passes us the address of the FAB, which must already be $PARSEd.
 * from it, we'll get the NAM block address and be off and running */

long lib$search(userfab)

struct FAB *userfab;

{
  int node_size;
  long rms_status, lib_status;
  int i;
  struct file_node *base;
  struct NAM *usernam;

/* if the stack is not empty, we must be in the process of emptying it */
  if (file_stack != (struct file_node *)NIL)
    return popstack(userfab);

/* the stack is empty; if next has something, put it on the stack, unless this
 * is the "no more files" case */
  node_size = NODESIZE;
  if (next != NIL)
    /* if next contains the "no more files" status, free memory and return */
    if (next->retsts == RMS$_NMF) {
      lib_status = lib$free_vm(&node_size, &next, 0);
      if (lib_status != SS$_NORMAL)
        lib$signal(lib_status);

      return RMS$_NMF;
    }
    else {
/* put this entry on the stack */
      next->prevfile = file_stack;
      file_stack = next;

/* and reset the NAM to prepare for the $SEARCH */
      restore_fab_nam(userfab, next);
    }

/* loop stacking $SEARCH results until the filespec changes */
  while (TRUE) {
/* get a chunk of memory for the next $SEARCH result */
    lib_status = lib$get_vm(&node_size, &next, 0);
    if (lib_status != SS$_NORMAL)
      lib$signal(lib_status);

/* do the $SEARCH and save away the results */
    rms_status = sys$search(userfab);

    /* for certain statuses, we just never got started; trap them and free
     * the memory, then return the error */
    if ((rms_status == RMS$_FNF) || (rms_status == RMS$_ESL)) {
      lib_status = lib$free_vm(&node_size, &next, 0);
      if (lib_status != SS$_NORMAL)
        lib$signal(lib_status);

      return rms_status;
    }

/* when the "no more files" status gets returned, all we need to do is empty
 * the stack */
    if ((next->retsts = rms_status) == RMS$_NMF)
      break;

/* save the changeable fields of the FAB and the NAM for when we need them */
    save_fab_nam(next, userfab);

/* at this point, we have the results of the $SEARCH in hand.  if this
 * filespec matches the previous one, we must stack this result and
 * continue searching (if the stack is empty, this filespec matches
 * vacuously).  if it is different, we must hold onto this one, empty
 * the stack, and then stack this filespec. */
    if ((file_stack == (struct file_node *)NIL) ||
			      (! semicmp(next->namrsa, file_stack->namrsa))) {
      /* put this entry on the stack */
      next->prevfile = file_stack;
      file_stack = next;
    }
    else
      /* no match; leave this filespec in next and begin emptying the stack */
      break;
  }

/* here when the file specification changes - unstack the previous filespec */
  return popstack(userfab);
}



/* here when a non-matching filespec is found.  return the filespecs from the
 * stack, which contains all versions of the previous filespec, one per call */

static long popstack(userfab)

struct FAB *userfab;

{
  struct file_node *base;
  long rms_status, lib_status;
  int node_size;

/* pop an entry off the stack */
  base = file_stack;
  file_stack = file_stack->prevfile;

/* restore the return status and the appropriate FAB and NAM fields */
  rms_status = base->retsts;
  restore_fab_nam(userfab, base);

/* and return the chunk of memory to VMS */
  node_size = NODESIZE;
  lib_status = lib$free_vm(&node_size, &base, 0);
  if (lib_status != SS$_NORMAL)
     lib$signal(lib_status);

/* and return the status of this $SEARCH to the caller */
  return rms_status;
}



/* save the changeable fields of the FAB and NAM into the given file node */

static void save_fab_nam(filenode, file_fab)

struct file_node *filenode;
struct FAB *file_fab;

{
  struct NAM *file_nam;
  int i;

/* copy the FAB status fields from the user FAB to the file node */
  filenode->fabsts = file_fab->fab$l_sts;
  filenode->fabstv = file_fab->fab$l_stv;

/* get the pointer to the NAM block and save the NAM if the pointer exists */
  if ((file_nam = file_fab->fab$l_nam) != (struct NAM *)NIL) {
    filenode->namfnb = file_nam->nam$l_fnb;
    filenode->namwcc = file_nam->nam$l_wcc;

    /* save the file ID */
    for (i = 0; i < 3; i++)
      filenode->namfid[i] = file_nam->nam$w_fid[i];

    /* if there is a resultant-string address, save the resultant string */
    filenode->namrsl = file_nam->nam$b_rsl;
    if (file_nam->nam$l_rsa != (char *)NIL)
      for (i = 0; i < filenode->namrsl; i++)
        filenode->namrsa[i] = file_nam->nam$l_rsa[i];
  }

/* we never fail */
  return;
}



/* restore the appropriate FAB and NAM fields from the given file node */

static void restore_fab_nam(file_fab, filenode)

struct FAB *file_fab;
struct file_node *filenode;

{
  struct NAM *file_nam;
  int i;

/* copy the FAB status fields from the file node to the user FAB */
  file_fab->fab$l_sts = filenode->fabsts;
  file_fab->fab$l_stv = filenode->fabstv;

/* get the pointer to the NAM block and fill in the NAM if pointer exists */
  if ((file_nam = file_fab->fab$l_nam) != (struct NAM *)NIL) {
    file_nam->nam$l_fnb = filenode->namfnb;
    file_nam->nam$l_wcc = filenode->namwcc;

    /* restore the file ID */
    for (i = 0; i < 3; i++)
      file_nam->nam$w_fid[i] = filenode->namfid[i];

    /* if there is a resultant-string address, fill in the resultant string */
    file_nam->nam$b_rsl = filenode->namrsl;
    if (file_nam->nam$l_rsa != (char *)NIL)
      for (i = 0; i < filenode->namrsl; i++)
        file_nam->nam$l_rsa[i] = filenode->namrsa[i];
  }

/* we never fail */
  return;
}



/* compare strings s and t until a semicolon is found; return 0 for match and
 * non-zero for no match.  note that this function expects both strings to con-
 * tain at least one semicolon and consequently does not check for nul-termi-
 * nation */

static int semicmp(s, t)

unsigned char *s, *t;

{
  int retval = 0;

  do
    if (retval = *s - *t)
      break;
  until ((*s++ == ';') || (*t++ == ';'));

/* return the lexicographic difference between the final characters */
  return retval;
}
