/*
 * Copyright (c) 1983, 1988, 1993, 1994
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright (c) 1983, 1988, 1993, 1994\n\
	The Regents of the University of California.  All rights reserved.\n";
#endif /* not lint */

#ifndef lint
static char sccsid[] = "@(#)syslogd.c	8.3 (Berkeley) 4/4/94";
#endif /* not lint */


/*
**++
**  FACILITY:  syslogd
**
**  MODULE DESCRIPTION:
**
**      syslogd for OpenVMS.  syslogd is a fairly standard part of most
**  UNIX systems.  It is a daemon process which deals with messages from
**  consoles, the kernel and from other machines.  This implementation
**  is OpenVMS specific.  It only deals with messages from other machines
**  since there are other facilities under OpenVMS to deal with console
**  and "kernel" messages.
**
**	This program is normally run as a detached process.  It creates
**  a UDP port and reads messages from it.  It then logs those messages.
**
**  AUTHORS:
**
**      John Vottero
**
**  CREATION DATE:  05-May-1995
**
**  DESIGN ISSUES:
**
**      {@tbs@}
**
**  [@optional module tags@]...
**
**  MODIFICATION HISTORY:
**
**      27-Jul-1995 Fixed a problem with uninitialized data
**		    which didn't show up until UCX V3.3
**
**	09-Aug-1995 syslog would loop if UCX was shutdown.
**--
*/


/*
**
**  INCLUDE FILES
**
*/

#include <stddef.h>
#include <ctype.h>
#include <assert.h>
#include <limits.h>
#include <signal.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <unixio.h>
#include <file.h>
#include <time.h>

#define SYSLOG_NAMES
#include "syslog.h"

/*
**  TCP/IP (UCX) Includes
*/
#include <in.h>
#include <inet.h>
#include <netdb.h>
#include <socket.h>

/*
**  VMS Specific Includes
*/
#include <brkdef.h>
#include <descrip.h>
#include <opcdef.h>
#include <starlet.h>

#define	dprintf		if (debugging) printf
#define bitmask(the_bit) (1 << (the_bit - 1))
#define MAXHOSTNAMELEN 64
#define TIMERINTVL 300

#define	MAXLINE		1024		/* maximum line length */
#define	MAXSVLINE	120		/* maximum saved line length */
#define DEFUPRI		(LOG_USER|LOG_NOTICE)
#define DEFSPRI		(LOG_KERN|LOG_CRIT)

#define UT_NAMESIZE 12
#define MAXUNAMES 32
#define LINE_MAX 1024
#define MAXPATHLEN 256

static char *ConfFile = "SYSLOGD_CONFIG";

/*
 * Flags to logmsg().
 */

#define IGN_CONS	0x001	/* don't print on console */
#define SYNC_FILE	0x002	/* do fsync on file after printing */
#define ADDDATE		0x004	/* add a date to the message */
#define MARK		0x008	/* this message is a mark */

/*
 * This structure represents the files that will have log
 * copies printed.
 */

struct filed {
	struct	filed *f_next;		/* next in linked list */
	short	f_type;			/* entry type, see below */
	short	f_file;			/* file descriptor */
	time_t	f_time;			/* time this was last written */
	u_char	f_pmask[LOG_NFACILITIES+1];	/* priority mask */
	union {
		char	f_uname[MAXUNAMES][UT_NAMESIZE+1];
		struct {
			char	f_hname[MAXHOSTNAMELEN+1];
			struct sockaddr_in	f_addr;
		} f_forw;		/* forwarding address */
		char	f_fname[MAXPATHLEN];
		unsigned int opcom_class_bm;
	} f_un;
	char	f_prevline[MAXSVLINE];		/* last message logged */
	char	f_lasttime[16];			/* time of last occurrence */
	char	f_prevhost[MAXHOSTNAMELEN+1];	/* host from which recd. */
	int	f_prevpri;			/* pri of f_prevline */
	int	f_prevlen;			/* length of f_prevline */
	int	f_prevcount;			/* repetition cnt of prevline */
	int	f_repeatcount;			/* number of "repeated" msgs */
};

/*
 * Intervals at which we flush out "message repeated" messages,
 * in seconds after previous message is logged.  After each flush,
 * we move to the next interval until we reach the largest.
 */
int	repeatinterval[] = { 30, 120, 600 };	/* # of secs before flush */
#define	MAXREPEAT ((sizeof(repeatinterval) / sizeof(repeatinterval[0])) - 1)
#define	REPEATTIME(f)	((f)->f_time + repeatinterval[(f)->f_repeatcount])
#define	BACKOFF(f)	{ if (++(f)->f_repeatcount > MAXREPEAT) \
				 (f)->f_repeatcount = MAXREPEAT; \
			}

/* values for f_type */
#define F_UNUSED	0		/* unused entry */
#define F_FILE		1		/* regular file */
#define F_TTY		2		/* terminal */
#define F_CONSOLE	3		/* console terminal */
#define F_FORW		4		/* remote machine */
#define F_USERS		5		/* list of users */
#define F_WALL		6		/* everyone logged on */

char	*TypeNames[7] = {
	"UNUSED",	"FILE",		"TTY",		"CONSOLE",
	"FORW",		"USERS",	"WALL"
};

struct	filed *Files;

static int	shutdown_requested;
static int	debugging;		/* debug flag */
static char	LocalHostName[MAXHOSTNAMELEN+1];	/* our hostname */
static char	syslog_line[1024];
static int	sd_syslog;		/* The syslog Socket Descriptor */
static int	syslog_port;		/* The UDP port syslogd is using */
static int	flush_count;		/* Number of times we have flushed */
					/* We log a message every 20 flushes */
static char time_str[32];		/* Work area for current_time_str */
static char *unknown_str = "Unknown";
static char *invalid_addr_str = "Invalid Address";

/*
**  OPCOM Classes Code Table
*/
CODE opcom_classes[] = {
    "cards",	    OPC$M_NM_CARDS,
    "central",	    OPC$M_NM_CENTRL,
    "cluster",	    OPC$M_NM_CLUSTER,
    "devices",	    OPC$M_NM_DEVICE,
    "disks",	    OPC$M_NM_DISKS,
    "license",	    OPC$M_NM_LICENSE,
    "network",	    OPC$M_NM_NTWORK,
    "oper1",	    OPC$M_NM_OPER1,
    "oper2",	    OPC$M_NM_OPER2,
    "oper3",	    OPC$M_NM_OPER3,
    "oper4",	    OPC$M_NM_OPER4,
    "oper5",	    OPC$M_NM_OPER5,
    "oper6",	    OPC$M_NM_OPER6,
    "oper7",	    OPC$M_NM_OPER7,
    "oper8",	    OPC$M_NM_OPER8,
    "oper9",	    OPC$M_NM_OPER9,
    "oper10",	    OPC$M_NM_OPER10,
    "oper11",	    OPC$M_NM_OPER11,
    "oper12",	    OPC$M_NM_OPER12,
    "printer",	    OPC$M_NM_PRINT,
    "security",	    OPC$M_NM_SECURITY,
    "tape",	    OPC$M_NM_TAPES,
    NULL,	    -1};


/*
**  Static Function Prototypes
*/
static void decode_line(
    char *raw_line,
    struct sockaddr_in *from,
    int from_len);

static void logerror(
    char *type);

static void open_log(
    struct filed *f,
    int new_file);

static void close_logs(
    char *close_msg);

static char *current_time_str(void);

static char *facility_to_name(
    int facility_int);

static char *priority_to_name(
    int priority_int);

static void flush_and_mark(
    int signal_code);

static void cleanup_and_exit(
    int signal_code);

void init(
    int signo);

void cfline(
    char *line,
    struct filed *f);

void logmsg(
    int pri,
    char *msg, 
    char *from,
    int flags);

void fprintlog(
    struct filed *f,
    int flags,
    char *msg);

void wallmsg(
    struct filed *f,
    struct iovec *iov);

int decode(
    const char *name,
    CODE *codetab);

int main(
    int argc,
    char **argv)
{
    int argi;
    char *p;
    struct servent *service_ent;
    struct sockaddr_in socket_name;

    /*
    **  Parse the command line options
    */
    debugging = FALSE;
    for (argi=1; argi<argc; argi++)
	{
	if (strcmp(argv[argi], "-d") == 0)
	    {
	    debugging = TRUE;
	    }
	else
	    {
	    /*
	    **  Bogus parameter
	    */
	    printf("%s is an invalid parameter\n",
		argv[argi]);
	    exit(0);
	    } 
	}

    /*
    **  Get the local host name
    */
    (void) gethostname(LocalHostName, sizeof LocalHostName);
    if ((p = strchr(LocalHostName, '.')) != NULL) {
	    *p++ = '\0';
    }

    /*
    **  Create a socket
    */
    sd_syslog = socket(AF_INET, SOCK_DGRAM, 0);
    if (sd_syslog < 0)
	{
	logerror("Unable to create socket");
	exit(0);
	}

    /*
    **  Get the syslogd service name
    */
    service_ent = getservbyname("syslog", "udp");
    if (service_ent == NULL)
	{
	errno = 0;
	logerror("syslog/udp: unknown service");
	exit(0);
	}
    syslog_port = service_ent->s_port;

    /*
    **  Declare signal handlers
    */
    flush_count = 0;
    (void) signal(SIGTERM, cleanup_and_exit);
    (void) signal(SIGINT, cleanup_and_exit);
    (void) signal(SIGALRM, flush_and_mark);
    (void) alarm(TIMERINTVL);

    /*
    **  Bind to the socket
    */
    socket_name.sin_family = AF_INET;
    socket_name.sin_port = service_ent->s_port;
    socket_name.sin_addr.s_addr = 0;
    memset(socket_name.sin_zero, 0, sizeof(socket_name.sin_zero));
    dprintf("Binding to %s on port %d\n", service_ent->s_name, 
	ntohs(service_ent->s_port));
    if (bind(
	    sd_syslog, 
	    (struct sockaddr *) &socket_name, 
	    sizeof(socket_name)) < 0)
	{
	printf("Bind returned %d\n", errno);
	logerror("bind failed");
	exit(0);
	}

    /*
    **  Read the Configuration File
    */
    init(0);

    /*
    **  Main loop !!
    */
    shutdown_requested = FALSE;
    while (!shutdown_requested)
	{
	int number_ready;
	int readfds;

	dprintf("Selecting %d\n", sd_syslog);
	readfds = (1 << sd_syslog);
	number_ready = select(
			    (sd_syslog + 1),	/* Highest to check */
			    &readfds,
			    NULL,	/* writefds */
			    NULL,	/* exceptfds */
			    NULL);	/* Timeout */

	if (number_ready > 0)
	    {
	    int line_len;
	    int from_len;
	    struct sockaddr_in from;

	    dprintf("Receiving\n");
	    from_len = sizeof(from);
	    line_len = recvfrom(
			    sd_syslog, 
			    syslog_line, 
			    sizeof(syslog_line) - 1,
			    0,		    /* flags */
			    (struct sockaddr *) &from,
			    &from_len);
	    if (line_len > 0)
		{
		syslog_line[line_len] = '\0';
		decode_line(syslog_line, &from, from_len);
		}
	    else if (line_len < 0)
		{
		/*
		**  recvfrom returned an error
		*/
		printf("recvfrom returned an error %d %d", line_len, errno);
		}
	    else
		{
		printf("Line length is zero ??\n");
		}
	    }
	else if (number_ready < 0)
	    {
	    /*
	    **  The select returned an error
	    */
	    logerror("select returned an error");
	    shutdown_requested = TRUE;
	    }
	else
	    {
	    /*
	    **  Why did select return ??
	    */
	    dprintf("Number ready is zero ???\n");
	    }
	}

    cleanup_and_exit(0);

    return;
}


/*
**  DECODE_LINE - Take a raw input line and decode it, format
**		  the sender info and log it
*/
static void decode_line(
    char *raw_line,
    struct sockaddr_in *from,
    int from_len)
{
    int priority_int;
    int facility_int;
    int command_sent;
    int reopen_log;
    char *message_ch;
    char *from_host;

    dprintf("Got >%s<\n", raw_line);
    dprintf("From >%s<\n", inet_ntoa(from->sin_addr));

    /*
    **  Get the priority
    */
    if (*raw_line == '<')
	{
	char *priority_ch;
	char *end_ch;

	priority_ch = raw_line + 1;
	end_ch = priority_ch;
	while(*end_ch)
	    {
	    if (*end_ch == '>')
		{
		*end_ch = '\0';
		priority_int = atoi(priority_ch);
		message_ch = end_ch + 1;
		break;
		}
	    else if (isdigit(*end_ch))
		{
		end_ch++;
		}
	    else
		{
		/*
		**  Bogus message !!
		**  It starts with < but there's non digits before >
		*/
		priority_int = 0;
		message_ch = raw_line;
		break;
		}
	    }
	}
    else
	{
	priority_int = LOG_NOTICE;
	facility_int = LOG_USER;
	message_ch = raw_line;
	}

    /*
    **  Extract Command bits
    */
    command_sent = priority_int & 0x00FF0000;

    /*
    **  Convert the from address to something nice
    */
    if (from->sin_family == AF_INET)
	{
	struct hostent *hp;

	hp = gethostbyaddr(
		 (char *)&from->sin_addr,
		 sizeof(from->sin_addr),
		 from->sin_family);
	if (hp == NULL)
	    {
	    /*
	    **  Not known !!
	    */
	    from_host = unknown_str;
	    }
	else
	    {
	    from_host = hp->h_name;
	    }
	}
    else
	{
	/*
	**  Not an Internet style address ??
	*/
	from_host = invalid_addr_str;
	}

    /*
    **  See if this is a command
    */
    reopen_log = FALSE;
    if (command_sent)
	{
	int the_command;

	the_command = LOG_PRI(priority_int);

	/*
	**  Process command
	*/
	switch (the_command)
	    {
	    case 1:
		{
		/*
		**  Shutdown Command
		*/
		shutdown_requested = TRUE;
		break;
		}
	    case 2:
		{
		/*
		**  Reopen the Log
		*/
		reopen_log = TRUE;
		break;
		}
	    default:
		{
		/*
		**  A bogus command !!
		*/
		}
	    }

	/*
	**  Now force Priority to something meaningful
	*/
	priority_int = LOG_NOTICE;
	}
    else
	{
	/*
	**  Log the message
	*/
	logmsg(priority_int, message_ch, from_host, 0);
	}

    /*
    **  Reopen the log if we were told to
    */
    if (reopen_log)
	{
	struct filed *f;

	close_logs("Reopening log files");

	for (f = Files; f != NULL; f = f->f_next)
	    {
	    if (f->f_type == F_FILE)
		open_log(f, TRUE);
	    }
	}

    return;
}


/*
**  LOGERROR - Log an error message
*/
static void logerror(
    char *type)
{
	char buf[2048];

	if (errno)
		(void)sprintf(buf,
		    "syslogd: %s: %s", type, strerror(errno));
	else
		(void)sprintf(buf, "syslogd: %s", type);
	errno = 0;
	dprintf("%s\n", buf);
	logmsg(LOG_SYSLOG|LOG_ERR, buf, LocalHostName, ADDDATE);

    return;
}



/*
**  OPEN_LOG - Open a new or existing log file
*/
static void open_log(
    struct filed *f,
    int new_file)
{
    int open_options;
    char startup_msg[128];

    if (new_file)
	{
	dprintf("Reopening Log file %s\n",
	    f->f_un.f_fname);
	open_options = (O_WRONLY | O_TRUNC);
	}
    else
	{
	dprintf("Opening Log file %s\n",
	    f->f_un.f_fname);
	open_options = (O_WRONLY | O_APPEND | O_CREAT);
	}

    f->f_file = open(
		    f->f_un.f_fname, 
		    open_options, 
		    0,
		    "shr=get",
		    "rfm=var",
		    "rat=cr",
		    "mrs=0",
		    "mbf=2",
		    "alq=30",
		    "deq=30");

    if (f->f_file < 0)
	{
	/*
	**  Error opening log file !!
	*/
	printf("Log open returned %d\n", f->f_file);
	logerror("log file open failed");
	}

    /*
    **  Write a Startup message to the log file
    */
    sprintf(
	startup_msg, 
	"\n***************\n%15.15s %s syslog: starting up\n",
	current_time_str(),
	LocalHostName);
    if (write(f->f_file, startup_msg, strlen(startup_msg)) < 0)
	{
	logerror("Error writing to log file\n");
	};

    return;
}


/*
**  CURRENT_TIME_STR - Returns the current time as a NULL terminated str
**                     with NO new line character and NO Day of week
*/
static char *current_time_str(void)
{
    time_t	now;

    (void) time(&now);
    strcpy(time_str, ctime(&now));
    time_str[24] = '\0';

    return &time_str[4];
}


/*
**  FACILITY_TO_NAME - Convert a facility number to a name
*/
static char *facility_to_name(
    int facility_int)
{
    int i;
    char *return_ptr;

    for (i=0;
	((facilitynames[i].c_val >= 0)
	    && (facilitynames[i].c_val != facility_int));
	i++);

    if (facilitynames[i].c_val >= 0)
	{
	return_ptr = facilitynames[i].c_name;
	}
    else
	{
	return_ptr = unknown_str;
	}

    dprintf("Decoded facility %d as %s\n",
	facility_int,
	return_ptr);

    return return_ptr;
}


/*
**  PRIORITY_TO_NAME - Convert a priority number to a name
*/
static char *priority_to_name(
    int priority_int)
{
    int i;
    char *return_ptr;

    for (i=0;
	((prioritynames[i].c_val >= 0)
	    && (prioritynames[i].c_val != priority_int));
	i++);

    if (prioritynames[i].c_val >= 0)
	{
	return_ptr = prioritynames[i].c_name;
	}
    else
	{
	return_ptr = unknown_str;
	}

    dprintf("Decoded priority %d as %s\n",
	priority_int,
	return_ptr);

    return return_ptr;
}


/*
**  FLUSH_AND_MARK - Flush the log file, mark every 12 flushes (60 minutes)
*/
static void flush_and_mark(
    int signal_code)
{
    struct filed *f;
    time_t	now;

    dprintf("Flushing, count is %d\n", flush_count);
    flush_count++;

    /*
    **  If we have reached the mark point, log a mark
    */
    if (flush_count > 3)
	{
	logmsg(LOG_INFO, "-- MARK --", LocalHostName, ADDDATE|MARK);
	flush_count = 0;
	}

    /*
    **  Look for LOGS which have a pending repeated message
    */
    now = time((time_t *)NULL);
    for (f = Files; f; f = f->f_next)
	{
	if (f->f_prevcount && now >= REPEATTIME(f))
	    {
	    dprintf("flush %s: repeated %d times, %d sec.\n",
		TypeNames[f->f_type], f->f_prevcount,
		repeatinterval[f->f_repeatcount]);
	    fprintlog(f, 0, (char *)NULL);
	    BACKOFF(f);
	    }
	}

    /*
    **  Flush each file
    */
    for (f = Files; f != NULL; f = f->f_next)
	{
	if (f->f_type == F_FILE)
	    {
	    fsync(f->f_file);
	    }
	}

    /*
    **  Reset signal and timer
    */
    (void) signal(SIGALRM, flush_and_mark);
    (void) alarm(TIMERINTVL);

    return;
}


/*
**  CLEANUP_AND_EXIT - What do you think it does ??
*/
static void cleanup_and_exit(
    int signal_code)
{
    struct filed *f;

    dprintf("Clean Exit\n");

    /*
    **  Shutdown
    */
    close_logs("Shutting down");
    close (sd_syslog);

    /*
    **  If this was called from a signal, really exit
    */
    if (signal_code != 0)
	{
	exit(1);
	}

    return;
}


/*
**  CLOSE_LOGS - Close all log files
*/
static void close_logs(
    char *close_msg)
{
    struct filed *f;

    for (f = Files; f != NULL; f = f->f_next)
	{
	/*
	**  flush any pending output
	*/
	if (f->f_prevcount)
	    fprintlog(f, 0, close_msg);

	if (f->f_type == F_FILE)
	    {
	    (void)close(f->f_file);
	    }
	}

    return;
}

/*
 *  INIT -- Initialize syslogd from configuration table
 */
void init(
    int signo)
{
	int i;
	FILE *cf;
	struct filed *f, **nextp;
	char *p;
	char cline[LINE_MAX];

	dprintf("init\n");

	/*
	**  Close all open log files.
	*/
	close_logs("Reinitializing");

	/*
	**  Free the current filed linked list
	*/
	f = Files;
	while(f != NULL)
	    {
	    struct filed *dead_f;

	    dead_f = f;
	    f = f->f_next;
	    free(dead_f);
	    }

	Files = NULL;
	nextp = &Files;

	/* open the configuration file */
	if ((cf = fopen(ConfFile, "r")) == NULL) {
		dprintf("cannot open %s\n", ConfFile);
		*nextp = (struct filed *)calloc(1, sizeof(*f));
		cfline("*.ERR\t/sys$login:syslogd.log", *nextp);
		return;
	}

	/*
	 *  Foreach line in the conf table, open that file.
	 */
	f = NULL;
	while (fgets(cline, sizeof(cline), cf) != NULL) {
		/*
		 * check for end-of-section, comments, strip off trailing
		 * spaces and newline character.
		 */

		/*
		**  Position p to first nonspace character
		*/
		for (p = cline; isspace(*p); ++p)
			continue;
		/*
		**  If it's a comment or a NUL line, skip the line
		*/
		if (*p == '\0' || *p == '#')
			continue;

		/*
		**  Strip off trailing spaces
		*/
		for (p = strchr(cline, '\0'); isspace(*--p);)
			continue;
		*++p = '\0';

		f = (struct filed *)calloc(1, sizeof(*f));
		*nextp = f;
		nextp = &f->f_next;
		cfline(cline, f);
	}

	/* close the configuration file */
	(void)fclose(cf);


	if (debugging) {
		for (f = Files; f != NULL; f = f->f_next) {
			for (i = 0; i <= LOG_NFACILITIES; i++)
				if (f->f_pmask[i] == INTERNAL_NOPRI)
					printf("X ");
				else
					printf("%d ", f->f_pmask[i]);
			printf("%s: ", TypeNames[f->f_type]);
			switch (f->f_type) {
			case F_FILE:
			case F_TTY:
				printf("%s", f->f_un.f_fname);
				break;

			case F_CONSOLE:
				printf("%X", f->f_un.opcom_class_bm);
				break;

			case F_FORW:
				printf("%s", f->f_un.f_forw.f_hname);
				break;

			case F_USERS:
				for (i = 0; i < MAXUNAMES && *f->f_un.f_uname[i]; i++)
					printf("%s, ", f->f_un.f_uname[i]);
				break;
			}
			printf("\n");
		}
	}

	logmsg(LOG_SYSLOG|LOG_INFO, "syslogd: restart", LocalHostName, ADDDATE);
	dprintf("syslogd: restarted\n");

	return;
}

/*
** CFLINE - Crack a configuration file line
**          Configuration lines look like:
**          facility.severity <tab> destination
**
**          For example:
**          syslog.info	    @othernode
**          *.err	    /disk:[dir]name.log
**          local2.*	    MIKE,JOE
*/
void cfline(
    char *line,
    struct filed *f)
{
	struct hostent *hp;
	int i, pri;
	char *bp, *p, *q;
	char buf[MAXLINE], ebuf[100];

	dprintf("cfline(%s)\n", line);

	errno = 0;	/* keep strerror() stuff out of logerror messages */

	/* clear out file entry */
	memset(f, 0, sizeof(*f));
	for (i = 0; i <= LOG_NFACILITIES; i++)
		f->f_pmask[i] = INTERNAL_NOPRI;

	/* scan through the list of selectors */
	for (p = line; *p && *p != '\t';) {

		/* find the end of this facility name list */
		for (q = p; *q && *q != '\t' && *q++ != '.'; )
			continue;

		/* collect priority name */
		for (bp = buf; *q && !strchr("\t,;", *q); )
			*bp++ = *q++;
		*bp = '\0';

		/* skip cruft */
		while (strchr(", ;", *q))
			q++;

		/* decode priority name */
		if (*buf == '*')
			pri = LOG_PRIMASK + 1;
		else {
			pri = decode(buf, prioritynames);
			if (pri < 0) {
				(void)sprintf(ebuf,
				    "unknown priority name \"%s\"", buf);
				logerror(ebuf);
				return;
			}
		}

		/* scan facilities */
		while (*p && !strchr("\t.;", *p)) {
			for (bp = buf; *p && !strchr("\t,;.", *p); )
				*bp++ = *p++;
			*bp = '\0';
			if (*buf == '*')
				for (i = 0; i < LOG_NFACILITIES; i++)
					f->f_pmask[i] = pri;
			else {
				i = decode(buf, facilitynames);
				if (i < 0) {
					(void)sprintf(ebuf,
					    "unknown facility name \"%s\"",
					    buf);
					logerror(ebuf);
					return;
				}
				f->f_pmask[i >> 3] = pri;
			}
			while (*p == ',' || *p == ' ')
				p++;
		}

		p = q;
	}

	/*
	**  Skip to action part
	**  Selection and action are separated by at least one tab (or space)
	*/
	while ((*p == '\t') || (*p == ' '))
		p++;

	switch (*p)
	{
	case '@':
		(void)strcpy(f->f_un.f_forw.f_hname, ++p);
		hp = gethostbyname(p);
		if (hp == NULL) {
			logerror("Can't get hostname");
			break;
		}
		memset(&f->f_un.f_forw.f_addr, 0,
			 sizeof(f->f_un.f_forw.f_addr));
		f->f_un.f_forw.f_addr.sin_family = AF_INET;
		f->f_un.f_forw.f_addr.sin_port = syslog_port;
		memmove(&f->f_un.f_forw.f_addr.sin_addr, hp->h_addr, hp->h_length);
		f->f_type = F_FORW;
		break;

	case '/':
		f->f_type = F_FILE;

		/*
		**  Skip the / (this is OpenVMS, remember!)
		*/
		p++;

		/*
		**  Save the Filename
		*/
		(void)strcpy(f->f_un.f_fname, p);

		/*
		**  Open the File
		*/
		open_log(f, FALSE);

		break;

	case '%':
		/*
		**  OpenVMS OPCOM Classes
		*/
		f->f_type = F_CONSOLE;
		p++;
		f->f_un.opcom_class_bm = 0;
		while(*p)
		    {
		    char saved_q;

		    /*
		    **  Advance to end of class
		    */
		    for (q = p; *q && *q != ',' && *q != ' '; )
			    q++;

		    /*
		    **  Terminate and decode
		    */
		    saved_q = *q;
		    *q = '\0';
		    i = decode(p, opcom_classes);
		    dprintf("OPCOM Class %s decoded as %d\n", p, i);
		    if (i < 0)
			{
			char ebuf[80];
			(void)sprintf(ebuf,
			    "unknown OPCOM Class \"%s\"",
			    p);
			logerror(ebuf);
			}
		    else
			{
			f->f_un.opcom_class_bm |= i;
			}

		    *q = saved_q;
		    p = q;
		    while (*p == ',' || *p == ' ')
			    p++;
		    }
		break;
	case '*':
		f->f_type = F_WALL;
		break;

	default:
		for (i = 0; i < MAXUNAMES && *p; i++) {
			for (q = p; *q && *q != ','; )
				q++;
			(void)strncpy(f->f_un.f_uname[i], p, UT_NAMESIZE);
			if ((q - p) > UT_NAMESIZE)
				f->f_un.f_uname[i][UT_NAMESIZE] = '\0';
			else
				f->f_un.f_uname[i][q - p] = '\0';
			while (*q == ',' || *q == ' ')
				q++;
			p = q;
		}
		f->f_type = F_USERS;
		break;
	}

	return;
}



/*
 * Log a message to the appropriate log files, users, etc. based on
 * the priority.
 */
void logmsg(
    int pri,
    char *msg,
    char *from,
    int flags)
{
	struct filed *f;
	int fac, msglen, omask, prilev;
	char *timestamp;
	time_t now;

	dprintf("logmsg: pri %o, flags %x, from %s, msg %s\n",
	    pri, flags, from, msg);

	/*
	**  Block signals until we are done
	*/
	omask = sigblock(bitmask(SIGHUP)|bitmask(SIGALRM));

	/*
	** Check to see if msg starts with the date and time
	** If it doesn't have a date, set the ADDDATE option bit
	*/
	msglen = strlen(msg);
	if (msglen < 16 || msg[3] != ' ' || msg[6] != ' ' ||
	    msg[9] != ':' || msg[12] != ':' || msg[15] != ' ')
		flags |= ADDDATE;

	(void)time(&now);
	if (flags & ADDDATE)
		timestamp = ctime(&now) + 4;
	else {
		timestamp = msg;
		msg += 16;
		msglen -= 16;
	}

	/*
	**  See if the message ends with a newline,
	**  if it does we strip it off because we always add one
	*/
	if (msglen > 0)
	    {
	    if (msg[(msglen - 1)] == '\n')
		{
		msglen--;
		msg[msglen] = '\0';
		}
	    }

	/* extract facility and priority level */
	if (flags & MARK)
		fac = LOG_NFACILITIES;
	else
		fac = LOG_FAC(pri);
	prilev = LOG_PRI(pri);

	for (f = Files; f; f = f->f_next) {
		/* skip messages that are incorrect priority */
		if (f->f_pmask[fac] < prilev ||
		    f->f_pmask[fac] == INTERNAL_NOPRI)
			{
			dprintf("Priority skip fac:%d, %d < %d\n",
			    fac, f->f_pmask[fac], prilev);
			continue;
			}

		/*
		 * suppress duplicate lines to this file
		 */
		if ((flags & MARK) == 0 && msglen == f->f_prevlen &&
		    !strcmp(msg, f->f_prevline) &&
		    !strcmp(from, f->f_prevhost)) {
			(void)strncpy(f->f_lasttime, timestamp, 15);
			f->f_prevcount++;
			dprintf("msg repeated %d times, %ld sec of %d\n",
			    f->f_prevcount, now - f->f_time,
			    repeatinterval[f->f_repeatcount]);
			/*
			 * If domark would have logged this by now,
			 * flush it now (so we don't hold isolated messages),
			 * but back off so we'll flush less often
			 * in the future.
			 */
			if (now > REPEATTIME(f)) {
				fprintlog(f, flags, (char *)NULL);
				BACKOFF(f);
			}
		} else {
			/* new line, save it */
			if (f->f_prevcount)
				fprintlog(f, 0, (char *)NULL);
			f->f_repeatcount = 0;
			(void)strncpy(f->f_lasttime, timestamp, 15);
			(void)strncpy(f->f_prevhost, from,
					sizeof(f->f_prevhost));
			if (msglen < MAXSVLINE) {
				f->f_prevlen = msglen;
				f->f_prevpri = pri;
				(void)strcpy(f->f_prevline, msg);
				fprintlog(f, flags, (char *)NULL);
			} else {
				f->f_prevline[0] = 0;
				f->f_prevlen = 0;
				fprintlog(f, flags, msg);
			}
		}
	}
	(void)sigsetmask(omask);

    return;
}

void fprintlog(
    struct filed *f,
    int flags,
    char *msg)
{
	int l;
	char *msg_base;
	int msg_len;
	char full_message[MAXLINE + 100];
	int full_message_len;
	char repbuf[80];
	time_t now;


	/*
	**  Get the current time
	*/
	(void)time(&now);

	/*
	**  Figure out what the message should be, it could be"
	**           The passed message
	**        or "last message repeated n times"
	**        or the previous message
	**
	**        The passed message is used if we received a message
	**        which was too long to save in the previous message area
	**        and by internal routines which want to log specific
	**        messages (i.e. Shutting down now... etc.)
	*/
	if (msg) {
		msg_base = msg;
		msg_len = strlen(msg);
	} else if (f->f_prevcount > 1) {
		msg_base = repbuf;
		msg_len = sprintf(repbuf, "last message repeated %d times",
		    f->f_prevcount);
	} else {
		msg_base = f->f_prevline;
		msg_len = f->f_prevlen;
	}

	/*
	**  Format the Full message
	*/
	if (f->f_type == F_FORW)
	    {
	    /*
	    **  Machine format
	    */
	    full_message_len = sprintf(
		full_message, 
		"<%d>%.15s %s",
		f->f_prevpri,
		f->f_lasttime,
		msg_base);
	    }
	else if (f->f_type == F_FILE)
	    {
	    /*
	    **  Log File format
	    */
	    full_message_len = sprintf(
		full_message, 
		"%15.15s %s %s\n",
		f->f_lasttime,
		f->f_prevhost,
		msg_base);
	    }
	else
	    {
	    /*
	    **  Human Format
	    */
	    char msg_prefix[8];

	    if (f->f_type == F_USERS)
		{
		/*
		**  Prefix broadcast message with two new lines
		*/
		strcpy(msg_prefix, "\r\n\r\n\7");
		}
	    else
		{
		msg_prefix[0] = '\0';
		}

	    full_message_len = sprintf(
		full_message, 
		"%sMessage from syslogd@%s at %.15s ...\r\n%s\r\n",
		msg_prefix,
		f->f_prevhost,
		f->f_lasttime,
		msg_base);
	    }

	dprintf("Logging to %s", TypeNames[f->f_type]);
	f->f_time = now;

	switch (f->f_type) {
	case F_UNUSED:
		dprintf("\n");
		break;

	case F_FORW:
		dprintf(" %s\n", f->f_un.f_forw.f_hname);
		if (sendto(sd_syslog, full_message, full_message_len, 0,
		    (struct sockaddr *)&f->f_un.f_forw.f_addr,
		    sizeof(f->f_un.f_forw.f_addr)) != full_message_len) {
			f->f_type = F_UNUSED;
			logerror("sendto");
		}
		break;

	case F_CONSOLE:
		{
		struct
		    {
		    char type;
		    char target[3];
		    int rqstid;
		    char msg[2000];
		    } msgbuf;
		struct dsc$descriptor msgbuf_d;

		dprintf(" %X\n", f->f_un.opcom_class_bm);

		msgbuf.type = OPC$_RQ_RQST;
		memcpy(msgbuf.target, &f->f_un.opcom_class_bm, 3);
		msgbuf.rqstid = 0;
		memcpy(msgbuf.msg,
		       full_message,
		       full_message_len);
		msgbuf_d.dsc$w_length = full_message_len + 8;
		msgbuf_d.dsc$b_dtype = DSC$K_DTYPE_T;
		msgbuf_d.dsc$b_class = DSC$K_CLASS_S;
		msgbuf_d.dsc$a_pointer = (void *)&msgbuf;

		sys$sndopr (
		    &msgbuf_d, 
		    0);

		break;
		}

	case F_TTY:
	case F_FILE:
		dprintf(" %s\n", f->f_un.f_fname);

		if (write(f->f_file, full_message, full_message_len) < 0) {
			(void)close(f->f_file);
			f->f_type = F_UNUSED;
			logerror(f->f_un.f_fname);
			}
		else if (flags & SYNC_FILE)
			(void)fsync(f->f_file);
		break;

	case F_USERS:
		{
		int i;
		struct dsc$descriptor full_message_d;
		struct dsc$descriptor sendto_d;

		full_message_d.dsc$w_length = full_message_len;
		full_message_d.dsc$b_dtype = DSC$K_DTYPE_T;
		full_message_d.dsc$b_class = DSC$K_CLASS_S;
		full_message_d.dsc$a_pointer = full_message;

		for (i = 0; i < MAXUNAMES && *f->f_un.f_uname[i]; i++)
		    {
		    printf("%s, ", f->f_un.f_uname[i]);

		    sendto_d.dsc$w_length = strlen(f->f_un.f_uname[i]);
		    sendto_d.dsc$b_dtype = DSC$K_DTYPE_T;
		    sendto_d.dsc$b_class = DSC$K_CLASS_S;
		    sendto_d.dsc$a_pointer = f->f_un.f_uname[i];
		    sys$brkthru (
			0, 
			&full_message_d, 
			&sendto_d, 
			BRK$C_USERNAME, 
			0, 
			0, 
			BRK$M_CLUSTER, 
			BRK$C_QUEUE, 
			120,	    /* Time out in 2 minutes */
			0, 
			0);
		    }

		break;
		}
	case F_WALL:
		dprintf("\n");
		break;
	}
	f->f_prevcount = 0;

    return;
}

/*
 *  Decode a symbolic name to a numeric value
 */
int decode(
    const char *name,
    CODE *codetab)
{
	CODE *c;
	char *p, buf[40];

	if (isdigit(*name))
		return (atoi(name));

	for (p = buf; *name && p < &buf[sizeof(buf) - 1]; p++, name++) {
		if (isupper(*name))
			*p = tolower(*name);
		else
			*p = *name;
	}
	*p = '\0';
	for (c = codetab; c->c_name; c++)
		if (!strcmp(buf, c->c_name))
			return (c->c_val);

	return (-1);
}

