#pragma module pppdutil_kernel "X-8"
/*
 *****************************************************************************
 *
 * Copyright  1996 Digital Equipment Corporation.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation,
 * advertising materials, and other materials related to such
 * distribution and use acknowledge that the software was developed
 * by Digital Equipment Corporation.  The name of the
 * Corporation may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 *****************************************************************************


	FACILITY:

		PPPD

	ABSTRACT:

                This module contains the kernel mode code for the user interface
        to the Point to Point (PPPD) Utility.

	AUTHOR:

		Iris Langstein Realmuto 	21-December-1995

	REVISION HISTORY:

        X-8     BWK008          Barry W. Kierstein      17-DEC-1996
                Replaced the standard Digital copyright with
                one compatible with the CMU copyright.

        X-7     BWK007          Barry W. Kierstein      24-SEP-1996
                Standardized error handling with the rest of the
                utility.  Fixed possible problem that would have
                left PSWAPM on in lock_pages and unlock_pages.  Moved
                some function prototypes to PPPDUTIL_KERNEL.H that
                are shared with PPPDUTIL.C, and properly defined
                the shared prototypes so that the header file can
                be included by both PPPDUTIL.C and PPPDUTIL_KERNEL.C.

        X-6     BWK006          Barry W. Kierstein      16-JUL-1996
                Added shutdown_logger routine to shut down the trace
                logger process.  Added calls to this routine if the
                PPP management routines fail to start properly.

        X-5     BWK005          Barry W. Kierstein      11-JUL-1996
                Finished DIAL_OUT and HELP commands, fixed numerous bugs.
 
        X-4     BWK004          Barry W. Kierstein      14-JUN-1996
                Finished the SET and SHOW sections, fixed numerous bugs.

        X-3     BWK003          Barry W. Kierstein      20-MAY-1996
                Added code to ensure that elevated privileges are on only
                only as needed.

        X-2     BWK002          Barry W. Kierstein      14-MAY-1996
                Implemented initial SHOW command and fixed numerous bugs.

        X-1     BWK001          Barry W. Kierstein      12-APR-1996
                Initial check in of this module.
*/

#include <stdio.h>
#include <descrip.h>
#include <ssdef.h>
#include <iodef.h>
#include <prvdef.h>
#include <starlet.h>
#include <str$routines.h>
#include <splcoddef.h>
#include <vms_drivers.h>
#include <vms_macros.h>
#include "asndef.h"
#include "pppddef.h"
#include "ppp_lib.h"
#include "ppp_mgmt.h"
#include "pppdutil_common.h"
#include "pppdutil.h"
#include "pppdutil_kernel.h"

globaldef {"$$$CODE_BEGIN"} code_begin;
globaldef {"ZZZ_CODE_END"} code_end;

globaldef {"$$$LINKAGE_BEGIN"} linkage_section_begin;
globaldef {"ZZZ_LINKAGE_END"} linkage_section_end;

PROC_STATIC void set_ppp_failure(DEVICE_REC *current_device, lineId line_id);
PROC_STATIC void unbind_asn_device(DEVICE_REC *current_device);
PROC_STATIC KERNEL_RETURN_STATUS_TYPE *lock_addr_range(char *beg_addr, char *end_addr);
PROC_STATIC void shutdown_logger(DEVICE_REC *current_device);
PROC_STATIC KERNEL_RETURN_STATUS_TYPE *unlock_addr_range(char *beg_addr, char *end_addr);

/* Vector of PPP routines */
VAR_STATIC mgmtRtns *mgmtFns;

/* Space allocated for device name */
VAR_STATIC char edited_device_name_buffer[PPPD$K_MAX_NAME];

/*
** Function Name
**
**    disconnect_ppp_link
**
** Functional Description
**
**    This routine disconnects a PPP link.
**
**    This routine is in a separate module of routines that need to be
**    locked down in memory to function correctly at elevated IPL.
**
** Environment
**
**    Kernel mode, IPL 0 and IPL 8
**    Code and data pages are assumed to be locked down upon entry.
**
** Inputs
**
**    current_device - ptr to the current device in the device list
**
** Outputs
**
**    None
**
** Returns
**
**    Kernel return status structure address
**
*/
KERNEL_RETURN_STATUS_TYPE
    *disconnect_ppp_link(DEVICE_REC *current_device, PRIV *mask, PRIV *prev_priv)
{
lineId line_id;
int saved_fipl;
PRIV local_mask = *mask;
PRIV local_prev_priv = *prev_priv;
VMS_STATUS status;
$DESCRIPTOR(edited_device_name,edited_device_name_buffer);

    STORE_DEBUG_LONGWORD(5,1,edited_device_name.dsc$w_length);

    /* Remove the CMKRNL privilege if it was not there before */
    REMOVE_PRIV(local_prev_priv.prv$v_cmkrnl,local_mask,local_prev_priv,status)
    STORE_DEBUG_LONGWORD(5,2,status);
    KERNEL_CHECK_STATUS ("disconnect_ppp_link (REMOVE_PRIV status)",
                          PPPD$_NOPPPACCESS, status);

    /* Store the device name in our locked-down memory before going to IPL 8 */
    status = str$copy_dx(&edited_device_name,
                         &current_device->edited_device_name);
    STORE_DEBUG_LONGWORD(5,3,status);
    KERNEL_CHECK_STATUS ("disconnect_ppp_link (str$copy_dx status)",
                          PPPD$_NOPPPACCESS, status);

    /* Get the fork lock for IOLOCK8 */
    fork_lock(SPL$C_IOLOCK8,&saved_fipl);
    STORE_DEBUG_LOCATION(5,4);

    /* Get the line id for the current device */
    status = (*mgmtFns->getLineId)(&line_id,
                           edited_device_name.dsc$a_pointer,
                           edited_device_name.dsc$w_length);
    STORE_DEBUG_LONGWORD(5,5,status);
    STORE_DEBUG_LONGWORD(5,5,line_id);

    if FAILURE(status)
    {
        /* release fork */
        STORE_DEBUG_LOCATION(5,6);
        fork_unlock(SPL$C_IOLOCK8,saved_fipl,SMP_RESTORE);
        KERNEL_ERROR_RETURN ("disconnect_ppp_link (mgmtFns->getLineId status)",
                              PPPD$_PPPCONNECTERR, status);
    }

    STORE_DEBUG_LOCATION(5,7);

    /* Disable the current line */
    status = (*mgmtFns->disableLine)(line_id);
    STORE_DEBUG_LONGWORD(5,8,status);

    /* Delete the connection */
    status = (*mgmtFns->deleteLine)(line_id);
    STORE_DEBUG_LONGWORD(5,9,status);

    /* release the fork lock */
    fork_unlock(SPL$C_IOLOCK8,saved_fipl,SMP_RESTORE);
    STORE_DEBUG_LOCATION(5,10);

    KERNEL_NORMAL_RETURN;
}
/*
** Function Name
**
**    get_ppp_items
**
** Functional Description
**
**    This routine will query for the PPP link's attributes for
**    the PPP device.
**
**    This routine is in a separate module of routines that need to be
**    locked down in memory to function correctly at elevated IPL.
**
** Environment
**
**    Kernel mode, IPL 0 and IPL 8
**    Code and data pages are assumed to be locked down upon entry.
**
** Inputs
**
**    current_device - ptr to the current device in the device list
**    mask - ptr to the privilege mask
**    prev_priv - ptr to the previous privilege set
**
** Outputs
**
**    current_device->ppp_items - the skeleton itemlist will be
**                                filled in by the PPP mgmt routines
**
** Returns
**
**    Kernel return status structure address
**
*/
KERNEL_RETURN_STATUS_TYPE *get_ppp_items(DEVICE_REC *current_device, PRIV *mask, PRIV *prev_priv)
{
lineId line_id;
VMS_STATUS status = SS$_NORMAL;
u_int qual;
int size;
int saved_fipl;
PRIV local_mask = *mask;
PRIV local_prev_priv = *prev_priv;
$DESCRIPTOR(edited_device_name,edited_device_name_buffer);
KERNEL_RETURN_STATUS_TYPE *kernel_status;

    STORE_DEBUG_LONGWORD(6,1,edited_device_name.dsc$w_length);

    /* Remove the CMKRNL privilege if it was not there before */
    REMOVE_PRIV(local_prev_priv.prv$v_cmkrnl,local_mask,local_prev_priv,status)
    STORE_DEBUG_LONGWORD(6,2,status);
    KERNEL_CHECK_STATUS ("get_ppp_items (REMOVE_PRIV status)",
                          PPPD$_NOPPPACCESS, status);

    /* Store the device name in our locked-down memory before going to IPL 8 */
    edited_device_name.dsc$w_length     =
        current_device->edited_device_name.dsc$w_length;
    status = str$copy_dx(&edited_device_name,
                         &current_device->edited_device_name);
    STORE_DEBUG_LONGWORD(6,3,status);
    KERNEL_CHECK_STATUS ("get_ppp_items (str$copy_dx status)",
                          PPPD$_NOPPPACCESS, status);

    /* Lock down the PPP itemlist */
    kernel_status = lock_addr_range((char *)current_device->ppp_items,
                        (char *)current_device->ppp_items+current_device->ppp_items->item_length);
    STORE_DEBUG_LONGWORD(6,4,kernel_status->status);
    KERNEL_CHECK_STATUS ("get_ppp_items (lock_addr_range status)",
                          PPPD$_NOPPPACCESS, kernel_status->status);

    /* get the fork lock for IOLOCK8 */
    STORE_DEBUG_LOCATION(6,5);
    fork_lock(SPL$C_IOLOCK8,&saved_fipl);

    /* Get the line ID for the current device */
    STORE_DEBUG_LOCATION(6,6);
    status = (*mgmtFns->getLineId)(&line_id,
                            edited_device_name.dsc$a_pointer,
                            edited_device_name.dsc$w_length);

    STORE_DEBUG_LONGWORD(6,7,status);
    if FAILURE(status)
    {
        /* release the fork lock */
        STORE_DEBUG_LOCATION(6,8);
        fork_unlock(SPL$C_IOLOCK8,saved_fipl,SMP_RESTORE);
        unlock_addr_range((char *)current_device->ppp_items,
                          (char *)current_device->ppp_items+current_device->ppp_items->item_length);
        KERNEL_ERROR_RETURN ("get_ppp_items (mgmtFns->getLineId status)",
                              PPPD$_PPPCONNECTERR, status);
    }

    /* Get the PPP items */
    status = (*mgmtFns->showLine)(line_id,
                                (ItemList *)current_device->ppp_items,&qual);

    STORE_DEBUG_LONGWORD(6,9,status);
    STORE_DEBUG_LONGWORD(6,9,qual);
    if FAILURE(status)
    {
        /* release the fork lock */
        STORE_DEBUG_LOCATION(6,10);
        fork_unlock(SPL$C_IOLOCK8,saved_fipl,SMP_RESTORE);
        unlock_addr_range((char *)current_device->ppp_items,
                          (char *)current_device->ppp_items+current_device->ppp_items->item_length);
        KERNEL_ERROR_RETURN ("get_ppp_items (mgmtFns->showLine status)",
                              PPPD$_PPPINFOERR, status);
    }

    /* release the fork lock */
    fork_unlock(SPL$C_IOLOCK8,saved_fipl,SMP_RESTORE);
    STORE_DEBUG_LOCATION(6,11);

    /* Unlock the PPP itemlist */
    STORE_DEBUG_LOCATION(6,12);
    unlock_addr_range((char *)current_device->ppp_items,
                      (char *)current_device->ppp_items+current_device->ppp_items->item_length);

    KERNEL_NORMAL_RETURN;
}
/*
** Function Name
**
**    init_ppp_mgmt_call_vector
**
** Functional Description
**
**    This routine will set up the call vector for the PPP management routines.
**
**    This routine is in a separate module of routines that need to be
**    locked down in memory to function correctly at elevated IPL.
**
** Environment
**
**    Kernel mode, IPL 0 and IPL 2
**
** Inputs
**
**    None
**
** Outputs
**
**    None
**
** Returns
**
**    Returns NULL on failure
**    Address of the vector if successful
*/
mgmtRtns *init_ppp_mgmt_call_vector(void)
{
int saved_ipl;

    /* raise IPL to 2 */
    STORE_DEBUG_LOCATION(4,2);
    dsbint(IPL$_ASTDEL,saved_ipl);

    /* Get the address of the PPP management call vector */
    STORE_DEBUG_LOCATION(4,3);
    mgmtFns = (mgmtRtns *)PPPD$LIBRTNS(PPP_MGMT_RTNS);

    STORE_DEBUG_LONGWORD(4,4,(long int)mgmtFns);

    /* lower IPL back to 0 */
    enbint(saved_ipl);
    STORE_DEBUG_LOCATION(4,5);

    return mgmtFns;
}
/*
** Function Name
**
**    lock_addr_range
**
** Functional Description
**
**    This routine is called to lock down passed in address parameters
**    the pages before entering kernel mode.
**
** Environment
**
**    Kernel mode, IPL 0
**
** Inputs
**
**    None
**
** Outputs
**
**    None
**
** Returns
**
**    Kernel return status structure address
**
*/
KERNEL_RETURN_STATUS_TYPE *lock_addr_range(char *beg_addr, char *end_addr)
{
VMS_STATUS status, status2;
char *lock_buf_in[2];
char *lock_buf_ret[2];
PRIV prev_priv;
PRIV mask;

    STORE_DEBUG_LONGWORD(7,1,(long) beg_addr);
    STORE_DEBUG_LONGWORD(7,1,(long) end_addr);

    /* set the privilege to lock pages */
    SET_PRIV(mask.prv$v_pswapm,mask,prev_priv,status)
    KERNEL_CHECK_STATUS(NULL,0L,status);

    /* lock down the code pages */
    lock_buf_in[0]      = beg_addr;
    lock_buf_in[1]      = end_addr;
    status2 = sys$lckpag(lock_buf_in,         /* inadr */
                         lock_buf_ret,        /* retadr */
                         0);                  /* acmode */
    if (FAILURE(status2))
    {
        REMOVE_PRIV(prev_priv.prv$v_pswapm,mask,prev_priv,status);
        KERNEL_ERROR_RETURN(NULL,0L,status2);
    };

    /* remove the privilege if it was not there before */
    REMOVE_PRIV(prev_priv.prv$v_pswapm,mask,prev_priv,status)
    KERNEL_CHECK_STATUS(NULL,0L,status);

    KERNEL_NORMAL_RETURN;
}
/*
** Function Name
**
**    lock_pages
**
** Functional Description
**
**    This routine is called to lock down this module's pages before entering
**    kernel mode.
**
** Environment
**
**    Kernel mode, IPL 0
**
** Inputs
**
**    None
**
** Outputs
**
**    None
**
** Returns
**
**    Kernel return status structure address
**
*/
KERNEL_RETURN_STATUS_TYPE *lock_pages(void)
{
VMS_STATUS status, status2;
int *lock_code_buf_in[2] = {&code_begin, &code_end};
int *lock_linkage_section_buf_in[2] = {&linkage_section_begin,
                                       &linkage_section_end};
int *lock_code_buf_ret[2];
int *lock_linkage_section_buf_ret[2];
PRIV prev_priv;
PRIV mask;

    /* set the privilege to lock pages */
    SET_PRIV(mask.prv$v_pswapm,mask,prev_priv,status)
    KERNEL_CHECK_STATUS(NULL,0L,status);

    /* lock down the code pages */
    status2 = sys$lckpag(lock_code_buf_in,         /* inadr */
                         lock_code_buf_ret,        /* retadr */
                         0);                       /* acmode */
    if (FAILURE(status2))
    {
        REMOVE_PRIV(prev_priv.prv$v_pswapm,mask,prev_priv,status);
        KERNEL_ERROR_RETURN(NULL,0L,status2);
    };

    /* lock down the linkage section */
    status2 = sys$lckpag(lock_linkage_section_buf_in,  /* inadr */
                         lock_linkage_section_buf_ret, /* retadr */
                         0);                           /* acmode */
    if (FAILURE(status2))
    {
        REMOVE_PRIV(prev_priv.prv$v_pswapm,mask,prev_priv,status);
        KERNEL_ERROR_RETURN(NULL,0L,status2);
    };

    /* remove the privilege if it was not there before */
    REMOVE_PRIV(prev_priv.prv$v_pswapm,mask,prev_priv,status)
    KERNEL_CHECK_STATUS(NULL,0L,status);

    KERNEL_NORMAL_RETURN;
}
/*
** Function Name
**
**    set_ppp_failure
**
** Functional Description
**
**    This routine cleans up the PPP link is there is any problem
**    creating/setting the line.
**
**    This routine is in a separate module of routines that need to be
**    locked down in memory to function correctly at elevated IPL.
**
** Environment
**
**    Kernel mode, IPL 0 and IPL 8
**    Code and data pages are assumed to be locked down upon entry.
**
** Inputs
**
**    current_device - ptr to the current device in the device list
**    line_id - line id of the PPP line being manipulated
**
** Outputs
**
**    None
**
** Returns
**
**    None
**
*/
PROC_STATIC void set_ppp_failure(DEVICE_REC *current_device, lineId line_id)
{
int saved_fipl;

    /* get the fork lock for IOLOCK8 even though it may already be set */
    fork_lock(SPL$C_IOLOCK8,&saved_fipl);

    /* Disable the current line */
    (*mgmtFns->disableLine)(line_id);

    /* Delete the connection */
    (*mgmtFns->deleteLine)(line_id);

    /* release the fork lock */
    fork_unlock(SPL$C_IOLOCK8,saved_fipl,SMP_RESTORE);

    /* Unbind the ASN device from the physical device */
    unbind_asn_device(current_device);

}
/*
** Function Name
**
**    set_ppp_items
**
** Functional Description
**
**    This routine create the PPP link and sets the attributes for
**    the PPP device.
**
**    This routine is in a separate module of routines that need to be
**    locked down in memory to function correctly at elevated IPL.
**
** Environment
**
**    Kernel mode, IPL 0 and IPL 8
**    Code and data pages are assumed to be locked down upon entry.
**
** Inputs
**
**    current_device - ptr to the current device in the device list
**    mask - ptr to the privilege mask
**    prev_priv - ptr to the previous privilege set
**
** Outputs
**
**    None
**
** Returns
**
**    Kernel return status structure address
**
*/
KERNEL_RETURN_STATUS_TYPE *set_ppp_items(DEVICE_REC *current_device, PRIV *mask, PRIV *prev_priv)
{
lineId line_id;
VMS_STATUS status = SS$_NORMAL;
u_int qual;
int size;
int saved_fipl;
PRIV local_mask = *mask;
PRIV local_prev_priv = *prev_priv;
$DESCRIPTOR(edited_device_name,edited_device_name_buffer);
KERNEL_RETURN_STATUS_TYPE *kernel_status;

    STORE_DEBUG_LONGWORD(3,1,edited_device_name.dsc$w_length);

    /* Remove the CMKRNL privilege if it was not there before */
    REMOVE_PRIV(local_prev_priv.prv$v_cmkrnl,local_mask,local_prev_priv,status)
    STORE_DEBUG_LONGWORD(3,2,status);
    KERNEL_CHECK_STATUS("set_ppp_items (REMOVE_PRIV status)",
                         PPPD$_NOPPPACCESS, status);

    /* Store the device name in our locked-down memory before going to IPL 8 */
    edited_device_name.dsc$w_length     =
        current_device->edited_device_name.dsc$w_length;
    status = str$copy_dx(&edited_device_name,
                         &current_device->edited_device_name);
    STORE_DEBUG_LONGWORD(3,3,status);
    KERNEL_CHECK_STATUS("set_ppp_items (str$copy_dx status)",
                         PPPD$_NOPPPACCESS, status);

    /* Lock down the PPP itemlist */
    kernel_status = lock_addr_range((char *)current_device->ppp_items,
                        (char *)current_device->ppp_items+current_device->ppp_items->item_length);
    STORE_DEBUG_LONGWORD(3,4,kernel_status->status);
    KERNEL_CHECK_STATUS ("set_ppp_items (lock_addr_range status)",
                          PPPD$_NOPPPACCESS, kernel_status->status);

    /* get the fork lock for IOLOCK8 */
    STORE_DEBUG_LOCATION(3,5);
    fork_lock(SPL$C_IOLOCK8,&saved_fipl);

    /* Create the line connect for the current device */
    STORE_DEBUG_LOCATION(3,6);
    status = (*mgmtFns->createLine)(&line_id,
                            edited_device_name.dsc$a_pointer,
                            edited_device_name.dsc$w_length);

    STORE_DEBUG_LONGWORD(3,7,status);
    STORE_DEBUG_LONGWORD(3,7,line_id);
    if FAILURE(status)
    {
        /* release the fork lock */
        STORE_DEBUG_LOCATION(3,8);
        fork_unlock(SPL$C_IOLOCK8,saved_fipl,SMP_RESTORE);
        shutdown_logger(current_device);
        unlock_addr_range((char *)current_device->ppp_items,
                          (char *)current_device->ppp_items+current_device->ppp_items->item_length);
        KERNEL_ERROR_RETURN("set_ppp_items (mgmtFns->createLine status)",
                             PPPD$_PPPCREATEERR, status);
    }

    /* Set the PPP items */
    status = (*mgmtFns->setLine)(line_id,
                                (ItemList *)current_device->ppp_items,&qual);

    STORE_DEBUG_LONGWORD(3,9,status);
    STORE_DEBUG_LONGWORD(3,9,qual);
    STORE_DEBUG_LONGWORD(3,9,line_id);
    if FAILURE(status)
    {
        /* release the fork lock */
        STORE_DEBUG_LOCATION(3,10);
        fork_unlock(SPL$C_IOLOCK8,saved_fipl,SMP_RESTORE);
        set_ppp_failure(current_device,line_id);
        shutdown_logger(current_device);
        unlock_addr_range((char *)current_device->ppp_items,
                          (char *)current_device->ppp_items+current_device->ppp_items->item_length);
        KERNEL_ERROR_RETURN("set_ppp_items (mgmtFns->setLine status)",
                             PPPD$_PPPSETERR, status);
    }

    /* Start the line going */
    status = (*mgmtFns->enableLine)(line_id);

    STORE_DEBUG_LONGWORD(3,11,status);
    STORE_DEBUG_LONGWORD(3,11,line_id);
    if FAILURE(status)
    {
        /* release the fork lock */
        STORE_DEBUG_LOCATION(3,12);
        fork_unlock(SPL$C_IOLOCK8,saved_fipl,SMP_RESTORE);
        set_ppp_failure(current_device,line_id);
        shutdown_logger(current_device);
        unlock_addr_range((char *)current_device->ppp_items,
                          (char *)current_device->ppp_items+current_device->ppp_items->item_length);
        KERNEL_ERROR_RETURN("set_ppp_items (mgmtFns->enableLine status)",
                             PPPD$_PPPENABLEERR, status);
    }

    /* release the fork lock */
    fork_unlock(SPL$C_IOLOCK8,saved_fipl,SMP_RESTORE);
    STORE_DEBUG_LOCATION(3,13);

    /* Unlock the PPP itemlist */
    unlock_addr_range((char *)current_device->ppp_items,
                      (char *)current_device->ppp_items+current_device->ppp_items->item_length);
    STORE_DEBUG_LOCATION(3,14);

    /* Execute the network callback routine if defined */
    if (current_device->network_callback != NULL)
    {
        status = current_device->network_callback(&edited_device_name);
        STORE_DEBUG_LONGWORD(3,15,status);
        if FAILURE(status)
        {
            STORE_DEBUG_LOCATION(3,16);
            set_ppp_failure(current_device,line_id);
            KERNEL_ERROR_RETURN("set_ppp_items (network_callback status)",
                                 PPPD$_CALLBACKERR, status);
        }
    }

    STORE_DEBUG_LOCATION(3,17);
    KERNEL_NORMAL_RETURN;
}
/*
** Function Name
**
**    shutdown_logger
**
** Functional Description
**
**    This routine will write an error record and an EOF record to the
**    trace logger mailbox, wait a bit, then delete the mailbox just
**    in case the logger process is gone or stuck somehow.
**
** Environment
**
**    Kernel mode, IPL 0
**
** Inputs
**
**    current_device - pointer to the current device in the device list
**
** Outputs
**
**    None
**
** Returns
**
**    None
**
*/
PROC_STATIC void shutdown_logger(DEVICE_REC *current_device)
{
static     char error_msg_text[] = "Unable to start the PPP management routines";
static     char eof_msg_text[]   = "Device closing";
logRecord  error_msg =
           { 0, LOG_ERR, sizeof(error_msg_text)-1, ' '},
           eof_msg =
           { 0, LOG_EOF, sizeof(eof_msg_text)-1, ' '};
VMS_STATUS status;
VMS_TIME timestamp;
CHAN channel;
static IOSB iosb1, iosb2;

    /* If trace was not set, then just return */
    STORE_DEBUG_LONGWORD(9,1,current_device->trace);
    if (!current_device->trace)
        return;

    /* Get the system time for the timestamp */
    status = sys$gettim (timestamp);
    STORE_DEBUG_LONGWORD(9,2,status);
    if FAILURE(status)
        return;

    /* Assign a channel to the mailbox.  If this fails, then just return. */
    status = sys$assign (&current_device->trace_mbx,  /* device name */
                         &channel,                    /* channel number */
                         0,                           /* acmode */
                         0,                           /* mbxnam */
                         0);                          /* flags */
    STORE_DEBUG_LONGWORD(9,3,status);
    STORE_DEBUG_LONGWORD(9,3,channel);
    if FAILURE(status)
        return;

    /* Finish initializing the error message records */
    memcpy (&error_msg.timeStamp, &timestamp, sizeof(VMS_TIME));
    memcpy (&eof_msg.timeStamp,   &timestamp, sizeof(VMS_TIME));

    memcpy (error_msg.msg, error_msg_text, error_msg.msgLen);
    memcpy (eof_msg.msg,   eof_msg_text,   eof_msg.msgLen);

    /* Write out the messages to the mailbox */
    status  =
        sys$qio  (PPPD_EFN,                    /* EFN */
                  channel,                     /* I/O channel */
                  IO$_WRITEVBLK|IO$M_NOW
                               |IO$M_NORSWAIT, /* Function code */
                  &iosb1,                      /* IOSB block */
                  0L,                          /* AST address */
                  0L,                          /* AST param */
                  &error_msg,                  /* P1 - data buffer address */
                  sizeof(logRecord),           /* P2 - data buffer size */
                  0L,                          /* P3 - timeout count */
                  0L,                          /* P4 - read term. desc. block address */
                  0L,                          /* P5 - prompt buffer address */
                  0L);                         /* P6 - prompt buffer size */
    STORE_DEBUG_LONGWORD(9,4,status);

    if SUCCESS(status)
        status  =
            sys$qio  (PPPD_EFN,                    /* EFN */
                      channel,                     /* I/O channel */
                      IO$_WRITEVBLK|IO$M_NOW
                                   |IO$M_NORSWAIT, /* Function code */
                      &iosb2,                      /* IOSB block */
                      0L,                          /* AST address */
                      0L,                          /* AST param */
                      &eof_msg,                    /* P1 - data buffer address */
                      sizeof(logRecord),           /* P2 - data buffer size */
                      0L,                          /* P3 - timeout count */
                      0L,                          /* P4 - read term. desc. block address */
                      0L,                          /* P5 - prompt buffer address */
                      0L);                         /* P6 - prompt buffer size */
    STORE_DEBUG_LONGWORD(9,5,status);

    /* Mark the mailbox for delete */
    sys$delmbx (channel);
    STORE_DEBUG_LOCATION(9,6);

    /* Deassign the mailbox */
    sys$dassgn (channel);
    STORE_DEBUG_LOCATION(9,7);

    return;
}
/*
** Function Name
**
**    unbind_asn_device
**
** Functional Description
**
**    This routine unbinds the asynch device to the physical device
**
** Environment
**
**    Kernel mode, IPL 0
**
** Inputs
**
**    current_device - pointer to the current device in the device list
**
** Outputs
**
**    None
**
** Returns
**
**    None
**
*/
PROC_STATIC void unbind_asn_device(DEVICE_REC *current_device)
{
IOSB iosb;
VMS_ITEMLIST items[2];

    VMS_ITEMLIST_INIT(items)
    VMS_ITEMLIST_ADDR(ASN$_DISCONNECT,ppp_str_len,
                      current_device->device_name.dsc$a_pointer,
                     &current_device->device_name.dsc$w_length)
    VMS_ITEMLIST_TERM

    sys$qiow(PPPD_EFN,                      /* EFN */
             current_device->asn_channel,   /* chan */
             IO$_SETMODE,                   /* func */
             &iosb,                         /* iosb */
             0,                             /* astadr */
             0,                             /* astprm */
             &items,                        /* p1 */
             sizeof(items),                 /* p2 */
             0, 0, 0, 0);                   /* p3 - p6 */

}
/*
** Function Name
**
**    unlock_addr_range
**
** Functional Description
**
**    This routine is called to unlock down the pages upon exiting
**    kernel mode.
**
** Environment
**
**    Kernel mode, IPL 0
**
** Inputs
**
**    None
**
** Outputs
**
**    None
**
** Returns
**
**    Kernel return status structure address
**
*/
KERNEL_RETURN_STATUS_TYPE *unlock_addr_range(char *beg_addr, char *end_addr)
{
VMS_STATUS status, status2;
char *lock_buf_in[2];
char *lock_buf_ret[2];
PRIV prev_priv;
PRIV mask;

    STORE_DEBUG_LONGWORD(8,1,(long) beg_addr);
    STORE_DEBUG_LONGWORD(8,1,(long) end_addr);

    /* set the privilege to unlock pages */
    SET_PRIV(mask.prv$v_pswapm,mask,prev_priv,status);
    KERNEL_CHECK_STATUS(NULL,0L,status);

    /* unlock the address range */
    lock_buf_in[0]      = beg_addr;
    lock_buf_in[1]      = end_addr;
    status2 = sys$ulkpag(lock_buf_in,         /* inadr */
                         lock_buf_ret,        /* retadr */
                         0);                  /* acmode */
    if (FAILURE(status2))
    {
        REMOVE_PRIV(prev_priv.prv$v_pswapm,mask,prev_priv,status);
        KERNEL_ERROR_RETURN(NULL,0L,status2);
    };

    /* remove the privilege if it was not there before */
    REMOVE_PRIV(prev_priv.prv$v_pswapm,mask,prev_priv,status);
    KERNEL_CHECK_STATUS(NULL,0L,status);

    KERNEL_NORMAL_RETURN;
}
/*
** Function Name
**
**    unlock_pages
**
** Functional Description
**
**    This routine is called to unlock down the pages upon exiting
**    kernel mode.
**
** Environment
**
**    Kernel mode, IPL 0
**
** Inputs
**
**    None
**
** Outputs
**
**    None
**
** Returns
**
**    Kernel return status structure address
**
*/
KERNEL_RETURN_STATUS_TYPE *unlock_pages(void)
{
VMS_STATUS status, status2;
int *lock_code_buf_in[2] = {&code_begin, &code_end};
int *lock_linkage_section_buf_in[2] = {&linkage_section_begin,
                                       &linkage_section_end};
int *lock_code_buf_ret[2];
int *lock_linkage_section_buf_ret[2];
PRIV prev_priv;
PRIV mask;

    /* set the privilege to unlock pages */
    SET_PRIV(mask.prv$v_pswapm,mask,prev_priv,status);
    KERNEL_CHECK_STATUS(NULL,0L,status);

    /* unlock the code pages */
    status2 = sys$ulkpag(lock_code_buf_in,         /* inadr */
                         lock_code_buf_ret,        /* retadr */
                         0);                       /* acmode */
    if (FAILURE(status2))
    {
        REMOVE_PRIV(prev_priv.prv$v_pswapm,mask,prev_priv,status);
        KERNEL_ERROR_RETURN(NULL,0L,status2);
    };

    /* unlock the linkage section */
    status2 = sys$ulkpag(lock_linkage_section_buf_in,  /* inadr */
                         lock_linkage_section_buf_ret, /* retadr */
                         0);                           /* acmode */
    if (FAILURE(status2))
    {
        REMOVE_PRIV(prev_priv.prv$v_pswapm,mask,prev_priv,status);
        KERNEL_ERROR_RETURN(NULL,0L,status2);
    };

    /* remove the privilege if it was not there before */
    REMOVE_PRIV(prev_priv.prv$v_pswapm,mask,prev_priv,status);
    KERNEL_CHECK_STATUS(NULL,0L,status);

    KERNEL_NORMAL_RETURN;
}
