/* mailsym.c
This program is a symbiont started by the queue manager with the command:
$ init/queue/start/proc=mailsym my_queue
The mailsym.exe image should be put in sys$system.

When printing to my_queue the print file will be mailed to the user
who issued the print command. The idea is that you use the VMS mail
set forward command to forward your mail to your office mail account.
The print file will be seen as an attachment in your mailer.

There are some logicals used to manage the symbiont:
- MAILSYM_<queuename>_DEST
  If this logical is defined the mail is sent to the value of this logical instead of the user who issued the print command.
- MAILSYM_<queuename>_FILETYPE
  This logical specifies the file type (extension) of the attachment. Default is extension of the file printed.
- MAILSYM_<queuename>_FROM_ADDRESS
  This logical specifies the origin address for the mail, default is system@thishost'sname.
- MAILSYM_<queuename>_SUBJECT
  This logical specifies the subject. Default is the filename of the print command without the extension.
- MAILSYM_<queuename>_BODYTEXT. Without this logical the mail message only contains the attachment. With this logical you can also
  put text in the body. If the equivalence string starts with an '@' the rest is considered as a filename. This works like the
  VMS logical SYS$ANNOUNCE. Note that if the file doesn't exist the symbiont will exit with a RMS-E-FNF status.
- MAILSYM_<queuename>_OPCOM. If this logical is defined the symbiont will issue an operator message for each job done.
- MAILSYM_<queuename>_CONTENT_TYPE. This logical is used for sending binary files. The value must be BINARY. For sending plain text
  the value must be ASCII (this is the default).
- MAILSYM_DEBUG This logical enables debugging. This logical is not specific for the queue. Debugging information is written in 
  sys$system:mailsym.debug. The value is interpreted as a bit mask:
  bit 0 -> enables debugging of the symbiont
  bit 1 -> enables debugging of the mailing process (send from file)
All logicals may be defined in the system or cluster logical name table.
These logicals (except the debugging one) are evaluated for each print job. Therefore it is not needed to restart the queue after
changing these logicals.
The symbiont will also honor user parameters. All parameters must be specified in the form <PARAM>=<VALUE>. These parameters
overrule the corresponding logicals. The following parameters are supported.

DESTINATION_ADDDRESS=xxx This parameter specifies the destination address.
FROM_ADDRESS=xxx         This parameter specifies the from address. This parameter is only allowed if the user has OPER priv.
FILETYPE=xxx             This parameter specifies the file type the attachment will have.
SUBJECT=xxx              This parameter specifies the subject of the mail.
BODYTEXT=xxx             This parameter specifies the body text. If a file is specified (BODYTEXT=@FOO.BAR) a check is done on the
                         users privs to ensure the user has read permission on the file. Also if the file is not found this will
                         not terminate the symbiont. If there is a problem accessing the file the job will fail with the
                         corresponding error status.
CONTENT_TYPE=xxx         This parameter is used for sending binary files. In this case the value must be BINARY. The effect is
                         that the file will be base64 encoded. This can be used for sending zip files and the like. The value can
                         also be ASCII for sending plain text files (this can be useful if the content_type logical is defined
                         as binary.
Some examples:
Send a single ascii file:
$ print/queue=my_queue login.com/param=(bodytext="Here is my login.com file",dest="some.guy@some.company",filetype=txt)
In this example your login.com will be sent to some.guy. The login.com will be seen as an attachment of type txt (so windows won't
go into paranoia mode for seeing a .com file). In the message body of the mail the text "Here is my login.com file" will be seen.
The mail will have LOGIN as subject.
Send a bunch of files:
$ zip my_files.zip *.txt
$ print/queue=my_queue my_files.zip/param=(content=binary)
First the zip tool is used to create a zip archive. The mail is either delivered to your VMS mail account (not very useful) or if
you have enabled mail forwarding to some office mail account to the latter. Your e-mail client should know how to call zip to open
the attachment. The mail has no text in the message body (unless the logical mailsym_my_queue_bodytext is defined).
Build:
$ cc mailsym
$ link mailsym,sys$command/opt
sys$share:tcpip$smtp_mailshr/share
*EXIT*
Oswald Knoppers, December 2003 initial version
                 April    2004 added support for the logical names for destination and file type
                 May      2004 added support for from, subject and body logicals, added debugging option
                 July     2004 finally got rid of opcom message with QMAN-I-INVSMBMSG error, change process name to queue name
                               additional debugging options
                 October  2004 support for print/copies=xxx
                 February 2005 support for sending binary attachments, added opcom message, added parameters
                 March    2005 additional debugging, also send opcom message with failed jobs and include final status, added
                               content_type logical
*/

#pragma module MAILSYM "V2.0"
#include stdio
#include ssdef
#include rmsdef
#include descrip
#include "smbdef.h"
#include string
#include maildef
#include unistd
#include syidef
#include lnmdef
#include psldef
#include stdlib
#include opcdef
#include acldef
#include armdef
#include chpdef
#include uaidef
void ast();
void wrdebug();
int getlogical();
int parseparam();
void encode();
void encodeblock();
int SYS$HIBER();
int SYS$WAKE();
int SYS$EXIT();
int SYS$GETSYIW();
int SMB$INITIALIZE();
int SMB$READ_MESSAGE();
int SMB$READ_MESSAGE_ITEM();
int SMB$SEND_TO_JOBCTL();
int STR$COPY_DX();
int TCPIP$SMTP_SEND_FROM_FILE();
int SYS$TRNLNM();
int SYS$ASCTIM();
int SYS$SETPRN();
int SYS$SNDOPR();
int SYS$CHECK_ACCESS();
int SYS$GETUAI();
#define VMSUSER 1
#define LOGICAL 2
#define FILENAME 1
#define SYMBIONTDEBUG 1
#define MAILDEBUG 2
#define DEBUG(message, value) if (debug & SYMBIONTDEBUG) wrdebug(message, value)
$DESCRIPTOR (tabnam, "LNM$SYSTEM");
$DESCRIPTOR (lognam, "");
globalvalue SMB$_NOMOREITEMS;
FILE *debugfile;
unsigned short int debug, loglen, binaryflag, bodytext, userbodytext, paramerror, checkoper;
static long stream, request;
char logvalue[255], filetypelogical[255], destlogical[255], fromlogical[255], subjectlogical[255],bodytextlogical[255], 
    tmpfilename[255], opclogical[255], opcstring[255], maildest[255], fromaddress[255], subjectstring[255], bodytextstring[255],
    filetype[255], contentlogical[255], contentstring[255];
struct itmlst {unsigned short length; unsigned short code; void *address; void *retlenadr;}
    lnmlst[] = {{sizeof(logvalue), LNM$_STRING, &logvalue, &loglen}, {0,0,0,0}};
static struct dsc$descriptor_d filename = {0, DSC$K_DTYPE_T, DSC$K_CLASS_D, 0}, username = {0, DSC$K_DTYPE_T, DSC$K_CLASS_D, 0},
    queuename = {0, DSC$K_DTYPE_T, DSC$K_CLASS_D, 0}, userbodyfile = {0, DSC$K_DTYPE_T, DSC$K_CLASS_D, 0};
static const char cb64[]="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
main()
{
    FILE *infile, *tmpfile, *bodyfile;
    long structure_level = SMBMSG$K_STRUCTURE_LEVEL, streams = 1, job_status[2], status, check_access = ARM$M_READ;
    unsigned short subject, opcom;
    int sendto, length, i;
    char infilename[255], thissys[7], user[255], buf[132], opcbuf[256], defpriv[8], priv[8], userstat[10], *p, *q, inchar;
    char debuglogical[] = "MAILSYM_DEBUG", debugfilename[] = "SYS$SYSTEM:MAILSYM.DEBUG";
    struct itmlst syilst[] = {{6, SYI$_SCSNODE, &thissys, 0}, {0,0,0,0}},
        chkxsitm[] = {{4, CHP$_ACCESS, &check_access, 0},{0,0,0,0}},
        uaiitm[] = {{8, UAI$_DEF_PRIV, &defpriv, 0}, {8, UAI$_PRIV, &priv, 0}, {0,0,0,0}};
    $DESCRIPTOR(opcbuf_desc, opcbuf);
    struct dsc$descriptor req_desc;
    struct OPC opcrequest;
/* debugging? */
    lognam.dsc$a_pointer = debuglogical;
    lognam.dsc$w_length = strlen(debuglogical);
    status = SYS$TRNLNM(0, &tabnam, &lognam, 0, &lnmlst);
    if (status == SS$_NOLOGNAM) debug = 0;
    else {
        logvalue[loglen] = 0;
        debug = atoi(logvalue);
	if (debug & SYMBIONTDEBUG || debug & MAILDEBUG) debugfile = fopen(debugfilename, "w");
    }
/* get the name of this system */
    if (!((status = SYS$GETSYIW(0, 0, 0, syilst, 0, 0, 0))&1)) return status;
    thissys[6] = 0;
    if ((p = strstr(thissys, " ")) != NULL) *p = 0;
    if (debug & SYMBIONTDEBUG) {
        sprintf(buf, "Symbiont running on %s\n", thissys);
        fwrite(buf, 1, strlen(buf), debugfile);
    }

/* specify opcom message type */
    opcrequest.opc$b_ms_type = OPC$_RQ_RQST;
    opcrequest.opc$b_ms_target = OPC$M_NM_PRINT;
    opcrequest.opc$l_ms_rqstid = 0;
/* let the queue manager know we are there */
    DEBUG("main: calling smb$initialize", 0);
    if(!((status = SMB$INITIALIZE(&structure_level, ast, &streams))&1)) return status;
    DEBUG("main: smb$initialize done", 0);
    username.dsc$w_length = 0;
    while (1) {
/* reset to defaults */
        binaryflag = 0;
        maildest[0] = 0;
        fromaddress[0] = 0;
        checkoper = 0;
        subjectstring[0] = 0;
        bodytextstring[0] = 0;
        bodytext = 0;
        userbodytext = 0;
        filetype[0] = 0;
        paramerror = SS$_NORMAL;
/* wait for a command */
        DEBUG("main: hibernating", 0);
        if (!((status = SYS$HIBER())&1)) return status;
        DEBUG("main: woke up", 0);
/* let the queue manager know we started the job */
        request = SMBMSG$K_START_TASK;
        DEBUG("main: sending start task to queue manager", 0);
        if (!((status = SMB$SEND_TO_JOBCTL(&stream, &request, 0, 0, 0, 0))&1)) return status;
        DEBUG("main: done", 0);
        job_status[1] = paramerror;
/* if the user has specified a from address he needs to have oper priv */
        if (username.dsc$w_length > 0 && paramerror == SS$_NORMAL && checkoper) {
            DEBUG("main: checking OPER priv for user", 0);
            defpriv[2] = 0;
            priv[2] = 0;
            if (!((status = SYS$GETUAI(0, 0, &username, uaiitm, 0, 0, 0))&1)) SYS$EXIT(status);
            if (!((defpriv[2] & 0x4) || (priv[2] & 0x4))) {
                paramerror = SS$_NOPRIV;
                DEBUG("main: user has no OPER priv", 0);
                job_status[1] = paramerror;
            }
        }
/* if no username or error in parameter(s) then there is nothing to do... */
        if (username.dsc$w_length > 0 && paramerror == SS$_NORMAL) {
            DEBUG("main: processing task", 0);
                           
/* evaluate logicals (user parameters overrule logicals) */
            if (filetype[0] == 0) if (getlogical(&filetypelogical, &filetype) == -1) sprintf(filetype, "");
            if (maildest[0] > 0) sendto = LOGICAL;
            else if (getlogical(&destlogical, &maildest) == -1) sendto = VMSUSER; else sendto = LOGICAL;
            if (!checkoper) if (getlogical(&fromlogical, &fromaddress) == -1) sprintf(fromaddress, "system@%s", thissys);
            if (subjectstring[0] > 0) subject = LOGICAL;
            else if (getlogical(&subjectlogical, &subjectstring) == -1) subject = FILENAME; else subject = LOGICAL;
            if (bodytextstring[0] > 0) {
                bodytext = 1;
                userbodytext = 1;
            } else if (getlogical(&bodytextlogical, &bodytextstring) == -1) bodytext = 0; else bodytext = 1;
            if (getlogical(&opclogical, &opcstring) == -1) opcom = 0; else opcom = 1;
            if (binaryflag == 0) {
                if (getlogical(&contentlogical, &contentstring) == SS$_NORMAL)
                    if (strcmp(contentstring, "BINARY") == 0) binaryflag = 1;
            } else if (binaryflag == 2) binaryflag = 0;
/* open input and output files */
            strncpy(infilename, filename.dsc$a_pointer, filename.dsc$w_length);
            infilename[filename.dsc$w_length] = 0;
            if ((infile = fopen(infilename, "r")) != NULL) {
                tmpfile = fopen(tmpfilename, "w");
                if (sendto == VMSUSER) {
                    strncpy(user, username.dsc$a_pointer, username.dsc$w_length);
                    user[username.dsc$w_length] = 0;
                } else strcpy(user, maildest);
/* put in the SMTP commands in the output file */
                sprintf(buf,"MAIL FROM:<%s>\n", fromaddress); fwrite(buf, 1, strlen(buf), tmpfile);
                sprintf(buf,"RCPT TO:<%s>\n", user); fwrite(buf, 1, strlen(buf), tmpfile);
                sprintf(buf,"DATA\nMime-Version: 1.0\n"); fwrite(buf, 1, strlen(buf), tmpfile);
/* isolate the input filename */
                while ((p = strstr(infilename, ">")) != NULL) *p = ']';
                if ((p = strstr(infilename, ".]")) != NULL) {
                    p+=2;
                    p=strstr(p, "]");
                }
                else p = strstr(infilename, "]");
                p++;
/* if no filetype specified via logical or parameter use the real filetype */
                *strstr(p,";") = 0;
                if (strcmp(filetype, "") == 0) strcpy(filetype, (strstr(p,".") + 1));
                *strstr(p,".") = 0;
/* put in the MIME headers */
                if (subject == FILENAME) sprintf(buf,"Subject: %s\nContent-Type: multipart/mixed;\n", p);
                else sprintf(buf,"Subject: %s\nContent-Type: multipart/mixed;\n", subjectstring);
                fwrite(buf, 1, strlen(buf), tmpfile);
                sprintf(buf,"    boundary=\"XXXXgrens_tekst\"\n"); fwrite(buf, 1, strlen(buf), tmpfile);
                sprintf(buf,"\nThis is a multi-part message in MIME format.\n"); fwrite(buf, 1, strlen(buf), tmpfile);
                if (bodytext) {
                    DEBUG("main: sending mail with bodytext", 0);
                    sprintf(buf,"--XXXXgrens_tekst\n"); fwrite(buf, 1, strlen(buf), tmpfile);
                    sprintf(buf,"Content-Type: text/plain; charset=us-ascii\n\n"); fwrite(buf, 1, strlen(buf), tmpfile);
                    if (bodytextstring[0] == '@') {
                        for (q = bodytextstring + 1; *q != 0; q++) if (*q >= 'a' && *q <= 'z') *q += 'A' - 'a';
                        q = bodytextstring + 1;
/* check if this user has read permission on the file specified as body text */
                        if (userbodytext) {
                            DEBUG("main: bodytext is user specified", bodytextstring);
                            userbodyfile.dsc$a_pointer = q;
                            userbodyfile.dsc$w_length = strlen(q);
                            job_status[1] = SYS$CHECK_ACCESS(&ACL$C_FILE, &userbodyfile, &username, chkxsitm, 0, 0, 0, 0);
                            if (!(job_status[1]&1)) {
                                sprintf(userstat,"%d",job_status[1]);
                                DEBUG("main: user has no access to body text file, status", userstat);
                            }
                        }
                        if (job_status[1]&1) {
                            if ((bodyfile = fopen(q, "r")) != NULL) {
                                DEBUG("main: adding bodyfile to mail", 0);
                                inchar = getc(bodyfile);
                                while (feof(bodyfile) == 0) {
                                    putc(inchar, tmpfile);
                                    inchar = getc(bodyfile);
                                }
                                fclose(bodyfile);
                            } else if (!userbodytext) {
                                DEBUG("main: body text file not found, exiting", 0);
                                SYS$EXIT(RMS$_FNF);
                            }
                        }
                    } else fwrite(bodytextstring, 1, strlen(bodytextstring), tmpfile);
                    sprintf(buf,"\n\n"); fwrite(buf, 1, strlen(buf), tmpfile);
                }
                if (job_status[1]&1) {
                    sprintf(buf,"--XXXXgrens_tekst\n"); fwrite(buf, 1, strlen(buf), tmpfile);
/* see if it is a binary attachment */
                    if (binaryflag) {
                        sprintf(buf,"Content-Type: application/octet-stream\n"); fwrite(buf, 1, strlen(buf), tmpfile);
                        sprintf(buf,"Content-Transfer-Encoding: base64\n");
                    }
                    else sprintf(buf,"Content-Type: text/plain; charset=us-ascii\n");
                    fwrite(buf, 1, strlen(buf), tmpfile);
                    sprintf(buf,"Content-Disposition: attachment;\n"); fwrite(buf, 1, strlen(buf), tmpfile);

/* rename the filetype real filetype or as specified in the logical or user parameter */
                    sprintf(buf,"    filename=\"%s.%s\"\n\n",p, filetype); fwrite(buf, 1, strlen(buf), tmpfile);
/* end of headers, copy the input file to the temp file */
                    if (binaryflag) encode(infile, tmpfile);
                    else {
                        inchar = getc(infile);
                        while (feof(infile) == 0) {
                            putc(inchar, tmpfile);
                            inchar = getc(infile);
                        }
                    }
                    fclose(infile);
                    fclose(tmpfile);
/* and let TCPIP handle the mailing */
                    if (debug & MAILDEBUG) job_status[1] = TCPIP$SMTP_SEND_FROM_FILE(tmpfilename, debugfile, 1);
                    else job_status[1] = TCPIP$SMTP_SEND_FROM_FILE(tmpfilename, 0, 0);
                    if (unlink(tmpfilename) != 0) SYS$EXIT(0);
                }

            }  else { /* job status in error */
                fclose(infile);
                fclose(tmpfile);
                if (unlink(tmpfilename) != 0) SYS$EXIT(0);
            }
        }
/* send opcom status message if requested */
        if (opcom && username.dsc$w_length > 0) {
            sprintf(userstat, "%d", job_status[1]);
            DEBUG("main: sending opcom message with status", userstat);
            sprintf(opcbuf,"%s: completed mail job %s for user %s, final status = %d", 
                queuename.dsc$a_pointer, p, username.dsc$a_pointer, job_status[1]);
            length = strlen(opcbuf);
            if (length > 128) length = 128;
            memcpy(&opcrequest.opc$l_ms_text, opcbuf, length);
            req_desc.dsc$w_length = length + 8;
            req_desc.dsc$a_pointer = (char *) &opcrequest;
            if (!((status = SYS$SNDOPR(&req_desc,0))&1)) return status;
        }
/* let the queue manager know we finished the job */
        if (!(job_status[1]&1)) job_status[0] = 1;
        else job_status[0] = 0;
        DEBUG("main: finished, informing queue manager", 0);
        request = SMBMSG$K_TASK_COMPLETE;
        if (!((status = SMB$SEND_TO_JOBCTL(&stream, &request, 0, 0, 0, &job_status))&1)) return status;
        DEBUG("main: queue manager informed", 0);
    }
}
/* the ast routine is called by the queue manager when a job is waiting */
void ast()
{

    static struct dsc$descriptor_d buffer = {0, DSC$K_DTYPE_T, DSC$K_CLASS_D, 0}, message = {0, DSC$K_DTYPE_T, DSC$K_CLASS_D, 0};
    static long status, item_code, size;
    char code[10];
    long context = 0;
    DEBUG("ast: reading message from queue manager", 0);
    if (!((status = SMB$READ_MESSAGE(&stream, &buffer, &request))&1)) SYS$EXIT(status);
    DEBUG("ast: read message", 0);
    paramerror = SS$_NORMAL;
    while ((status = SMB$READ_MESSAGE_ITEM(&buffer, &context, &item_code, &message, &size)) == SS$_NORMAL)
        switch (item_code) {
            case SMBMSG$K_FILE_SPECIFICATION:
                STR$COPY_DX(&filename, &message);
                filename.dsc$a_pointer[message.dsc$w_length] = 0;
                DEBUG("ast: message = SMBMSG$K_FILE_SPECIFICATION", filename.dsc$a_pointer);
                break;
            case SMBMSG$K_USER_NAME:
                STR$COPY_DX(&username, &message);
                username.dsc$a_pointer[message.dsc$w_length] = 0;
                DEBUG("ast: message = SMBMSG$K_USER_NAME", username.dsc$a_pointer);
                break;
            case SMBMSG$K_EXECUTOR_QUEUE:
                STR$COPY_DX(&queuename, &message);
                queuename.dsc$a_pointer[queuename.dsc$w_length] = 0;
                DEBUG("ast: message = SMBMSG$K_EXECUTOR_QUEUE", queuename.dsc$a_pointer);
/* construct logical names from queuename */
                sprintf(filetypelogical, "MAILSYM_%s_FILETYPE", queuename.dsc$a_pointer);
                sprintf(destlogical, "MAILSYM_%s_DEST", queuename.dsc$a_pointer);
                sprintf(fromlogical, "MAILSYM_%s_FROM_ADDRESS", queuename.dsc$a_pointer);
                sprintf(subjectlogical, "MAILSYM_%s_SUBJECT", queuename.dsc$a_pointer);
                sprintf(bodytextlogical, "MAILSYM_%s_BODYTEXT", queuename.dsc$a_pointer);
                sprintf(opclogical, "MAILSYM_%s_OPCOM", queuename.dsc$a_pointer);
                sprintf(contentlogical, "MAILSYM_%s_CONTENT_TYPE", queuename.dsc$a_pointer);
/* name of temporary file */

                sprintf(tmpfilename, "SYS$SYSTEM:MAILSYM_%s.TMP", queuename.dsc$a_pointer);
/* set process name */
                if (queuename.dsc$w_length <= 15) if (!((status = SYS$SETPRN(&queuename))&1)) SYS$EXIT(status);
                break;
            case SMBMSG$K_PARAMETER_1:
            case SMBMSG$K_PARAMETER_2:
            case SMBMSG$K_PARAMETER_3:
            case SMBMSG$K_PARAMETER_4:
            case SMBMSG$K_PARAMETER_5:
            case SMBMSG$K_PARAMETER_6:
            case SMBMSG$K_PARAMETER_7:
            case SMBMSG$K_PARAMETER_8:
                message.dsc$a_pointer[message.dsc$w_length] = 0;
                DEBUG("ast: message = SMBMSG$K_PARAMETER_x", message.dsc$a_pointer);
                if (!((status = parseparam(message.dsc$a_pointer, message.dsc$w_length))&1)) paramerror = status;
                break;
            default:
                sprintf(code, "%d", item_code);
                DEBUG("ast: message is default clause", code);
                break;
        }
    if (status != SMB$_NOMOREITEMS) SYS$EXIT(status);
    DEBUG("ast: no more items", 0);
    switch (request) {
        case SMBMSG$K_START_STREAM:
            DEBUG("ast: request = SMBMSG$K_START_STREAM, informing queue manager", 0);
            if (!((status = SMB$SEND_TO_JOBCTL(&stream, &request, 0, 0, 0, 0))&1)) SYS$EXIT(status);
            DEBUG("ast: done", 0);
            break;
        case SMBMSG$K_STOP_STREAM:
        case SMBMSG$K_RESET_STREAM:
            DEBUG("ast: request = SMBMSG$K_STOP_STREAM or SMBMSG$K_RESET_STREAM, informing queue manager", 0);
            if (!((status = SMB$SEND_TO_JOBCTL(&stream, &request, 0, 0, 0, 0))&1)) SYS$EXIT(status);
            DEBUG("ast: done, exiting", 0);
            SYS$EXIT(SS$_NORMAL);
            break;
        case SMBMSG$K_START_TASK:
            DEBUG("ast: request = SMBMSG$K_START_TASK, calling sys$wake", 0);
            if (!((status = SYS$WAKE(0, 0))&1)) SYS$EXIT(status);
            break;
        case SMBMSG$K_STOP_TASK:
        case SMBMSG$K_PAUSE_TASK:
        case SMBMSG$K_RESUME_TASK:
            DEBUG("ast: request = SMBMSG$K_STOP/PAUSE/RESUME_TASK informing queue manager", 0);
            if (!((status = SMB$SEND_TO_JOBCTL(&stream, &request, 0, 0, 0, 0))&1)) SYS$EXIT(status);
            DEBUG("ast: done", 0);
            break;
        default:
            sprintf(code, "%d", request);
            DEBUG("ast: request is default clause", code);
            break;
    }
}
int getlogical(char *name, char *dest)
{
    int status;
    lognam.dsc$a_pointer = name;
    lognam.dsc$w_length = strlen(name);
    status = SYS$TRNLNM(0, &tabnam, &lognam, 0, &lnmlst);
    if (status == SS$_NOLOGNAM) return -1;
    if (!(status&1)) SYS$EXIT(status);
    strncpy(dest, logvalue, loglen);
    dest[loglen] = 0;
    return SS$_NORMAL;
}
void wrdebug(char *message, char *val)
{
    char time[24], buf[132];
    int status;
    $DESCRIPTOR (value,"");
    value.dsc$a_pointer = time;
    value.dsc$w_length = 23;
    if (!((status = SYS$ASCTIM(0,&value,0,0))&1)) SYS$EXIT(status);
    time[23] = 0;
    if (val == 0) sprintf(buf, "%s: %s\n", time, message);
    else sprintf(buf, "%s: %s : %s\n", time, message, val);
    fwrite(buf, 1, strlen(buf), debugfile);
}
int parseparam(char *param, int len)
{
    char *p, *q;
    DEBUG("parseparam: parsing parameter", 0);
    param[len] = 0;
    for (p = param; (*p != 0) && (*p != '='); p++) if (*p >= 'a' && *p <= 'z') *p += 'A' - 'a'; /* upcase param */
    if ((p = strstr(param, "=")) == NULL) {
        DEBUG("parseparam: no equal sign found", 0);
        return SS$_IVPARAM;
    }
    if (strlen(++p) == 0) {
        DEBUG("parseparam: no value given", 0);
        return SS$_IVPARAM;
    }
    if (strncmp(param, "CONTENT_TYPE", p - param - 1) == 0) {
        for (q = p; *q != 0; q++) if (*q >= 'a' && *q <= 'z') *q += 'A' - 'a';
        DEBUG("parseparam: found CONTENT_TYPE parameter", 0);
        if (strncmp(p, "BINARY", q - p) == 0) binaryflag = 1;
        else if (strncmp(p, "ASCII", q - p) != 0) {
            DEBUG("parseparam: unsupported CONTENT_TYPE", 0);
            return SS$_IVPARAM;
        } else binaryflag = 2;
        return SS$_NORMAL;
    }
    if (strncmp(param, "DESTINATION_ADDRESS", p - param - 1) == 0) {
        DEBUG("parseparam: found DESTINATION_ADDRESS parameter", 0);
        strcpy(maildest, p);
        return SS$_NORMAL;
    }
    if (strncmp(param, "FROM_ADDRESS", p - param - 1) == 0) {
        DEBUG("parseparam: found FROM_ADDRESS parameter", 0);
        strcpy(fromaddress, p);
        checkoper = 1;
        return SS$_NORMAL;
    }
    if (strncmp(param, "SUBJECT", p - param - 1) == 0) {
        DEBUG("parseparam: found SUBJECT parameter", 0);
        strcpy(subjectstring, p);
        return SS$_NORMAL;
    }
    if (strncmp(param, "BODYTEXT", p - param - 1) == 0) {
        DEBUG("parseparam: found BODYTEXT parameter", 0);
        strcpy(bodytextstring, p);
        userbodytext = 1;
        return SS$_NORMAL;
    }

    if (strncmp(param, "FILETYPE", p - param - 1) == 0) {
        DEBUG("parseparam: found FILETYPE parameter", 0);
        strcpy(filetype, p);
        return SS$_NORMAL;
    }
    DEBUG("parseparam: parameter not recognized", 0);
    return SS$_IVPARAM;
}
void encode(FILE *infile, FILE *outfile)
{
    unsigned char in[3], out[4];
    int i, len, blocksout = 0;
    while( !feof( infile ) ) {
        len = 0;
        for( i = 0; i < 3; i++ ) {
            in[i] = (unsigned char) getc( infile );
            if( !feof( infile ) ) {
                len++;
            }
            else {
                in[i] = 0;
            }
        }
        if( len ) {
            encodeblock( in, out, len );
            for( i = 0; i < 4; i++ ) {
                putc( out[i], outfile );
            }
            blocksout++;
        }
        if( blocksout >= (72/4) || feof( infile ) ) {
            if( blocksout ) {
                fprintf( outfile, "\r\n" );
            }
            blocksout = 0;
        }
    }
}
void encodeblock( unsigned char in[3], unsigned char out[4], int len )
{
    out[0] = cb64[ in[0] >> 2 ];
    out[1] = cb64[ ((in[0] & 0x03) << 4) | ((in[1] & 0xf0) >> 4) ];
    out[2] = (unsigned char) (len > 1 ? cb64[ ((in[1] & 0x0f) << 2) | ((in[2] & 0xc0) >> 6) ] : '=');
    out[3] = (unsigned char) (len > 2 ? cb64[ in[2] & 0x3f ] : '=');
}
(
